#include <stdlib.h>
#include "fib.h"

struct fibElementT
{
  fibElementP parent;
  fibElementP child;
  fibElementP next;
  fibElementP prev;
  int rank;
  int marked;
  unsigned long priority;
  void *data;
};

typedef struct fibRootT
{
  fibElementP child;
  int rank;
} fibRootT,*fibRootP;

struct fibT
{
  /* The Root of roots :-) */
  fibRootT root;
  
  /* Maximum Root */
  fibElementP max;
  int len;
};

/* Local helper functions */

/*
 * Create a new Fibonacci Heap element with priority, and data
 */ 
static fibElementP fibNewElement(unsigned long priority,void *data)
{
  fibElementP result;
  result=(fibElementP)malloc(sizeof(fibElementT));
  result->parent=NULL;
  result->child=NULL;
  result->rank=0;
  result->priority=priority;
  result->data=data;
  return result;
}

/* 
 * The roots are kept in a circular double linked list to 
 * Make O(1) merges of fibonacci heaps possible            
 */
static inline void fibRootInsert(fibRootP root,fibElementP el)
{
  root->rank++;
  el->parent=NULL;
  if (root->child!=NULL)
  {
    el->prev=root->child;
    el->next=root->child->next;
    root->child->next->prev=el;
    root->child->next=el;
  }
  else
  {
    el->prev=el;
    el->next=el;
    root->child=el;
  }
}

static inline void fibRootRemove(fibRootP root,fibElementP el)
{
  root->rank--;
  if (el->next==el)
  {
    root->child=NULL;
    return;
  }
  if (root->child==el) 
    root->child=el->next;
  el->next->prev=el->prev;
  el->prev->next=el->next;
}


/* 
 * The Childs of a node are kept in a NON-circular double linked list
 */
 
/*
 * Insert an element into child list of parent
 */
static inline void fibChildInsert(fibElementP parent,fibElementP el)
{
  el->parent=parent;
  el->marked=0;
  parent->rank++;
  el->next=parent->child;
  el->prev=NULL;
  if (parent->child!=NULL) parent->child->prev=el;
  parent->child=el;
}

/*
 * Removes an element from child list of parent
 * updates rank of parent
 */
static inline void fibChildRemove(fibElementP parent,fibElementP el)
{
  parent->rank--; 
  if (parent->child==el) parent->child=el->next;
  if (el->prev!=NULL) el->prev->next=el->next;
  if (el->next!=NULL) el->next->prev=el->prev;
}

/*
 * Merges two roots of fibonacci heap 
 * (makes the root with smaller priority child of root with
 *  higher priority)
 */
static fibElementP fibMergeRoots(fibP heap,fibElementP r1,fibElementP r2)
{
  if (r1->priority >= r2->priority)
  {
    fibRootRemove(&(heap->root),r2);
    fibChildInsert(r1,r2);
    return r1;
  }
  fibRootRemove(&(heap->root),r1);
  fibChildInsert(r2,r1);
  return r2;
}

/*
 * Merges roots of the same rank. 
 * Because a tree of a root of rank r has F(r+2) 
 * nodes, 256 as maximum rank should be quite enough because
 * F(258) > 10^45 (a very very rough estimate).
 */
static void fibMergeRanks(fibP heap)
{
  fibElementP ranks[256];
  fibElementP n,nn;
  int r,maxRank,i,c;
  
  maxRank=-1;
  
  nn=heap->root.child;
  c=heap->root.rank;
  for(i=0;i<c;i++)
  {
    n=nn;
    nn=nn->next;
    r=n->rank;
    while (r<=maxRank && ranks[r]!=NULL)
    {
      n=fibMergeRoots(heap,ranks[r],n);
      ranks[r]=NULL;
      r++;
    }
    while(r>maxRank) ranks[++maxRank]=NULL;
    ranks[r]=n;
  }
}

/* 
 * Finds the maximum root and updates
 * the pointer to it.
 */
static void fibUpdateMax(fibP heap)
{
  fibElementP n,max;
  int i,r;
  
  r=heap->root.rank;
  n=heap->root.child;
  max=n;
  for (i=1;i<r;i++)
  {
    n=n->next;
    if (n->priority>max->priority)
      max=n;
  }
  heap->max=max;
}

/*
 * Reorganises Fibonacci Heap
 * by merging ranks and updating the maximum
 */
static void fibReorganise(fibP heap)
{

  if (heap->len==0) 
  {
    heap->max=NULL;
    return;
  }
  
  fibMergeRanks(heap);
  fibUpdateMax(heap);
}

/*
 * Puts a list of child nodes into the list of roots
 * (which is needed when a DeleteMax happens.)
 */
static void fibAddListToRoots(fibRootP root,fibElementP list)
{
  fibElementP node,last;
  int r;

  r=0;
  node=last=list;
  while (node!=NULL)
  {
    last=node;
    node->parent=NULL;
    r++;
    node=node->next;
  }
  if (root->child != NULL)
  {
    list->prev=root->child;
    last->next=root->child->next;
    root->child->next->prev=last;
    root->child->next=list;
  }
  else
  {
    last->next=list;
    list->prev=last;
    root->child=list;
  }
  root->rank+=r;
}

/****************** Interface functions ************************/

/*
 * Creates an empty fibonacci heap
 */
fibP fibNew()
{
  fibP result;
  result=(fibP)malloc(sizeof(fibT));
  result->max=NULL;
  result->len=0;
  result->root.child=NULL;
  result->root.rank=0;
  return result;
}

/*
 * Returns the number of elements in a fibonacci heap
 */
int fibLen(fibP fib)
{
  return fib->len;
}

/*
 * Returns priority of element
 */
unsigned long fibPriority(fibElementP el)
{
  return el->priority;
}

/*
 * Returns data of element
 */
void *fibData(fibElementP el)
{
  return el->data;
}

/*
 * Inserts data with priority into fibonacci heap
 * (returns a pointer to the element, which is neccessary to call
 *  IncreasePriority later.)
 */
fibElementP fibInsert(fibP heap,unsigned long priority,void *data)
{
  fibElementP el;
  el=fibNewElement(priority,data);
  
  heap->len++;

  fibRootInsert(&(heap->root),el);

  if (heap->max==NULL)
    heap->max=el;
  else if (el->priority > heap->max->priority)
    heap->max=el;

  return el;
}

/*
 * Increases Priority of en element
 * Implements cascading cuts.
 */
void fibIncreasePriority(fibP heap,fibElementP el,unsigned long newPrio)
{
  fibElementP parent;
  fibRootP root;
  
  el->priority=newPrio;
  if (newPrio > heap->max->priority)
    heap->max=el;

  root=&(heap->root);
  do
  {
    parent=el->parent;
    
    /* If it's already a root node we are finished */
    if (parent == NULL) 
      break;
  
    fibChildRemove(parent,el);
    fibRootInsert(root,el);
    
    /* If parent was unmarked, mark it and return */
    if (!parent->marked)
    {
      parent->marked=1;
      break;
    }
    /* Otherwise contiue cascading cut with parent */
    el=parent;
  } while(1);
}

/* 
 * Merges two fibonacci heaps.
 * the second heap (q2) will be freed.
 */ 
void fibMerge(fibP q1,fibP q2)
{
  fibElementP list;
  fibRootP root;
  list=q2->root.child;
  if (list==NULL) 
  {
    free(q2);
    return;
  }
  root=&(q1->root);
  if (root->child!=NULL)
  { 
    list->prev->next=root->child->next;
    root->child->next->prev=list->prev;
    root->child->next=list;
    list->prev=root->child;
  }
  else
    root->child=list;
  
  root->rank+=q2->root.rank;
  q1->len+=q2->len;
  if (q1->max->priority<q2->max->priority)
    q1->max=q2->max;
  free(q2);
}
 
/*
 * finds, returns and deletes maximum
 * from fibonacci heap
 */
void fibDeleteMax(fibP heap,unsigned long *priority,void **data)
{
  fibElementP max,n;
  fibRootP root;

  heap->len--;
  max=heap->max;
  root=&(heap->root);

  fibRootRemove(root,max);
  
  if (data!=NULL)
    *data=max->data;
  if (priority!=NULL)
    *priority=max->priority;

  n=max->child;
  free(max);
  
  if (n!=NULL) fibAddListToRoots(root,n);
  fibReorganise(heap);
}
