#include <stdlib.h>
#include <stdio.h>
#include "kbuck.h"

/* K is number of levels */
#define K (4)

/* B is 2^(32/K) */
#define B (1<<(32/K))

typedef struct kbuckLevelT kbuckLevelT,*kbuckLevelP;

struct kbuckLevelT
{
  unsigned int bound;
  unsigned int div;
  int filled;
  int min;
  kbuckElementP buckets[B];
};

struct kbuckElementT
{
  kbuckLevelP level;
  kbuckElementP prev;
  kbuckElementP next;
  unsigned int prio;
  void *data;
};

struct kbuckT
{
  kbuckLevelT levels[K];
  int len;
};

static kbuckElementP kbuckNewElement(unsigned int prio,void *data)
{
  kbuckElementP el;
  el=(kbuckElementP)malloc(sizeof(kbuckElementT));
  el->prio=prio;
  el->data=data;
  return el;
}

static inline void kbuckListInsert(kbuckElementP *head,kbuckElementP el)
{
  el->prev=NULL;
  el->next=(*head);
  if ( (*head) != NULL) (*head)->prev=el;
  (*head) =el;
}

static inline void kbuckListRemove(kbuckElementP *head,kbuckElementP el)
{
  if ((*head) == el) (*head)=el->next;
  if (el->next!=NULL) el->next->prev=el->prev;
  if (el->prev!=NULL) el->prev->next=el->next;
}

static inline void kbuckLevelInsert(kbuckLevelP level,int bucket,
                                     kbuckElementP el)
{
  el->level=level;
  if (level->buckets[bucket] == NULL) level->filled++;
  kbuckListInsert(&(level->buckets[bucket]),el);
}

static inline void kbuckLevelRemove(kbuckLevelP level,int bucket,
                                     kbuckElementP el)
{
  kbuckListRemove(&(level->buckets[bucket]),el);
  if (level->buckets[bucket]==NULL) level->filled--;
}

static inline int kbuckBucketNumber(kbuckLevelP level,kbuckElementP el)
{
  return (el->prio - level->bound)/level->div;
}

/* Inserts an Element into the heap */
static void kbuckElementInsert(kbuckP heap,kbuckElementP el)
{
  kbuckLevelP level;
  int l_nu,b_nu;

  l_nu=0;
  while (1)
  {
    level=&(heap->levels[l_nu]);
    b_nu=kbuckBucketNumber(level,el);
    if (l_nu==K-1 || level->min != b_nu)
    {
      /* If we are at lowest level, or bucket isn't           *
       * handled by lower level, insert element into bucket   *
       * and return                                           */
      kbuckLevelInsert(level,b_nu,el);
      return;
    }
    l_nu++;
  }
}

/* Remove element from heap. Trivial. */
static void kbuckElementRemove(kbuckElementP el)
{
  kbuckLevelP level;
  int b_nu;
  level=el->level;
  b_nu=kbuckBucketNumber(level,el);
  kbuckLevelRemove(level,b_nu,el);
}

/*
 * Distributes Elements from list to levels from
 * l_nu to K-1
 */
static kbuckElementP kbuckFillBuckets(kbuckP heap,int l_nu,
                                      unsigned int bound,kbuckElementP list)
{
  kbuckElementP min;
  kbuckLevelP level;
  int b_nu,m_nu;

  b_nu=0;
  do
  {  
    level=&(heap->levels[l_nu]);
    level->bound=bound;

    /* Put list into new buckets
     * m_nu will be the number of the smallest non-empty Bucket */  
    m_nu=B; 
    while(list!=NULL)
    {
      min=list->next;
      b_nu=kbuckBucketNumber(level,list);
      kbuckLevelInsert(level,b_nu,list);
      if (b_nu < m_nu) m_nu=b_nu;
      list=min;
    }
    
    /* Remember new minimum bucket */
    level->min=m_nu;
       
    if (l_nu==K-1) 
    {
      /* If we are lowest level, we get a minimal element from 
       * bucket list and return */
      min=level->buckets[m_nu];
      kbuckLevelRemove(level,m_nu,min);
      return min;
    }
    
    /* If we are not lowest level, get the elment list
     * from smallest non-empty bucket and distrubute the
     * elements at the next level                        */
    list=level->buckets[m_nu];
    level->buckets[m_nu]=NULL;
    level->filled--;
    l_nu++;
    bound=level->bound+m_nu*level->div;
  } while(1);
}

/* Looks for a new smallest non-empty bucket in a level.
 * Updates lower levels and returns found minimum.
 */
static kbuckElementP kbuckUpdateMin(kbuckP heap,int l_nu)
{
  kbuckElementP min;
  kbuckLevelP level;
  int m_nu;
  level=&(heap->levels[l_nu]);

  if (level->filled == 0)  
  {
    /* No more non-empty buckets at this level so return */
    level->min=B;
    return NULL;
  }
  
   /* We could start at min+1 if we are not in the lowest
    * level, but this doesn't improve speed */
  m_nu=level->min;
  while (m_nu<B)
  {
    if (level->buckets[m_nu]!=NULL) break;
    m_nu++;
  }
  level->min=m_nu;

  min=level->buckets[m_nu];
  if (l_nu==K-1)
  {
    /* If we are in lowest level we found new minimum. */
    kbuckLevelRemove(level,m_nu,min);
    return min;
  }
  /* If we are not in lowest level we have to distribute elements
   * from smallest non-empty bucket to lower levels to find minimum */
  level->buckets[m_nu]=NULL;
  level->filled--;
  min=kbuckFillBuckets(heap,l_nu+1,m_nu*level->div+level->bound,min);
  return min;
}

/* Find minimum of level */
static kbuckElementP kbuckGetMin(kbuckP heap,int l_nu)
{
  kbuckElementP min;

  min=NULL;
  
  /* If we are not lowest level first ask level below */
  if (l_nu<K-1) min=kbuckGetMin(heap,l_nu+1);
  
  /* If there are no more elements below, we have to check
   * in current level for new minimal bucket */
  if (min==NULL) min=kbuckUpdateMin(heap,l_nu);
  return min;
}

kbuckP kbuckNew()
{
  kbuckP heap;
  int i,j;
  unsigned int div;
  heap=(kbuckP)malloc(sizeof(kbuckT));
  
  /* div will be the bucket width of current level */
  div=1; 
  
  /* We will initialise the levels so that the lowest level
   * holds elements from 0 .. B-1 */
  for (i=K-1;i>=0;i--)
  {
    heap->levels[i].min=0;
    heap->levels[i].div=div;
    heap->levels[i].bound=0;
    heap->levels[i].filled=0;
    for (j=0;j<B;j++)
      heap->levels[i].buckets[j]=NULL;
    div*=B;
  }
  return heap;
}

int kbuckLen(kbuckP heap)
{
  return heap->len;
}

unsigned int kbuckPriority(kbuckElementP el)
{
  return el->prio;
}

void *kbuckData(kbuckElementP el)
{
  return el->data;
}

kbuckElementP kbuckInsert(kbuckP heap,unsigned int prio,void *data)
{
  kbuckElementP el;
  
  el=kbuckNewElement(prio,data);
  heap->len++;
  kbuckElementInsert(heap,el);
  return el;
}

void kbuckDecreasePriority(kbuckP heap,kbuckElementP el,unsigned int newPrio)
{
  kbuckElementRemove(el);
  el->prio=newPrio;
  kbuckElementInsert(heap,el);
}

void kbuckDeleteMin(kbuckP heap,unsigned int *prio,void **data)
{
  kbuckElementP min;
  min=kbuckGetMin(heap,0);
  heap->len--;
  if (prio!=NULL) *prio=min->prio;
  if (data!=NULL) *data=min->data;
  free(min);
}
