#include <stdio.h>
#include <stdlib.h>
#include "krdx.h"

/* Each bucket contains (2^BITS)-1 segments.
   If BITS == 1 you obtain a regular radix heap.
*/
#define BITS (10)
#define SEGMENTS ((1<<BITS)-1)
#define MAXBUCK ((30/BITS)+2)

struct krdxElementT
{
  int bucket;
  krdxElementP prev,next;
  unsigned int prio;
  void *data;
};

typedef struct krdxBucketT krdxBucketT,*krdxBucketP;
struct krdxBucketT
{
  int filled;
  unsigned int bound;
  int first;
  krdxElementP segments[SEGMENTS];
};

struct krdxT
{
  int len;
  krdxBucketT buckets[MAXBUCK];
};

/* This internal function calculates the bucket bounds.
   Conceptually the calculation ends at maxBucket,nextSeg
   or heap->buckets[maxBucket].segments[nextSeg], where
   the next undistributed elements should be.
*/
static void krdxCalculateBounds(krdxP heap, 
                                unsigned int min,
                                int maxBucket,
                                int nextSeg,
                                unsigned int maxBound)
{
  krdxBucketP buckets;
  int i;
  unsigned int p;
  
  buckets=heap->buckets;
  
  buckets[0].bound=min;
  buckets[0].first=0;

  p=SEGMENTS;
  min++;
  for (i=1;i<maxBucket;i++)
  {
    buckets[i].first=0;
    buckets[i].bound=min;
    min+=p;
    p<<=BITS;
    if (min>=maxBound)
    {
      min=maxBound;
      p=0;
    }
  }
  buckets[maxBucket].first=nextSeg;
  buckets[maxBucket].bound=min;
}

static void krdxListInsert(krdxElementP *head,krdxElementP el)
{
  el->prev=NULL;
  el->next=*head;
  if (*head!=NULL) (*head)->prev=el;
  *head=el;
}

static void krdxListRemove(krdxElementP *head,krdxElementP el)
{
  if (*head==el)
    *head=el->next;
  if (el->prev!=NULL) el->prev->next=el->next;
  if (el->next!=NULL) el->next->prev=el->prev;
}

static krdxElementP krdxNewElement(unsigned int prio,void *data)
{
  krdxElementP res;
  res=(krdxElementP)malloc(sizeof(krdxElementT));
  res->prio=prio;
  res->data=data;
  return res;
}

static void krdxBucketInsert(krdxBucketP bucket,int buck,krdxElementP el)
{
  unsigned int seg;
  
  el->bucket=buck;
  
  seg=(el->prio - bucket->bound);
  if (buck>1) seg>>=((buck-1)*BITS);
  seg+=bucket->first;
  
  if ( bucket->segments[seg] == NULL) bucket->filled++;
  krdxListInsert(&(bucket->segments[seg]),el);
}

static void krdxBucketRemove(krdxBucketP bucket,int buck,krdxElementP el)
{
  unsigned int seg;
  
  seg=(el->prio - bucket->bound);
  if (buck>1) seg>>=((buck-1)*BITS);
  seg+=bucket->first;
  
  krdxListRemove(&(bucket->segments[seg]),el);
  if ( bucket->segments[seg] == NULL) bucket->filled--;
}
  
static int krdxFindCorrectBucket(krdxP heap,unsigned int prio,int start)
{
  int res;
  res=start;
  while (heap->buckets[res].bound > prio) res--;
  return res;
}

static void krdxDistribute(krdxP heap,krdxElementP list,int start)
{
  int bucket;
  krdxElementP nn;
  while (list!=NULL)
  {
    nn=list->next;
    bucket=krdxFindCorrectBucket(heap,list->prio,start);
    krdxBucketInsert(&(heap->buckets[bucket]),bucket,list);
    list=nn;
  }
}

static krdxElementP krdxListFindMin(krdxElementP list)
{
  krdxElementP min;
  min=list;
  list=list->next;
  while (list!=NULL)
  {
    if (list->prio<min->prio) min=list;
    list=list->next;
  }
  return min;
}

static krdxElementP krdxRemoveMin(krdxP heap,krdxElementP *list,
                                   int *firstSeg,int *maxBucket,
                                   unsigned int *maxBound)
{
  unsigned int b,s,i;
  krdxElementP min;
  krdxBucketP buckets;

  i=0;
  buckets=heap->buckets;
  while (buckets[i].filled==0) i++;
  
  if (i==0)
  {
    min=buckets[0].segments[0];
    krdxListRemove(&(buckets[0].segments[0]),min);
    if (buckets[0].segments[0] == NULL) buckets[0].filled=0;
    *maxBucket=0;
    return min;
  }
  
  *maxBucket=i;
  
  s=buckets[i].first;
  while (buckets[i].segments[s]==NULL) s++;
  
  min=krdxListFindMin(buckets[i].segments[s]);
  krdxListRemove(&(buckets[i].segments[s]),min);

  *list=buckets[i].segments[s];
  buckets[i].segments[s]=NULL;  
  buckets[i].filled--;
  
  s++;  
  if (s>=SEGMENTS)
  { 
    s=0; 
    b=buckets[i+1].bound;
  }
  else
  {
    /* We have to calculate the bound of the next segment */
    
    b=s-buckets[i].first;   // get relative segment number
    b<<=((i-1)*BITS);       // b = 2^(BITS*(i-1)) = width of one segment
    b+=buckets[i].bound;    // and add to bucket_bound to next segment bound
  }  
  *firstSeg=s;
  *maxBound=b;
  return min;
}

krdxP krdxNew()
{
  krdxP res;
  int i,j;
  res=(krdxP)malloc(sizeof(krdxT));
  res->len=0;
  krdxCalculateBounds(res,0,MAXBUCK-1,0,1073741824);
  for (i=0;i<MAXBUCK;i++)
  {
    for (j=0;j<SEGMENTS;j++)
    {
      res->buckets[i].segments[j]=NULL;
    }
  }
  return res;
}

int krdxLen(krdxP heap)
{
  return (heap->len);
}

unsigned int krdxPriority(krdxElementP el)
{
  return el->prio;
}

void *krdxData(krdxElementP el)
{
  return el->data;
}

krdxElementP krdxInsert(krdxP heap,unsigned int prio,void *data)
{
  krdxElementP el;
  int buck;
  el=krdxNewElement(prio,data);
  buck=krdxFindCorrectBucket(heap,prio,MAXBUCK-1);
  krdxBucketInsert(&(heap->buckets[buck]),buck,el);
  heap->len++;
  return el;
}

void krdxDecreasePriority(krdxP heap,krdxElementP el,unsigned int newPrio)
{
  int start;
  start=el->bucket;
  krdxBucketRemove(&(heap->buckets[start]),start,el);
  el->prio=newPrio;
  start=krdxFindCorrectBucket(heap,newPrio,start);
  krdxBucketInsert(&(heap->buckets[start]),start,el);
}

void krdxDeleteMin(krdxP heap,unsigned int *prio,void **data)
{
  krdxElementP min,restList;
  int nextSeg,maxBucket;
  unsigned int maxBound;
  
  heap->len--;
  min=krdxRemoveMin(heap,&restList,&nextSeg,&maxBucket,&maxBound);
  if (maxBucket>0)
  {
    krdxCalculateBounds(heap,min->prio,maxBucket,nextSeg,maxBound);
    if (restList!=NULL)
      krdxDistribute(heap,restList,maxBucket-1);
  }
  if (prio !=NULL) *prio=min->prio;
  if (data !=NULL) *data=min->data;
  free(min);
  return;
}

void krdxPrintStatus(krdxP heap)
{
  int i,j;
  unsigned int b,bb;
  
  printf("Heap: len %d\n",heap->len);
  for (i=0;i<MAXBUCK;i++)
  {
    printf("  Bucket %d\n",i);
    printf("    segments : %d\n",SEGMENTS-heap->buckets[i].first);
    printf("    bounds : ");
    b=heap->buckets[i].bound;
    if (i>0)
    {
      bb=0;
      for (j=heap->buckets[i].first;j<SEGMENTS;j++)
      {
        printf("%d ",b+bb);
        bb+=(1<<((i-1)*BITS));
      }
    }
    else
      printf("%d",b);
    printf("\n");
    printf("    filled : %d\n",heap->buckets[i].filled);
  }
}

