#include <stdlib.h>
#include "rdx.h"

/* 32 Buckets are enough for values from 0..(2^31-1) */
#define MAXBUCK 32

struct rdxElementT
{
  rdxElementP next;
  rdxElementP prev;
  int bucket;
  unsigned int priority;
  void *data;
};

typedef struct rdxBucketT
{
  rdxElementP list;
  unsigned int bound;
} rdxBucketT,*rdxBucketP;

struct rdxT
{
  int len;
  rdxBucketT buckets[MAXBUCK];
};

static rdxElementP rdxNewElement(unsigned int priority,void *data)
{
  rdxElementP res;
  res=(rdxElementP)malloc(sizeof(rdxElementT));
  res->priority=priority;
  res->data=data;
  return res;
}

static void rdxListInsert(rdxElementP *head,rdxElementP el)
{
  el->prev=NULL;
  el->next=*head;
  if (*head!=NULL) (*head)->prev=el;
  *head=el;
}

static void rdxListRemove(rdxElementP *head,rdxElementP el)
{
  if (*head==el)
    *head=el->next;
  if (el->prev!=NULL) el->prev->next=el->next;
  if (el->next!=NULL) el->next->prev=el->prev;
}

/*
 * Find correct bucket, starting at bucket "start"
 * and insert element into this bucket
 */
static void rdxBucketInsert(rdxP heap,rdxElementP el,int start)
{
  rdxBucketP buckets;
  int i;
  
  buckets=heap->buckets;
  
  i=start;
  while (buckets[i].bound>el->priority) i--;
  el->bucket=i;
  rdxListInsert(&(heap->buckets[i].list),el);
}

/*
 * Finds and returns minimal element in a list
 */
static rdxElementP rdxListFindMin(rdxElementP list)
{
  rdxElementP min;
  min=list;
  list=list->next;
  while (list!=NULL)
  {
    if (list->priority<min->priority)
      min=list;
    list=list->next;
  }
  return min;
}

/*
 * Finds minimal Element in heap, removes it from the heap
 * and returns it.
 */
static rdxElementP rdxGetMin(rdxP heap)
{
  rdxBucketP buckets;
  rdxElementP min;
  int b;
  
  buckets=heap->buckets;  
  b=0;
  while (b<MAXBUCK && buckets[b].list==NULL) b++;
  if (b>1)
    min=rdxListFindMin(buckets[b].list);
  else
    min=buckets[b].list;
  
  rdxListRemove(&(buckets[b].list),min); 
  return min;
}

/* 
 * Calculate new bounds for bucket 0..maxBucket,
 * starting with bound "minPrio"
 */
static void rdxCalculateBounds(rdxP heap,unsigned int minPrio,int maxBucket)
{
  rdxBucketP buckets;
  unsigned int p;
  int i;
  
  buckets=heap->buckets;
  buckets[0].bound=minPrio;
  buckets[1].bound=minPrio+1;
  p=1;
  for (i=2;i<=maxBucket;i++)
  {
    buckets[i].bound=buckets[i-1].bound+p;
    if (buckets[i].bound>=0x80000000) p=0;
    p*=2;
  }
}

/*
 * Distribute Elments from bucket "bucket" to
 * smaller buckets.
 */
static void rdxDistributeElements(rdxP heap,int bucket)
{
  rdxElementP el,nn;
  el=heap->buckets[bucket].list;
  heap->buckets[bucket].list=NULL;
  bucket--;
  while (el!=NULL)
  {
    nn=el->next;
    rdxBucketInsert(heap,el,bucket);
    el=nn;
  }
}

static void rdxReorganise(rdxP heap,unsigned int minPrio,int maxBucket)
{
  rdxCalculateBounds(heap,minPrio,maxBucket);
  rdxDistributeElements(heap,maxBucket);  
}


rdxP rdxNew()
{
  rdxP res;
  int i;
  
  res=(rdxP)malloc(sizeof(rdxT));
  res->len=0;
  rdxCalculateBounds(res,0,MAXBUCK-1);
  for (i=0;i<MAXBUCK;i++)
    res->buckets[i].list=NULL;
  return res;
}

int rdxLen(rdxP heap)
{
  return heap->len;
}

unsigned int rdxPriority(rdxElementP el)
{
  return el->priority;
}

void *rdxData(rdxElementP el)
{
  return el->data;
}

rdxElementP rdxInsert(rdxP heap,unsigned int priority,void *data)
{
  rdxElementP el;

  heap->len++;
  el=rdxNewElement(priority,data);
  rdxBucketInsert(heap,el,MAXBUCK-1);
  return el;
}

void rdxDecreasePriority(rdxP heap,rdxElementP el,unsigned int newPrio)
{
  int b;
  
  b=el->bucket;
  el->priority=newPrio;  
  if (heap->buckets[b].bound<=newPrio) return;
  
  rdxListRemove(&(heap->buckets[b].list),el);
  rdxBucketInsert(heap,el,b-1);
}

void rdxDeleteMin(rdxP heap,unsigned int *priority,void **data)
{
  rdxElementP min;

  heap->len--;

  min=rdxGetMin(heap);
  
  if (priority!=NULL) *priority=min->priority;
  if (data != NULL) *data=min->data;
  
  if (min->bucket>0) rdxReorganise(heap,min->priority,min->bucket);
  free(min);
}
