package Blatt6;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.util.Date;
import java.util.Iterator;
import java.util.Random;
import java.util.Vector;

import javax.swing.JPanel;

/**
 * Das Spielfeld
 * @author Stefan Pfingstl
 */
public class GamePanel extends JPanel implements Runnable {
	private int offset = 50;
	private int gridSize = 30;
	private int cols;
	private int rows;
	private int size = 20;
	private int playerNr = 0;
	private int nrMoves = 0;
	private Vector moves = new Vector();
	private Vector tmpMoves = new Vector();
	private int winningPlayer = -1;

	private int[][] game;
	private Color[] colors = new Color[2];
	private Player[] players = new Player[2];

	/** Erzeugt ein Spielfeld mit col Spalten und row Zeilen 
	 * @param cols Anzahl der Spalten
	 * @param rows Anzahl der Zeilen
	 * Das Spielfeld wird mit -1 vorbelegt
	 */
	public GamePanel(int cols, int rows) {
		this.cols = cols;
		this.rows = rows;

		game = new int[cols][rows];

		colors[0] = Color.YELLOW;
		colors[1] = Color.GREEN;

		newGame(cols, rows);
	}

	/** Liefert die Farbe des Feldes
	 * @param col Spalte
	 * @param row Zeile
	 * @return 0, 1 oder -1
	 */
	public int getPosition(int col, int row) {
		return game[col][row];
	}

	
	/** Gibt den letzten zu testenden Spielzug zur&uuml;ck
	 * @return letzer Testzug
	 */
	public Move getLastTestMove() {
		return (Move) tmpMoves.get(tmpMoves.size() - 1);
	}

	/** Gibt den letzten Spielzug zur&uuml;ck
	 * @return letzter Spielzug
	 */
	public Move getLastMove() {
		return (Move) moves.get(moves.size() - 1);
	}

	
	
	/** Beginnt ein neues Spiel
	 * @param cols Anzahl der Spalten
	 * @param rows Anzahl der Zeilen
	 */
	public void newGame(int cols, int rows) {
		this.cols = cols;
		this.rows = rows;

		// reset all data
		game = new int[cols][rows];

		for (int i = 0; i < cols; i++) {
			for (int j = 0; j < rows; j++) {
				game[i][j] = -1;
			}
		}
		nrMoves = 0;
		winningPlayer = -1;
		moves.removeAllElements();
		tmpMoves.removeAllElements();

		// who begins?
		playerNr = new Random(new Date().getTime()).nextInt(99) % 2;

		repaint();
	}

	/** Setzt Spieler und Gegner
	 * @param p Spieler
	 * @param nr 0 = Spieler, 1 = Gegner
	 */
	public void setPlayer(Player p, int nr) {
		players[nr] = p;
	}

	public void paint(Graphics g) {
		try {
			((Graphics2D) g).setRenderingHint(
				RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);
		} catch (Exception e) {
		}

		g.setColor(this.getBackground());
		g.fillRect(
			0,
			0,
			(int) this.getSize().getWidth(),
			(int) this.getSize().getHeight());

		g.setColor(this.getForeground());
		int i, j;
		for (i = 0; i <= rows; i++) {
			g.drawLine(
				offset,
				offset + i * gridSize,
				offset + gridSize * cols,
				offset + i * gridSize);
		}

		for (i = 0; i <= cols; i++) {
			g.drawLine(
				offset + i * gridSize,
				offset,
				offset + i * gridSize,
				offset + gridSize * rows);
		}

		for (i = 0; i < cols; i++) {
			for (j = 0; j < rows; j++) {
				if (game[i][j] >= 0) {
					g.setColor(colors[game[i][j]]);
					g.fillOval(
						offset + i * gridSize + gridSize / 2 - size / 2,
						offset
							+ (rows - j - 1) * gridSize
							+ gridSize / 2
							- size / 2,
						size,
						size);
				}
			}
		}

		g.setColor(colors[0]);
		g.fillOval(offset + cols * gridSize + 100, 100, size, size);
		g.setColor(Color.BLACK);
		g.drawString(
			players[0].getName(),
			offset + cols * gridSize + 120 + size,
			100 + size);

		g.setColor(colors[1]);
		g.fillOval(offset + cols * gridSize + 100, 140, size, size);
		g.setColor(Color.BLACK);
		g.drawString(
			players[1].getName(),
			offset + cols * gridSize + 120 + size,
			140 + size);

		if (nrMoves >= cols * rows) {
			g.setColor(Color.BLACK);
			g.drawString(
				"Game over",
				offset + cols * gridSize + 120 + size,
				180 + size);
		} else {
			if (winningPlayer == -1) {
				g.setColor(colors[playerNr]);
				g.fillOval(offset + cols * gridSize + 100, 180, size, size);
				g.setColor(Color.BLACK);
				g.drawString(
					"Am Zug: " + players[playerNr].getName(),
					offset + cols * gridSize + 120 + size,
					180 + size);
			} else {
				g.setColor(colors[winningPlayer]);
				g.fillOval(offset + cols * gridSize + 100, 180, size, size);
				g.setColor(Color.BLACK);
				g.drawString(
					players[winningPlayer].getName() + " hat gewonnen.",
					offset + cols * gridSize + 120 + size,
					180 + size);
			}
		}
	}

	/** F&uuml;rt einen Zug in Spalte <i>col</i> aus.
	 * Die Testz&uuml;ge werden zuerst entfernt.
	 * @param col Spalte
	 * @param player Spieler
	 * @return Falls der Zug g&uml;ltig ist, die Zeile in der der Stein gefallen ist, -1 sonst
	 */
	public int play(int col, int player) {
		removeAllTestPlay();
		for (int j = 0; j < rows; j++) {
			if (game[col][j] == -1) {
				game[col][j] = player;
				moves.addElement(new Move(col, j, player, nrMoves));
				nrMoves++;
				repaint();

				return j;
			}
		}

		return -1;
	}

	/** F&uuml;rt einen Testzug in Spalte <i>col</i> aus.
	 * @param col Spalte
	 * @param player Spieler
	 * @return Falls der Zug g&uml;ltig ist, die Zeile in der der Stein gefallen ist, -1 sonst
	 */
	public int testPlay(int col, int player) {
		for (int j = 0; j < rows; j++) {
			if (game[col][j] == -1) {
				game[col][j] = player;
				tmpMoves.addElement(new Move(col, j, player, nrMoves));
				return j;
			}
		}

		return -1;
	}

	/** Liefert die noch freien Spalten im Spielfeld
	 * @return freie Spalten
	 */
	public int[] getPossibleCols() {
		int ret[];
		int count = 0;

		for (int i = 0; i < cols; i++) {
			if (game[i][rows - 1] < 0)
				count++;
		}

		ret = new int[count];

		count = 0;
		for (int i = 0; i < cols; i++) {
			if (game[i][rows - 1] < 0) {
				ret[count++] = i;
			}
		}

		return ret;
	}

	/**
	 * L&ouml;scht den letzten Testzug
	 */
	public void removeTestPlay() {
		if (tmpMoves.size() > 0) {
			Move move = (Move) tmpMoves.elementAt(tmpMoves.size() - 1);
			tmpMoves.removeElementAt(tmpMoves.size() - 1);
			game[move.col][move.row] = -1;
		}
	}

	/**
	 * L&ouml;scht alle Testz&uuml;ge
	 */
	public void removeAllTestPlay() {
		Move move;
		Iterator it = tmpMoves.iterator();
		while (it.hasNext()) {
			move = (Move) it.next();
			game[move.col][move.row] = -1;
		}
		tmpMoves.removeAllElements();
	}

	/** Ermittelt, ob die aktuelle Stellung eine Gewinnstellung f&uuml;r Spieler <i>player</i> ist
	 * @param player Spielernummer
	 * @return 
	 */
	public boolean isWinningPosition(int player) {

		boolean winning = false;
		for (int i = 0; i < cols; i++) {
			for (int j = 0; j < rows; j++) {
				if (game[i][j] == player) {

					// Zeile, Spalte, Diagonale nach rechst oben, Diagonale nach links oben
					if ((i + 3) < cols) {
						winning =
							(game[i + 1][j] == player)
								&& (game[i + 2][j] == player)
								&& (game[i + 3][j] == player);
					}
					if ((j + 3) < rows) {
						winning =
							winning
								|| ((game[i][j + 1] == player)
									&& (game[i][j + 2] == player)
									&& (game[i][j + 3] == player));
					}
					if (((j + 3) < rows) && ((i + 3) < cols)) {
						winning =
							winning
								|| ((game[i + 1][j + 1] == player)
									&& (game[i + 2][j + 2] == player)
									&& (game[i + 3][j + 3] == player));
					}

					if ((j > 2) && ((i + 3) < cols)) {
						winning =
							winning
								|| ((game[i + 1][j - 1] == player)
									&& (game[i + 2][j - 2] == player)
									&& (game[i + 3][j - 3] == player));
					}

					if (winning)
						return true;

				}
			}
		}

		return winning;
	}

	public void sleep(int time) {
		try {
			Thread.sleep(time);
		} catch (Exception e) {
		}
	}

	public void run() {
		// Wer beginnt?
		playerNr = new Random(new Date().getTime()).nextInt(99) % 2;
		repaint();

		while (true) {
			// abwechselnd ein Zug
			//System.out.println(players[playerNr] + " ist am Zug");
			players[playerNr].play();
			sleep(1000);
			if (isWinningPosition(playerNr)) {
				winningPlayer = playerNr;
				repaint();
				return;
			}
			playerNr = (playerNr + 1) % 2;
			repaint();

			// ist das Spiel zu Ende?
			if (nrMoves >= cols * rows) {
				return;
			}
		}
	}

	/** Liefert die Anzahl der Spalten
	 * @return 
	 */
	public int getCols() {
		return cols;
	}

	/** Liefert die Anzahl der Zeilen
	 * @return
	 */
	public int getRows() {
		return rows;
	}


	/** Liefert die Anzahl der Z&uuml;ge
	 * @return
	 */
	public int getNrMoves() {
		return nrMoves;
	}

	/**
	 * @return
	 */
	public int getGridSize() {
		return gridSize;
	}

	public int getOffset() {
		return offset;
	}

	/** Liefert die beiden Spieler
	 * @return
	 */
	public Player[] getPlayers() {
		return players;
	}

}