
#include "dar.h"

using namespace std;

namespace papi
{

    Dar::Dar():
        ind(CHAR_MIN,CHAR_MAX)
    {}

    Dar::~Dar()
    {
        if(alpha_prob)
            delete [] alpha_prob;
        if(character_prob)
            delete [] character_prob;
        if(last_k)
            delete last_k;
    }

    void Dar::estimateParameters(long double *autocorrelation)
    {

        if(order <= 0) {
            rho = 0.0;
            return;
        }
        boost::numeric::ublas::matrix<double> A(order,order);
        boost::numeric::ublas::vector<double> b(order);
        
        /*
         * - (a_p * r_{p-1}) = a_1 * r_0 + ... + a_{p-1} * r_{p-2} - 1/rho * r_1
         * ...
         * - (a_p * r_0) = a_1 * r_{p-1} + ... + a_{p-1} * r_1 - 1/rho * r_p
         *
         * The r_i's are the autocorrelation function values and are given by 
         * the parameter file.
         * By assuming a_p is known and using a_p as the unit of the solution 
         * vector, the remaining unknowns are a_1,..,a_{p-1} and 1/rho.
         * The equation system is then linear, with matrix A and vector b.
         * The vector b contains the coefficients of a_p.
         * 
         * Let alpha_prob be the solution of the linear equation.
         * Because  1 = a_1                 + ... + a_{p-1}                 + a_p
         *          1 = a_p * alpha_prob_1  + ... + a_p * alpha_prob_{p-1}  + a_p
         * and      alpha_prob_1 + ... + alpha_prob_{p-1} + 1 = 1/a_p
         * holds.
         * Since 1/rho = alpha_prob_p * a_p it follows
         * rho = (alpha_prob_1 + ... + alpha_prob_{p-1} + 1) / alpha_prob_p
         */
        for(int row=0;row<order;++row) 
        {
            for(int column=0;column<order;++column)
            {
                if(column<order-1)
                    A(row,column) = (double)autocorrelation[abs(row-column)];
                else {
                    A(row,column) = -(double)autocorrelation[row+1];
                    b(row) = -(double)autocorrelation[abs(row-column)];
                }
                
            }    
        }
        
        solveLinearEquation(A, b);
        
        alpha_prob = new double[order];
        
        for(int i=0; i<order; ++i) {
            alpha_prob[i] = b(i);
        }

        long double alpha_p = 1;
        
        for(int i=0;i<order-1;++i)
        {
            alpha_p+=alpha_prob[i];
        }
        
        if(alpha_prob[order-1]<0.00001)
        {
            rho = 0;
            return;
        }
        rho = alpha_p/alpha_prob[order-1];
        
        
        if(rho<=0.0)
        {
            rho = 0.0;
            return;
        }
        alpha_prob[order-1] = 1;
        long double sum=0;
        for(int i=0;i<order;++i)
        {
            alpha_prob[i]/=alpha_p;
            if(alpha_prob[i]<0.0)
                alpha_prob[i] = 0.0;
            sum+=alpha_prob[i];
        }
        
        for(int i=0;i<order;++i)
        {
            alpha_prob[i]/=sum;

        }    
    }

    void Dar::init(const char *file_autocorrelation, const char *file_prob)
    {
        long double *autocorrelation;
        
        ParameterParser parser;
        
        parser.requestCharacterDistribution(file_prob, ind, character_prob);
        parser.requestAutocorrelationDar(file_autocorrelation, autocorrelation, order);
        parser.start();
        estimateParameters(autocorrelation);
        delete [] autocorrelation;
    }


    void Dar::generateSequence(long long length)
    {
        
        last_k = new QueueFixedSizeRandomAccess<int>(order, 0);

        long long l;
        
        double p[2];
        p[0] = rho;
        p[1] = 1-rho;

        for(l=0;l<(long long)order && l<length;++l)
        {
            int c = die.roll(ind.size,character_prob);
            write(ind.getValue(c));
            last_k->push_back(c);
        }
        
        for(;l<length;++l)
        {
            int decision = die.roll(2, p);
            // Copy operation
            if(decision==0)
            {
                int index = die.roll(order, alpha_prob);
                int c = (*last_k)[last_k->size-index-1];
                write(ind.getValue(c));
                last_k->push_back(c);
            }
            // No copy operation
            else {
                int c = die.roll(ind.size,character_prob);
                write(ind.getValue(c));
                last_k->push_back(c);
            }
        }
    }

    void Dar::generate( long long length,  map<string,string> &parameters)
    {
        map<string,string>::iterator it;
        
        const char *file_autocorrelation = NULL, *file_prob = NULL;
        
        if( (it = parameters.find("autocorrelation_dar"))!=parameters.end()) {
            file_autocorrelation = it->second.c_str();
        }

        if( (it = parameters.find("character_distribution"))!=parameters.end()) {
            file_prob = it->second.c_str();
        }
        
        init(file_autocorrelation,file_prob);
        
        generateSequence(length);
    }
        
    
}

