#ifndef _format_helper_in_h_
#define _format_helper_in_h_

#include <fstream>
#include <iostream>
#include <string>

#include <boost/unordered_map.hpp>

#include "csv_in_stream.h"
#include "index_map.h"
#include "matrix.h"
#include "random.h"

using namespace std;

namespace papi {
    
    /**
     * Skip the header of csv formatted parameter files.
     * @param[in] csv The csv stream
     */
    void skipCsvHeader(CsvInStream &csv);
    
    /**
     * @class ParameterParser
     * A class to parse csv formatted parameter files.
     */
    class ParameterParser {
    public:
        /**
         * Request the parsing of a character distribution file.
         * 
         * @param[in] file_name Path to the file or NULL if read from stdin
         * @param[out] ind_map The index map which should be updated with the read values
         * @param[out] distribution Pointer to the array where the distribution is stored
         */
        void requestCharacterDistribution(const char *file_name,
                                         IndexMap<char> &ind_map,
                                         double *&distribution);
        
        /**
         * Request parsing qgram distribution file and claculation of 
         * conditional character distribution.
         *
         * @param[in] file_name Path to the parameter file, or NULL if stdin
         * @param[out] ind_map Index map which should be updatet
         * @param[out] cp_order The markov order of the conditional probabilities
         * @param[out] conditional_probability Pointer to the matrix where the 
         *              the distribution were stored
         * @param[out] start_word The start word for a markov chain which was chosen by the parser.
         */
        void requestConditionalDistribution(const char *file_name, IndexMap<char> &ind_map, 
                                            int &cp_order, NDimMat<double *> *&conditional_probability,
                                            char *&start_word);
        
        /**
         * Request parsing parameters of a repeat machine
         *
         * @param[in] file_name Path to the parameter file, or NULL if stdin
         * @param[in] machine_name Name of the machine type
         * @param[out] prob_start Start probability of the repeat machine
         * @param[out] prob_end End probability of the repeat machine
         * @param[out] prob_match Match probability of the repeat machine
         * @param[out] prob_change Change probability of the repeat machine
         * @param[out] prob_insert Insert probability of the repeat machine
         * @param[out] prob_delete Delete probability of the repeat machine
         * @param[out] complement Pointer to the array which stores the complement
         *                          of a character (only for inverted repeat machines)
         */
        void requestRepeatMachine(const char *file_name,
                                  const char *machine_name,
                                  double &prob_start,
                                  double &prob_end,
                                  double &prob_match,
                                  double &prob_change,
                                  double &prob_insert,
                                  double &prob_delete,
                                  char **complement);
        
        /**
         * Request parsing of autocorrelation parameters of a discrete autoregresive proces.
         * 
         * @param[in] file_name Path to the parameter file, or NULL if stdin
         * @param[out] autocorrelation_dar Array of the parsed autocorrelation parameters
         * @param[out] order Order of the dar process
         */
        void requestAutocorrelationDar(const char *file_name,
                                       long double *&autocorrelation_dar,
                                       int &order);
        
        /**
         * Start parsing the files.
         * When this function finishes all parameter files were parsed.
         * No parser will work correctly after this function call.
         */
        void start();
        
    protected:
        /**
         * @class ParserFunction
         *
         * Base class for a parser function which can be associated with a
         * parameter file type and will be called by the parser.
         */
        class ParserFunction {
        public:
            /// File stream to the parameter file
            ifstream file_stream;
            /**
             * Execute the function 
             * 
             * @param[in] stream Stream from which to parse the parameters
             */
            virtual void operator() (istream &stream) = 0;
            
            /**
             * Constructor
             *
             * @param[in] file_name Path to the parameter file or NULL if stdin
             */
            ParserFunction(const char *file_name);
            
            /**
             * Destructor
             */
            virtual ~ParserFunction();
        };
        
        
        /**
         * @class ParseConditionalDistribution
         * 
         * Function to parse qgram distribution file and calculate
         * conditional distribution.
         */
        class ParseConditionalDistribution : public ParserFunction {
        private:
            /// Index map which should be updated
            IndexMap<char> &ind_map;
            /// Markov order of the conditional distribution
            int &cp_order;
            /// Conditional distributions
            NDimMat<double *> *&conditional_probability;
            /// Chosen start word
            char *&start_word;
        public:
            /**
             * Constructor
             *
             * @param[in] file_name Path to the parameter file, or NULL if stdin
             * @param[out] ind_map Index map which should be updatet
             * @param[out] cp_order The markov order of the conditional probabilities
             * @param[out] conditional_probability Pointer to the matrix where the 
             *              the distribution were stored
             * @param[out] start_word The start word for a markov chain which was chosen by the parser.
             */
            ParseConditionalDistribution(const char *file_name, IndexMap<char> &ind_map, 
                                         int &cp_order,NDimMat<double *> *&conditional_probability,
                                         char *&start_word);
            
            /// Overrides ParserFunction::operator()
            void operator() (istream &stream);
        };
        
        /**
         * @class ParseCharacterDistribution
         *
         * Function to parse character distribution parameter file
         */
        class ParseCharacterDistribution : public ParserFunction {
        private:
            /// Index map which should be updated
            IndexMap<char> &ind_map;
            /// The character distribution
            double *&distribution;
        public:
            /**
             * Constructor
             *
             * @param[in] file_name Path to the file or NULL if read from stdin
             * @param[out] ind_map The index map which should be updated with the read values
             * @param[out] distribution Pointer to the array where the distribution is stored
             */
            ParseCharacterDistribution(const char *file_name,
                                      IndexMap<char> &ind_map,
                                      double *&distribution);
            /// Overrides ParserFunction::operator()
           void operator() (istream &stream);
        };
        
        /**
         * @class ParseAutocorrelationDar 
         * 
         * Function to parse autocorrelation parameters for a dar process
         */
        class ParseAutocorrelationDar : public ParserFunction {
        private:
            /// The autocorrelation parameters
            long double *&autocorrelation_dar;
            /// Order of the dar process
            int &order;
        public:
            /**
             * Constructor
             *
             * @param[in] file_name Path to the parameter file, or NULL if stdin
             * @param[out] autocorrelation_dar Array of the parsed autocorrelation parameters
             * @param[out] order Order of the dar process
             */
            ParseAutocorrelationDar(const char *file_name,
                                    long double *&autocorrelation_dar,
                                    int &order);
            /// Overrides ParserFunction::operator()
            void operator() (istream &stream);
        };
        
        /**
         * @class ParseRepeatMachine
         * 
         * Function to parse repeat machine parameters
         */
        class ParseRepeatMachine : public ParserFunction {
        private:
            /// Start probability of the repeat machine
            double &prob_start;
            /// End probability of the repeat machine
            double &prob_end;
            /// Match probability of the repeat machine
            double &prob_match;
            /// Change probability of the repeat machine
            double &prob_change;
            /// Insert probability of the repeat machine
            double &prob_insert;
            /// Delete probability of the repeat machine
            double &prob_delete; 
            /// Complement to each character or NULL if not inverted repeat machine
            char **complement;
            
        public:
            /**
             * Constructor
             *
             * @param[in] file_name Path to the parameter file, or NULL if stdin
             * @param[out] prob_start Start probability of the repeat machine
             * @param[out] prob_end End probability of the repeat machine
             * @param[out] prob_match Match probability of the repeat machine
             * @param[out] prob_change Change probability of the repeat machine
             * @param[out] prob_insert Insert probability of the repeat machine
             * @param[out] prob_delete Delete probability of the repeat machine
             * @param[out] complement Pointer to the array which stores the complement
             *                          of a character (only for inverted repeat machines)
             */
            ParseRepeatMachine(const char *file_name,
                               double &prob_start,
                               double &prob_end,
                               double &prob_match,
                               double &prob_change,
                               double &prob_insert,
                               double &prob_delete,
                               char **complement);
            /// Overrides ParserFunction::operator()
            void operator() (istream &stream);
        };
              
        /// Map which maps a parameter file type to a function to be executed
        boost::unordered_map<string, ParserFunction*> requests;
        
    };
    
}

#endif
