#ifndef _hash_map_h_
#define _hash_map_h_

#include <list>
#include <string>

#include <boost/unordered_map.hpp>

using namespace std;
using namespace boost;

namespace papi
{
    /**
     * @struct eq
     *
     * Collection of equality functions for use with hash maps
     */
    struct eq {
        
        /**
         * @struct fixed_length
         * Compares C-Strings with fixed size and without terminating null-character
         */
        struct fixed_length : binary_function<unsigned char const *, unsigned  char const *, bool>
        {
            /// length of the C-Strings
            const int length;
            
            /**
             * Constructor
             *
             * @param[in] length Length of the Strings
             */
            fixed_length(int length):
            length(length)
            {}
            
            /**
             * Compares the strings
             *
             * @param[in] a First string
             * @param[in] b Second string
             * @return True, if strings are equal. Else return false
             */
            bool operator()( unsigned char const * a, unsigned  char const * b) const;
        };
        
        
        /**
         * @struct pair_unsigned_cstring_int
         *
         * Compares C-Strings of variable length without terminating character. 
         * Strings are given as a pair of length and char array.
         */
        struct pair_unsigned_cstring_int : binary_function<pair<unsigned char const*,int>&, pair<unsigned char const*,int>&, bool> {
            /**
             * Compares the strings
             *
             * @param[in] a First string
             * @param[in] b Second string
             * @return True, if strings are equal. Else return false
             */
            bool operator()(const pair<unsigned char const*,int>& a, const pair<unsigned char const*,int>& b) const;            
        };
        
    };
        
    /**
     * @struct hash_djb2
     *
     * Collection of hash djb2 functions for use with hash maps
     */
    struct hash_djb2{
        /**
         * @struct fixed_length
         * 
         * DJB2 hash function for C-Strings with fixed size and without terminating character
         */
        struct fixed_length : unary_function<std::string, long long>
        {
            /// length of the C-Strings
            const int length;
            
            /**
             * Constructor
             *
             * @param[in] length Length of the Strings
             */
            fixed_length(int length):
            length(length)
            {}
            
            /**
             * Calculate the hash value
             *
             * @param[in] str The string
             * @return The hash value
             */
            long long operator()( unsigned char const * str) const;
        };
        
        /**
         * @struct pair_unsigned_cstring_int
         * 
         * Hash function of C-Strings with variable length without terminating character. 
         * Strings are given as a pair of length and char array.
         */
        struct pair_unsigned_cstring_int :  unary_function<std::string, long long> {
            /**
             * Calculate the hash value
             *
             * @param[in] str The string
             * @return The hash value
             */
            long long operator()(const pair<unsigned char const*,int> &str) const;
        };
    };

    ///hash map of type <unsigned char const* , std::list<int>> with fixed C-string length
    typedef unordered_map<unsigned char const *, std::list<long long>, hash_djb2::fixed_length, eq::fixed_length> unsigned_cstring_fixed_length_list_long_long_hash_map;
    ///hash map of type <unsigned char const* , float *> with fixed C-string length
    typedef unordered_map<unsigned char const *, float*, hash_djb2::fixed_length, eq::fixed_length> unsigned_cstring_fixed_length_ptr_float_hash_map;
    ///hash map of type <unsigned char const* , long long> with fixed C-string length
    typedef unordered_map<unsigned char const *, long long, hash_djb2::fixed_length, eq::fixed_length> unsigned_cstring_fixed_length_long_long_hash_map;
    ///hash map of type <pair<unsigned char const*,int>, long long> for C-Strings with variable length and no terminating character
    typedef unordered_map<pair<unsigned char const*,int>, long long, hash_djb2::pair_unsigned_cstring_int, eq::pair_unsigned_cstring_int> pair_unsigned_cstring_int_long_long_hash_map;

}

#endif

