#ifndef _queue_fixed_size_h_
#define _queue_fixed_size_h_

#include <iostream>

namespace papi
{
    /**
     * @class QueueFixedSizeRandomAccess
     *
     * Queue of fixes size which stores its content sequentially making it 
     * possible to treat it as an ordinary array
     *
     * It is only possible to add elements. If the queue hits its maximum size
     * old elements are deleted.
     *
     */
    template <class T>
    class QueueFixedSizeRandomAccess
    {
    protected:
        /// Buffer containing the content
        T *buffer;
    public:
        ///Pointer to the start of the queue
        T *start;
        ///Pointer to the last element of the queue (a chosen delimiter character)
        T *end;
        /// Delimiter char which is garantueed to be the last element in the queue. Makes it possible to use the queue as a C string
        T delimiter;
        /// current size
        int size;
        /// maximum size
        int max_size;
        
        /**
         * Constructor
         * @param max_size Maximal size of the queue
         * @param delimiter Delimiter to be used after the last element in the queue
         */
        QueueFixedSizeRandomAccess(int max_size, T delimiter):
        delimiter(delimiter),size(0),max_size(max_size)
        {
            buffer = new T[max_size<<1];
            start = end = buffer;
            
            memset(buffer, delimiter, sizeof(T)*(max_size<<1));
        }
        
        /**
         * Insert element at the end of the queue
         * @param[in] el New element
         */
        void push_back(T el)
        {
            if(max_size==0)
                return;
            if(size == max_size)
            {
                if(start==(buffer+max_size-1))
                {
                    memcpy(buffer, start+1, sizeof(T)*(max_size-1));
                    buffer[max_size-1] = el;
                    memset(start+1,delimiter,sizeof(T)*(max_size-1));
                    start = buffer;
                    end = buffer+max_size;
                }
                else {
                    ++start;
                    *end++ = el;
                }
                
            }
            else {
                ++size;
                *end++ = el;
            }
        }
        
        /**
         * Access queue like an array
         *
         * @param[in] i Index
         * @return Element at position i
         */
        T& operator[](const int i)
        {
            return start[i];
        }
    };

}

#endif

