#ifndef _random_util_h_
#define _random_util_h_


#include <algorithm>
#include <ctime>
#include <fstream>
#include <numeric>
#include <vector>

#include <boost/random.hpp>


namespace papi
{
    /**
     * @class RandomDouble
     * Random number generator with uniform distribution in a chosen interval
     */
    class RandomDouble
    {
    public:
        /**
         * Constructor. Generator not initialized.
         */
        RandomDouble();
        
        /**
         * Constructor. Generator is initialized.
         * @param min lower bound (inclusive)
         * @param max upper bound (exclusive)
         */
        RandomDouble(double min, double max);
        
        /**
         * Destructor
         */
        ~RandomDouble();
        
        /**
         * Generate uniformly distributed random number
         * @return Uniformaly distributed random number in [min,max)
         */
        double generate();
        
        /// Initialize the generator if the default constructor was used
        /**
         * @param min lower bound (inclusive)
         * @param max upper bound (exclusive)
         */
        void init(double min, double max);
    protected:
        /// Underlying random number generator
        boost::variate_generator<boost::mt19937, boost::uniform_real<> > *generator;
        
    };

    /**
     * @class Die
     * A weighted die
     */
    class Die
    {
    public:
        /**
         * Constructor
         */
        Die():
            random(0,1)
        {}
        
        /**
         * Given a distribution over the indices this function chooses one. 
         * If the probabilities do not sum up to one they are normalized.
         * Calls #roll(num,prob,-1)
         *
         * @param num Size of distribution array
         * @param prob Distribution array
         * @return Chosen index or -1 on failure
         */
        int roll(int num, const double *prob);
        
        /**
         * Given a distribution over the indices this function chooses one. 
         * A single index may be excluded from the distribution.
         * If the probabilities do not sum up to one they are normalized.
         *
         * @param num Size of distribution array
         * @param prob Distribution array
         * @param excludedIndex index to be excluded or -1 if none
         * @return Chosen index or -1 on failure
         */
        int rollWithout(int num, const double *prob, int excludedIndex);
        
    protected:
        /// Underlying random number generator
        RandomDouble random;
    };
	
}

#endif

