// ==========================================================================
//                        find_index_approx_benchmark
// ==========================================================================
// Copyright (c) 2006-2010, Knut Reinert, FU Berlin
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * Neither the name of Knut Reinert or the FU Berlin nor the names of
//       its contributors may be used to endorse or promote products derived
//       from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL KNUT REINERT OR THE FU BERLIN BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
// DAMAGE.
//
// ==========================================================================
// Author: Johannes Krugel <krugel@in.tum.de>
// ==========================================================================
// Benchmark for the SeqAn module find_index_approx.
// ==========================================================================

// This program has to be compiled with template parameters specifying the index
// class and its parameters. Details see below and in ../scripts/benchmark.php

#ifndef SANDBOX_TUM_APPS_FIND_INDEX_APPROX_BENCHMARK_FIND_INDEX_APPROX_BENCHMARK_H_
#define SANDBOX_TUM_APPS_FIND_INDEX_APPROX_BENCHMARK_FIND_INDEX_APPROX_BENCHMARK_H_

#include <cstdio>
#include <ctime>
#include <locale>
#include <fstream>
#include <iostream>
#include <sstream>
#include <string>

// core
#include <seqan/basic.h>
#include <seqan/sequence.h>
#include <seqan/find.h>
#include <seqan/index.h>
#include <seqan/misc/misc_cmdparser.h>

// extras
#include <seqan/index_extras.h>

// sandbox/tum
#include <seqan/find_index_approx.h>
#include <seqan/index_compressed.h>
#include <seqan/index_qgram_ext.h>
#include <seqan/index_suffix_trees.h>

// Is created during build in bin folder
#include "misc_resources.h"

using namespace seqan;

// ============================================================================
// Forwards
// ============================================================================

// ============================================================================
// Tags, Classes, Enums
// ============================================================================

enum EncodingFlag {ENCODING_DIRECT, ENCODING_UTF8};
enum FileFormatFlag {FILE_FORMAT_RAW, FILE_FORMAT_FASTA};
enum AlphabetFlag {ALPHABET_BOOL, ALPHABET_CHAR, ALPHABET_WCHAR, ALPHABET_DNA, ALPHABET_DNA5, ALPHABET_AMINOACID};
enum QueryType {QUERY_ONE, QUERY_ALL_END, QUERY_ALL_BEGIN_END};
const std::string ALPHABET_NAMES[6] = {"bool", "char", "wchar", "Dna", "Dna5", "AminoAcid"};
const unsigned L2_CACHE_SIZE = 500 * 1024;

#ifndef NDEBUG
    const CharString BUILD_TYPE = "Debug";
#else
    const CharString BUILD_TYPE = "Release";
#endif
#ifdef SEQAN_BENCHMARK_PARAM
    // https://gcc.gnu.org/onlinedocs/cpp/Stringification.html
    #define SEQAN_BENCHMARK_ALLOC 0
    #define SEQAN_BENCHMARK_EXTERNAL 1
    #define STRINGIFY(X) STRINGIFY_HELPER(X)
    #ifdef SEQAN_BENCHMARK_IndexEsa
        #define STRINGIFY_HELPER(A,B) #A ", " #B
        #define SEQAN_BENCHMARK_PAGESIZE_(A,B) A
        #define SEQAN_BENCHMARK_PAGESIZE(X) SEQAN_BENCHMARK_PAGESIZE_(X)
        #define SEQAN_BENCHMARK_SUFFBUFF_(A,B) B
        #define SEQAN_BENCHMARK_SUFFBUFF(X) SEQAN_BENCHMARK_SUFFBUFF_(X)
    #endif
    #if defined SEQAN_BENCHMARK_IndexWotd || defined SEQAN_BENCHMARK_IndexSttd64
        //SEQAN_BENCHMARK_PARAM = STORAGE, PREFIX_LENGTH, PAGESIZE, STRBUFF, TREEBUFF, TEMPBUFF, SUFFBUFF, TOTAL_MEM_SIZE
        #define STRINGIFY_HELPER(A,B,C,D,E,F,G,H) #A ", " #B ", " #C ", " #D ", " #E ", " #F ", " #G ", " #H
        #define SEQAN_BENCHMARK_STORAGE_(A,B,C,D,E,F,G,H) A
        #define SEQAN_BENCHMARK_STORAGE(X) SEQAN_BENCHMARK_STORAGE_(X)
        #define SEQAN_BENCHMARK_PREFIX_LENGTH_(A,B,C,D,E,F,G,H) B
        #define SEQAN_BENCHMARK_PREFIX_LENGTH(X) SEQAN_BENCHMARK_PAGESIZE_(X)
        #define SEQAN_BENCHMARK_PAGESIZE_(A,B,C,D,E,F,G,H) C
        #define SEQAN_BENCHMARK_PAGESIZE(X) SEQAN_BENCHMARK_PAGESIZE_(X)
        #define SEQAN_BENCHMARK_STRBUFF_(A,B,C,D,E,F,G,H) D
        #define SEQAN_BENCHMARK_STRBUFF(X) SEQAN_BENCHMARK_STRBUFF_(X)
        #define SEQAN_BENCHMARK_TREEBUFF_(A,B,C,D,E,F,G,H) E
        #define SEQAN_BENCHMARK_TREEBUFF(X) SEQAN_BENCHMARK_TREEBUFF_(X)
        #define SEQAN_BENCHMARK_TEMPBUFF_(A,B,C,D,E,F,G,H) F
        #define SEQAN_BENCHMARK_TEMPBUFF(X) SEQAN_BENCHMARK_TEMPBUFF_(X)
        #define SEQAN_BENCHMARK_SUFFBUFF_(A,B,C,D,E,F,G,H) G
        #define SEQAN_BENCHMARK_SUFFBUFF(X) SEQAN_BENCHMARK_SUFFBUFF_(X)
        #define SEQAN_BENCHMARK_CONFIG_(A,B,C,D,E,F,G,H) B,C,D,E,F,G,H
        #define SEQAN_BENCHMARK_CONFIG(X) SEQAN_BENCHMARK_CONFIG_(X)
    #endif
    #ifdef SEQAN_BENCHMARK_IndexDigest
        //SEQAN_BENCHMARK_PARAM = ExternalConfig<...>
        #define STRINGIFY_HELPER(A,B,C,D,E) #A ", " #B ", " #C ", " #D ", " #E
        #define SEQAN_BENCHMARK_CONFIG_(A,B,C,D,E) A,B,C,D,E
        #define SEQAN_BENCHMARK_CONFIG(X) X
    #endif
    #ifdef SEQAN_BENCHMARK_IndexSadakane
        #define STRINGIFY_HELPER(A,B,C) #A ", " #B ", " #C
        #define SEQAN_BENCHMARK_CONFIG_(A,B,C) A,B,C
        #define SEQAN_BENCHMARK_CONFIG(X) X
    #endif
    #ifdef SEQAN_BENCHMARK_IndexQGram
        //SEQAN_BENCHMARK_PARAM = STORAGE, PAGESIZE, SUFFBUFF
        #define STRINGIFY_HELPER(A,B,C) #A ", " #B ", " #C
        #define SEQAN_BENCHMARK_STORAGE_(A,B,C) A
        #define SEQAN_BENCHMARK_STORAGE(X) SEQAN_BENCHMARK_STORAGE_(X)
        #define SEQAN_BENCHMARK_PAGESIZE_(A,B,C) B
        #define SEQAN_BENCHMARK_PAGESIZE(X) SEQAN_BENCHMARK_PAGESIZE_(X)
        #define SEQAN_BENCHMARK_SUFFBUFF_(A,B,C) C
        #define SEQAN_BENCHMARK_SUFFBUFF(X) SEQAN_BENCHMARK_SUFFBUFF_(X)
    #endif
    const CharString STATIC_INDEX_PARAMETERS = STRINGIFY(SEQAN_BENCHMARK_PARAM);
#else
    const CharString STATIC_INDEX_PARAMETERS = "";
#endif

struct Options {
    bool showHelp;
    bool showVersion;

    AlphabetFlag alphabet;
    EncodingFlag encoding;
    FileFormatFlag fileFormat;
    CharString textFile;
    CharString index;
    String<CharString> indexParameters;

    CharString patternFile;
    String<unsigned int> patternLengths;
    String<CharString> distanceMeasures;
    String<int> tolerances;
    double maxErrorLevel;
    String<QueryType> queryTypes;
    String<CharString> finders;
    
    bool flushSeqanCache;
    bool flushDiskCache;
    bool flushL2Cache;
    
    CharString version;
    
    // Set defaults.
    Options() {
        showHelp = false;
        showVersion = false;
        alphabet = ALPHABET_CHAR;
        encoding = ENCODING_DIRECT;
        fileFormat = FILE_FORMAT_RAW;
        maxErrorLevel = 1.0;
        appendValue(queryTypes, QUERY_ONE);
        version = "";
        flushSeqanCache = false;
        flushDiskCache = false;
        flushL2Cache = false;
    }
};

struct _OnlinePseudoIndex;
typedef Tag<_OnlinePseudoIndex> OnlinePseudoIndex;

struct _Optimum;
typedef Tag<_Optimum> Optimum;

// ============================================================================
// Metafunctions
// ============================================================================

template <typename TString, typename TIndexSpec>
struct HaystackFromIndexSpec {
    typedef Index<TString, TIndexSpec>                      Type;
};

template <typename TString>
struct HaystackFromIndexSpec<TString, OnlinePseudoIndex> {
    // Does not work because Reference<TString> uses Value<TString>& which is char&
    //typedef typename Reference<TString>::Type               Type;
    typedef TString&                                        Type;
};

template <typename TNeedle, typename TIndex, typename TPatternSpec>
struct IndexPattern {
    typedef TPatternSpec                                    Type;
};

template <typename TNeedle, typename TIndex, typename TSpec, typename TScore, typename TVerifyPatternSpec>
struct IndexPattern<TNeedle, TIndex, Partitioning<TSpec, TScore, Default, TVerifyPatternSpec> > {
    typedef typename DefaultIndexPattern<TNeedle, TIndex>::Type TPiecePatternSpec;
    typedef Partitioning<TSpec, TScore, TPiecePatternSpec, TVerifyPatternSpec> Type;
};

namespace seqan {

template <typename TText, typename TExternalConfig>
struct Fibre<Index<TText, IndexEsa<External<TExternalConfig> > >, FibreSA>
{
    typedef Index<TText, IndexEsa<External<TExternalConfig> > > TIndex;
    typedef typename SAValue<TIndex>::Type                  TSAValue;
    typedef String<TSAValue, External<TExternalConfig> >    Type;
};

template <typename TText, typename TExternalConfig>
struct Fibre<Index<TText, IndexEsa<External<TExternalConfig> > >, FibreLcp>
{
    typedef Index<TText, IndexEsa<External<TExternalConfig> > > TIndex;
    typedef typename Size<TIndex>::Type                     TSize;
    typedef String<TSize, External<TExternalConfig> >       Type;
};

// Lcpe in external memory does not compile?
//template <typename TText, typename TExternalConfig>
//struct Fibre<Index<TText, IndexEsa<External<TExternalConfig> > >, FibreLcpe>
//{
//    typedef Index<TText, IndexEsa<External<TExternalConfig> > > TIndex;
//    typedef typename Size<TIndex>::Type                     TSize;
//    typedef String<TSize, External<TExternalConfig> >       Type;
//};

template <typename TText, typename TExternalConfig>
struct Fibre<Index<TText, IndexEsa<External<TExternalConfig> > >, FibreChildtab>
{
    typedef Index<TText, IndexEsa<External<TExternalConfig> > > TIndex;
    typedef typename Size<TIndex>::Type                     TSize;
    typedef String<TSize, External<TExternalConfig> >       Type;
};

#ifdef SEQAN_BENCHMARK_IndexWotd
#if SEQAN_BENCHMARK_STORAGE(SEQAN_BENCHMARK_PARAM) == SEQAN_BENCHMARK_EXTERNAL
template <typename TText, typename TIndexSpec>
struct Fibre<Index<TText, IndexWotd<TIndexSpec> >, FibreSA>
{
    typedef Index<TText, IndexWotd<TIndexSpec> >            TIndex;
    typedef typename SAValue<TIndex>::Type                  TSAValue;
    static const size_t VALUE_SIZE = BytesPerValue<TSAValue>::VALUE;
    typedef ExternalConfig<ExternalConfig<>::TFile, SEQAN_BENCHMARK_PAGESIZE(SEQAN_BENCHMARK_PARAM) / VALUE_SIZE, SEQAN_BENCHMARK_SUFFBUFF(SEQAN_BENCHMARK_PARAM)> TExternalConfig;
    typedef String<TSAValue, External<TExternalConfig> >    Type;
};

template <typename TText, typename TIndexSpec>
struct Fibre<Index<TText, IndexWotd<TIndexSpec> >, FibreDir>
{
    typedef Index<TText, IndexWotd<TIndexSpec> >            TIndex;
    typedef typename Size<TIndex>::Type                     TSize;
    static const size_t VALUE_SIZE = BytesPerValue<TSize>::VALUE;
    typedef ExternalConfig<ExternalConfig<>::TFile, SEQAN_BENCHMARK_PAGESIZE(SEQAN_BENCHMARK_PARAM) / VALUE_SIZE, SEQAN_BENCHMARK_TREEBUFF(SEQAN_BENCHMARK_PARAM)> TExternalConfig;
    typedef String<TSize, External<TExternalConfig> >       Type;
};
#endif
#endif

#ifdef SEQAN_BENCHMARK_IndexQGram
#if SEQAN_BENCHMARK_STORAGE(SEQAN_BENCHMARK_PARAM) == SEQAN_BENCHMARK_EXTERNAL

// Dir should be held in main memory
//template <typename TText, typename TShapeSpec, typename TSpec>
//struct Fibre<Index<TText, IndexQGram<TShapeSpec, TSpec> >, FibreDir>
//{
//    typedef Index<TText, IndexQGram<TShapeSpec, TSpec> >    TIndex;
//    typedef typename Size<TIndex>::Type                     TSize;
//    static const size_t VALUE_SIZE = BytesPerValue<TSize>::VALUE;
//    typedef ExternalConfig<ExternalConfig<>::TFile, SEQAN_BENCHMARK_PAGESIZE(SEQAN_BENCHMARK_PARAM) / VALUE_SIZE, SEQAN_BENCHMARK_DIRBUFF(SEQAN_BENCHMARK_PARAM)> TExternalConfig;
//    typedef String<TSize, External<TExternalConfig> >       Type;
//};

template <typename TText, typename TShapeSpec, typename TSpec>
struct Fibre<Index<TText, IndexQGram<TShapeSpec, TSpec> >, FibreSA>
{
    typedef Index<TText, IndexQGram<TShapeSpec, TSpec> >    TIndex;
    typedef typename SAValue<TIndex>::Type                  TSAValue;
    static const size_t VALUE_SIZE = BytesPerValue<TSAValue>::VALUE;
    typedef ExternalConfig<ExternalConfig<>::TFile, SEQAN_BENCHMARK_PAGESIZE(SEQAN_BENCHMARK_PARAM) / VALUE_SIZE, SEQAN_BENCHMARK_SUFFBUFF(SEQAN_BENCHMARK_PARAM)> TExternalConfig;
    //typedef ExternalConfig<>                                TExternalConfig;
    typedef String<TSAValue, External<TExternalConfig> >    Type;
};
#endif
#endif

} // seqan namespace

// ============================================================================
// Functions
// ============================================================================

// ----------------------------------------------------------------------------
// Helper functions
// ----------------------------------------------------------------------------

template<typename T>
std::string myToString(const T& t) {
    std::ostringstream stream;
    stream << t;
    return stream.str();
}

template<typename T, typename TStringSpec>
String<T> stringConcat(const String<T, TStringSpec> & str) {
    String<T> result;
    for (unsigned int i = 0; i < length(str); ++i) {
        if (i != 0) append(result, ", ");
        append(result, str[i]);
    }
    return result;
}

std::string currentTime() {
    std::ostringstream stream;
    std::time_t timestamp = std::time(0u);
    std::tm *localTime = std::localtime(&timestamp);
    stream << localTime->tm_year+1900u << "-" << localTime->tm_mon+1 << "-" << localTime->tm_mday << " ";
    stream << localTime->tm_hour << ':' << localTime->tm_min << ':' << localTime->tm_sec;
    return stream.str();
}

// This function is defined for Pattern<Partitioning>.
// For all other kinds of patterns we won't call it, but the compiler wants a definition also for these cases.
template <typename TPattern, typename TPiecesSize>
void setNumberOfPieces(TPattern & /*pattern*/, TPiecesSize /*numberOfPieces*/) {
    SEQAN_CHECKPOINT;
    // do nothing
}

template <typename TFinder>
void printDebugInfo(TFinder & /*finder*/) {
    SEQAN_CHECKPOINT;
    // do nothing
}

// ----------------------------------------------------------------------------
// Command line parsing functions
// ----------------------------------------------------------------------------

void setupCommandLineParser(CommandLineParser & parser, Options const & /*options*/) {
    addVersionLine(parser, "0.1");
    
    addTitleLine(parser, "*******************************");
    addTitleLine(parser, "* find_index_approx_benchmark *");
    addTitleLine(parser, "*******************************");
    addTitleLine(parser, "");
    addTitleLine(parser, "(c) 2011 by Johannes Krugel <krugel@in.tum.de>");

    addUsageLine(parser, "find_index_approx_benchmark [OPTIONS]");    
    addLine(parser, "Benchmark for approximate pattern matching.");
    
    addSection(parser, "Main Options");
    addOption(parser, CommandLineOption('a', "alphabet",     "alphabet (bool, char, Dna, Dna5, AminoAcid, wchar)",
        OptionType::String | OptionType::Label | OptionType::Mandatory));
    addOption(parser, CommandLineOption('u', "utf-8",        "use UTF-8 input encoding",
        OptionType::Bool   | OptionType::Label));
    addOption(parser, CommandLineOption('g', "fasta",        "use FASTA file format",
        OptionType::Bool   | OptionType::Label));
    addOption(parser, CommandLineOption('t', "text_file",    "name of the text file",
        OptionType::String | OptionType::Label | OptionType::Mandatory));
    addOption(parser, CommandLineOption('i', "index",        "index spezialization "
        "(online | sa | salcp | stkurtz | wotd | sttd64 | digest | esa | fm | sadakane | lz | qgram | qsample | qgram2l)",
        OptionType::String | OptionType::Label | OptionType::Mandatory));
    addOption(parser, CommandLineOption('j', "index_param",  "index parameters",
        OptionType::String | OptionType::Label | OptionType::List));

    addOption(parser, CommandLineOption('p', "pattern_file", "base name of the pattern file",
        OptionType::String | OptionType::Label | OptionType::Mandatory));
    addOption(parser, CommandLineOption('m', "pattern_length", "pattern length",
        OptionType::Int    | OptionType::Label | OptionType::Mandatory | OptionType::List));
    addOption(parser, CommandLineOption('d', "distance_measure", "distance measure (edit | weighted | hamming)",
        OptionType::String | OptionType::Label | OptionType::Mandatory | OptionType::List));
    addOption(parser, CommandLineOption('k', "tolerance",    "search tolerance",
        OptionType::Int    | OptionType::Label | OptionType::Mandatory | OptionType::List));
    addOption(parser, CommandLineOption('e', "error_level",  "maximal error level",
        OptionType::Double | OptionType::Label));
    addOption(parser, CommandLineOption('q', "query_type",   "query type (one, all_end, all_begin_end)",
        OptionType::String | OptionType::Label | OptionType::List));

    addOption(parser, CommandLineOption('f', "finder",       "finder specialization  (DPBacktracking | IntoExactSearch | Intermediate | Hierarchical)",
        OptionType::String | OptionType::Label | OptionType::Mandatory | OptionType::List));

    addOption(parser, CommandLineOption('s', "flush_seqan_cache", "flush buffers of SeqAn external strings between benchmarks",
        OptionType::Bool   | OptionType::Label));
    addOption(parser, CommandLineOption('c', "flush_disk_cache", "flush the disk cache between benchmarks",
        OptionType::Bool   | OptionType::Label));
    addOption(parser, CommandLineOption('l', "flush_l2_cache", "flush the disk cache between benchmarks",
        OptionType::Bool   | OptionType::Label));
}

int parseCommandLineAndCheck(Options & options, CommandLineParser & parser, int argc, char const ** argv) {
    bool stop = !parse(parser, argc, argv);
    if (stop) {
        help(parser, std::cerr);
        return 1;
    }
    if (isSetLong(parser, "help")) {
        options.showHelp = true;
        return 0;
    }
    if (isSetLong(parser, "version")) {
        options.showVersion = true;
        return 0;
    }
    
    CharString alphabetString;
    getOptionValueLong(parser, "alphabet", alphabetString);
    if (alphabetString == "char") {
        options.alphabet = ALPHABET_CHAR;
    } else if (alphabetString == "bool") {
        options.alphabet = ALPHABET_BOOL;
    } else if (alphabetString == "Dna") {
        options.alphabet = ALPHABET_DNA;
    } else if (alphabetString == "Dna5") {
        options.alphabet = ALPHABET_DNA5;
    } else if (alphabetString == "AminoAcid") {
        options.alphabet = ALPHABET_AMINOACID;
    } else if (alphabetString == "wchar") {
        options.alphabet = ALPHABET_WCHAR;
    } else {
        return 1;
    }

    bool useUtf8 = false;
    getOptionValueLong(parser, "utf-8", useUtf8);
    options.encoding = useUtf8 ? ENCODING_UTF8 : ENCODING_DIRECT;
    if (options.alphabet == ALPHABET_WCHAR) {
        options.encoding = ENCODING_UTF8;
    }

    bool fasta = false;
    getOptionValueLong(parser, "fasta", fasta);
    options.fileFormat = fasta ? FILE_FORMAT_FASTA : FILE_FORMAT_RAW;

    getOptionValueLong(parser, "text_file", options.textFile);
    getOptionValueLong(parser, "index", options.index);
    
    Size<CharString>::Type cnt;
    
    cnt = length(getOptionValuesLong(parser, "index_param"));
    resize(options.indexParameters, cnt);
    for (Size<CharString>::Type i = 0; i < cnt; ++i) {
        getOptionValueLong(parser, "index_param", i, options.indexParameters[i]);
    }

    getOptionValueLong(parser, "pattern_file", options.patternFile);

    cnt = length(getOptionValuesLong(parser, "finder"));
    resize(options.finders, cnt);
    for (Size<CharString>::Type i = 0; i < cnt; ++i) {
        getOptionValueLong(parser, "finder", i, options.finders[i]);
    }

    cnt = length(getOptionValuesLong(parser, "pattern_length"));
    resize(options.patternLengths, cnt);
    for (Size<CharString>::Type i = 0; i < cnt; ++i) {
        getOptionValueLong(parser, "pattern_length", i, options.patternLengths[i]);
    }

    cnt = length(getOptionValuesLong(parser, "distance_measure"));
    resize(options.distanceMeasures, cnt);
    for (Size<CharString>::Type i = 0; i < cnt; ++i) {
        getOptionValueLong(parser, "distance_measure", i, options.distanceMeasures[i]);
    }

    cnt = length(getOptionValuesLong(parser, "tolerance"));
    resize(options.tolerances, cnt);
    for (Size<CharString>::Type i = 0; i < cnt; ++i) {
        getOptionValueLong(parser, "tolerance", i, options.tolerances[i]);
    }
    
    getOptionValueLong(parser, "error_level", options.maxErrorLevel);

    CharString queryTypeString;

    cnt = length(getOptionValuesLong(parser, "query_type"));
    resize(options.queryTypes, cnt);
    for (Size<CharString>::Type i = 0; i < cnt; ++i) {
        getOptionValueLong(parser, "query_type", i, queryTypeString);
        if (queryTypeString == "one") {
            options.queryTypes[i] = QUERY_ONE;
        } else if (queryTypeString == "all_end") {
            options.queryTypes[i] = QUERY_ALL_END;
        } else if (queryTypeString == "all_begin_end") {
            options.queryTypes[i] = QUERY_ALL_BEGIN_END;
        } else {
            return 1;
        }
    }
    if (empty(options.queryTypes)) {
        appendValue(options.queryTypes, QUERY_ONE);
    }

    getOptionValueLong(parser, "flush_seqan_cache", options.flushSeqanCache);
    getOptionValueLong(parser, "flush_disk_cache", options.flushDiskCache);
    getOptionValueLong(parser, "flush_l2_cache", options.flushL2Cache);

    return 0;
}

// ----------------------------------------------------------------------------
// Construction
// ----------------------------------------------------------------------------

template <typename TChar, typename TSpec>
void indexConstruct(String<TChar, TSpec> & haystack, Options const & /*options*/) {
    ignoreUnusedVariableWarning(haystack);
    // Do nothing for online
}

// indexParameters[0] = Alloc / External
// indexParameters[1] = suffix array construction algorithm
template <typename TText, typename TSpec>
void indexConstruct(Index<TText, IndexEsa<TSpec> > & index, Options const & options) {
    if (length(options.indexParameters) == 1u || options.indexParameters[1u] == "Default") {
        indexRequire(index, EsaSA());
    } else if (options.indexParameters[1u] == "SAQSort") {
        indexCreate(index, EsaSA(), SAQSort());
    } else if (options.indexParameters[1u] == "ManberMyers") {
        indexCreate(index, EsaSA(), ManberMyers());
    } else if (options.indexParameters[1u] == "LarssonSadakane") {
        indexCreate(index, EsaSA(), LarssonSadakane());
    } else if (options.indexParameters[1u] == "Skew3") {
        indexCreate(index, EsaSA(), Skew3());
    } else if (options.indexParameters[1u] == "Skew7") {
        indexCreate(index, EsaSA(), Skew7());
    } else if (options.indexParameters[1u] == "BwtWalkFast") {
        indexCreate(index, EsaSA(), BwtWalk<BwtWalkFast>());
    } else if (options.indexParameters[1u] == "BwtWalkInPlace") {
        indexCreate(index, EsaSA(), BwtWalk<BwtWalkInPlace>());
    } else {
        std::cerr << "Unsupported construction algorithm: \"" << options.indexParameters[1u] << "\"." << std::endl;
        indexRequire(index, EsaSA());
    }
    if (options.index == "salcp") {
        indexRequire(index, EsaLcpe());
    } else if (options.index == "esa") {
        indexRequire(index, EsaLcp());
        indexRequire(index, EsaChildtab()); // for TopDown iterators
    }
}

// indexParameters[0] = Alloc / External
// indexParameters[1] = suffix array construction algorithm
template <typename TText, typename TExternalConfig>
void indexConstruct(Index<TText, IndexEsa<External<TExternalConfig> > > & index, Options const & options) {
    typedef Index<TText, IndexEsa<External<TExternalConfig> > > TIndex;
    typedef typename Fibre<TIndex, EsaSA>::Type             TSA;
    typedef typename Fibre<TIndex, EsaLcp>::Type            TLcp;
    typedef typename Fibre<TIndex, EsaChildtab>::Type       TChildtab;
    typedef typename AllowsFastRandomAccess<TText>::Type    TRandomText;
    typedef typename AllowsFastRandomAccess<TSA>::Type      TRandomSA;
    typedef typename AllowsFastRandomAccess<TLcp>::Type     TRandomLcp;
    typedef typename AllowsFastRandomAccess<TChildtab>::Type TRandomChildtab;
    typedef typename SACreatorRandomAccess_<TSA, TText, Skew7>::Type TSACreatorRandomAccess;

    // SAQSort and ManberMyers only work with Alloc strings?
    if (length(options.indexParameters) == 1u || options.indexParameters[1u] == "Default") {
        indexRequire(index, EsaSA());
    } else if (options.indexParameters[1u] == "LarssonSadakane") {
        indexCreate(index, EsaSA(), LarssonSadakane());
    } else if (options.indexParameters[1u] == "Skew3") {
        indexCreate(index, EsaSA(), Skew3());
    } else if (options.indexParameters[1u] == "Skew7") {
        indexCreate(index, EsaSA(), Skew7());
    } else if (options.indexParameters[1u] == "BwtWalkFast") {
        indexCreate(index, EsaSA(), BwtWalk<BwtWalkFast>());
    } else if (options.indexParameters[1u] == "BwtWalkInPlace") {
        indexCreate(index, EsaSA(), BwtWalk<BwtWalkInPlace>());
    } else {
        std::cerr << "Unsupported construction algorithm: \"" << options.indexParameters[1u] << "\"." << std::endl;
        indexRequire(index, EsaSA());
    }
    if (options.index == "salcp") {
        std::cerr << "Lcpe not supported for external tables." << std::endl;
        //TODO(krugel) indexRequire(index, EsaLcpe());  // does not work with external strings
    } else if (options.index == "esa") {
        indexRequire(index, EsaLcp());
        indexRequire(index, EsaChildtab()); // for TopDown iterators
    }
}

template <typename TText, typename TSpec>
void indexConstruct(Index<TText, IndexStKurtz<TSpec> > & index, Options const & /*options*/) {
    indexRequire(index, StKurtzNodeTab());
    indexRequire(index, StKurtzLeafTab());
}

// indexParameters[0] = eager / first / lazy
template <typename TText, typename TSpec>
void indexConstruct(Index<TText, IndexWotd<TSpec> > & index, Options const & options) {
    typedef Index<TText, IndexWotd<TSpec> >                 TIndex;
    typedef typename Iterator<TIndex, TopDown<ParentLinks<> > >::Type TIter;

    if (length(options.indexParameters) == 0u || options.indexParameters[0u] == "eager") {
        // Construct first level
        indexRequire(index, WotdDir());
        
        // Explicitely construct the whole tree
        TIter itDfs(index);
        while (!atEnd(itDfs)) {
            ++itDfs;
        }
    } else if (options.indexParameters[0u] == "first") {
        // Construct first level
        indexRequire(index, WotdDir());
    } else if (options.indexParameters[0u] == "lazy") {
        // Do nothing
    } else {
        std::cerr << "Unsupported construction algorithm: \"" << options.indexParameters[0u] << "\"." << std::endl;
    }
    // Clear all temporary data structures
    clear(index.tempSA);
    shrinkToFit(index.tempSA);
    clear(index.tempOcc);
    shrinkToFit(index.tempOcc);
    clear(index.tempBound);
    shrinkToFit(index.tempBound);
}

// indexParameters[0] = inMem / inExternal
template <typename TText, typename TSpec>
void indexConstruct(Index<TText, IndexSttd64<TSpec> > & index, Options const & options) {
    typedef Index<TText, IndexSttd64<TSpec> >               TIndex;
    typedef typename TIndex::TTrees                         TTrees;
    typedef typename Iterator<TTrees, Rooted>::Type         TTreesIter;
    typedef typename TIndex::TPartTree                      TPartTreeNodes;
    typedef typename Iterator<TPartTreeNodes, Rooted>::Type TPartTreeNodesIter;

    if (length(options.indexParameters) == 0u || options.indexParameters[0u] == "inMem") {
        index.inMem = true;
    } else if (options.indexParameters[0u] == "inExternal") {
        index.inMem = false;
    } else {
        std::cerr << "Unsupported construction algorithm: \"" << options.indexParameters[0u] << "\"." << std::endl;
    }
    
    indexRequire(index, Sttd64Partitions());
    indexRequire(index, Sttd64SuffixTrees());
    _createLeafDepths(index);

    clear(index.partitions);
    clear(index.workPartitions);
}

// indexParameters[0] = Suffix array construction algorithm
template <typename TText, typename TSpec>
void indexConstruct(Index<TText, IndexDigest<TSpec> > & index, Options const & options) {
    if (length(options.indexParameters) == 0u || options.indexParameters[0u] == "Default") {
        indexRequire(index, DigestSuffixArrays());
    } else if (options.indexParameters[0u] == "LarssonSadakane") {
        indexCreate(index, DigestSuffixArrays(), LarssonSadakane());
    } else if (options.indexParameters[0u] == "Skew3") {
        indexCreate(index, DigestSuffixArrays(), Skew3());
    } else if (options.indexParameters[0u] == "Skew7") {
        indexCreate(index, DigestSuffixArrays(), Skew7());
    } else if (options.indexParameters[0u] == "BwtWalkFast") {
        indexCreate(index, DigestSuffixArrays(), BwtWalk<BwtWalkFast>());
    } else if (options.indexParameters[0u] == "BwtWalkInPlace") {
        indexCreate(index, DigestSuffixArrays(), BwtWalk<BwtWalkInPlace>());
    } else {
        std::cerr << "Unsupported construction algorithm: \"" << options.indexParameters[0u] << "\"." << std::endl;
        indexRequire(index, DigestSuffixArrays());
    }

    indexRequire(index, DigestSuffixTrees());
}

// indexParameters[0] = SBM / WT
// indexParameters[1] = compressionFactor
template <typename TText, typename TSpec>
void indexConstruct(Index<TText, FMIndex<TSpec> > & index, Options const & options) {
    if (length(options.indexParameters) >= 2u) {
        index.compressionFactor = lexicalCast<unsigned int>(options.indexParameters[1u]);
    }
    indexRequire(index, FibreSaLfTable());
}

// indexParameters[0] = block length
template <typename TText, typename TBitBin, size_t PSI_SAMPLE_RATE, size_t SA_SAMPLE_RATE, size_t ISA_SAMPLE_RATE>
void indexConstruct(Index<TText, IndexSadakane<TBitBin, PSI_SAMPLE_RATE, SA_SAMPLE_RATE, ISA_SAMPLE_RATE> > & index, Options const & options) {
    size_t blockLength = 0u;
    if (length(options.indexParameters) >= 1u) {
        blockLength = lexicalCast<unsigned int>(options.indexParameters[0u]);
    }
    indexCreate(index, FibreSA(), blockLength);
}

template <typename TText, typename TBitBin, size_t LZTRIE_NODE_END_TEXT_POSTION_SAMPLE_RATE>
void indexConstruct(Index<TText, IndexLZ<TBitBin, LZTRIE_NODE_END_TEXT_POSTION_SAMPLE_RATE> > & index, Options const & /*options*/) {
    indexRequire(index, FibreSA());
}

// indexParameters[0] = addressing
// indexParameters[1] = q
// indexParameters[2] = sample step size
template <typename TText, typename TShapeSpec, typename TSpec>
void indexConstruct(Index<TText, IndexQGram<TShapeSpec, TSpec> > & index, Options const & options) {
    // For q-sample index
    if (length(options.indexParameters) >= 3u && options.indexParameters[2u] != 1) {
        setStepSize(index, lexicalCast<unsigned int>(options.indexParameters[2u]));
    }

    CharString constructionAlgo = (length(options.indexParameters) >= 4u) ? options.indexParameters[3u] : "default";
    if (constructionAlgo == "default") {
        indexRequire(index, QGramSADir());
    } else if (constructionAlgo == "external") {
        resize(indexSA(index), _qgramQGramCount(index), Exact());
        resize(indexDir(index), _fullDirLength(index), Exact());
        createQGramIndexExt(index);
        resize(indexSA(index), back(indexDir(index)), Exact());     // shrink if some buckets were disabled

    // createQGramIndexExtSA works only for string sets
    //} else if (constructionAlgo == "external2") {
    //    resize(indexSA(index), _qgramQGramCount(index), Exact());
    //    resize(indexDir(index), _fullDirLength(index), Exact());
    //    createQGramIndexExtSA(index);
    //    resize(indexSA(index), back(indexDir(index)), Exact());     // shrink if some buckets were disabled
    } else {
        std::cerr << "Unsupported construction algorithm: \"" << options.indexParameters[3u] << "\"." << std::endl;
    }

    //std::cerr << "dir: " << length(indexDir(index)) << " (" << capacity(indexDir(index)) << ")" << std::endl;
    //std::cerr << "sa:  " << length(indexSA(index)) << " (" << capacity(indexSA(index)) << ")"  << std::endl;

    // Not needed here (or created automatically)
    //indexRequire(index, QGramSA());
    //indexRequire(index, QGramDir());
    //indexRequire(index, QGramCounts());
    //indexRequire(index, QGramCountsDir());
    //indexRequire(index, QGramBucketMap());
}

template <typename TText, typename TShapeSpec, typename TAddressingSpec, typename TSuffixTreeSpec>
void indexConstruct(Index<TText, IndexQGram2L<TShapeSpec, TAddressingSpec, TSuffixTreeSpec> > & index, Options const & options) {
    index.clearRedundantFibres = true;
    if (length(options.indexParameters) >= 4u) {
        setSubsequenceLength(index, lexicalCast<unsigned int>(options.indexParameters[2u]));
    }
    indexRequire(index, QGram2LBackEndIndex());
    indexRequire(index, QGram2LFrontEndIndex());

    // Some shortcuts
    unsigned int Q = lexicalCast<unsigned int>(options.indexParameters[1u]);
    unsigned int frontEndIndexDirSize   = length(indexDir(index.frontEndIndex));
    unsigned int frontEndIndexSASize    = length(indexSA(index.frontEndIndex));
    unsigned int backEndIndexDirSize    = length(indexDir(index.backEndIndex));
    unsigned int backEndIndexSASize     = length(indexSA(index.backEndIndex));

    unsigned int frontEndIndexSize      = frontEndIndexDirSize + frontEndIndexSASize;
    unsigned int backEndIndexSize       = backEndIndexDirSize  + backEndIndexSASize;
    unsigned int indexSize              = frontEndIndexSize    + backEndIndexSize;
    unsigned int indexQGramDirSize      = frontEndIndexDirSize;
    unsigned int indexQGramSASize       = length(indexText(index)) - Q + 1u;
    unsigned int indexQGramSize         = indexQGramDirSize + indexQGramSASize;
    
    unsigned int numberOfSequences      = backEndIndexDirSize - 1u;
    // Sequcences have already been cleared in indexCreate! The expression below is therefore always 0.
    //unsigned int concatenatedLength     = length(concat(index.sequences));
    
    std::cerr << "Text length         = " << length(indexText(index)) << std::endl;
    std::cerr << "Number of sequences = " << numberOfSequences << std::endl;
    //std::cerr << "Concatenation       = " << concatenatedLength << std::endl;
    std::cerr << "Front end index     = " << frontEndIndexSize << " = " << frontEndIndexDirSize << " + " << frontEndIndexSASize << std::endl;
    std::cerr << "Back end index      = " << backEndIndexSize  << " = " << backEndIndexDirSize  << " + " << backEndIndexSASize  << std::endl;
    std::cerr << "IndexQGram2L        = " << indexSize << std::endl;
    std::cerr << "IndexQGram          = " << indexQGramSize << " = " << indexQGramDirSize << " + " << indexQGramSASize << std::endl;

    //std::cerr << "frontdir: " << length(indexDir(index.frontEndIndex)) << " (" << capacity(indexDir(index.frontEndIndex)) << ")" << std::endl;
    //std::cerr << "frontsa:  " << length(indexSA(index.frontEndIndex)) << " (" << capacity(indexSA(index.frontEndIndex)) << ")" << std::endl;
    //std::cerr << "backdir:  " << length(indexDir(index.backEndIndex)) << " (" << capacity(indexDir(index.backEndIndex)) << ")" << std::endl;
    //std::cerr << "backsa:   " << length(indexSA(index.backEndIndex)) << " (" << capacity(indexSA(index.backEndIndex)) << ")" << std::endl;

    //clear(indexDir(index.backEndIndex));
    //clear(indexSA(index.backEndIndex));
    //clear(indexBucketMap(index.backEndIndex).qgramCode);
    //clear(indexDir(index.frontEndIndex));
    //clear(indexSA(index.frontEndIndex));
    //clear(indexBucketMap(index.frontEndIndex).qgramCode);
    //clear(index.backEndIndex);
    //clear(index.frontEndIndex);
    //clear(index.sequences);

    shrinkToFit(indexDir(index.backEndIndex));
    shrinkToFit(indexSA(index.backEndIndex));
    shrinkToFit(indexDir(index.frontEndIndex));
    shrinkToFit(indexSA(index.frontEndIndex));
}


// ----------------------------------------------------------------------------
// getNeedles
// ----------------------------------------------------------------------------

template <typename TSize, typename TNeedles>
bool getNeedles(Options const & options, CharString const & patternFile, TSize patternLength, TNeedles & needles) {
    typedef String<typename Value<typename Value<TNeedles>::Type>::Type> TNeedle;  // = String<char>, String<Dna5>, etc.
    
    TNeedle needlesString;
    CharString encodingString = options.encoding == ENCODING_DIRECT ? "direct" : "utf-8";
    bool success = readTextFile(toCString(patternFile), needlesString, CharString("raw"), encodingString);
    if (!success) return false;
    
    for (TSize pos = 0u; pos < length(needlesString); pos += patternLength + 2) {
        TNeedle ndl = infixWithLength(needlesString, pos, patternLength);
        //std::cerr << length(ndl);
        appendValue(needles, ndl);
    }
    return true;
}

// ----------------------------------------------------------------------------
// flushBuffers
// ----------------------------------------------------------------------------

template <typename TString>
void flushAndFree(TString const & /*str*/) {
    // Do nothing for Alloc strings.
}

template <typename THaystack>
void flushBuffers(THaystack & haystack, Options const & /*options*/) {
    ignoreUnusedVariableWarning(haystack);
    // Do nothing by default
    std::cerr << "Flushing nothing." << std::endl;
}

template <typename TText, typename TSpec>
void flushBuffers(Index<TText, IndexEsa<TSpec> > & index, Options const & /*options*/) {
    std::cerr << "Flushing IndexEsa." << std::endl;
    flushAndFree(indexSA(index));
    flushAndFree(indexLcp(index));
    //flushAndFree(indexLcpe(index));
    flushAndFree(indexChildtab(index));
}

template <typename TText, typename TSpec>
void flushBuffers(Index<TText, IndexWotd<TSpec> > & index, Options const & /*options*/) {
    std::cerr << "Flushing IndexWotd." << std::endl;
    flushAndFree(indexSA(index));
    flushAndFree(indexDir(index));
}

template <typename TText, typename TSpec>
void flushBuffers(Index<TText, IndexSttd64<TSpec> > & index, Options const & /*options*/) {
    typedef Index<TText, IndexSttd64<TSpec> >               TIndex;
    typedef typename Iterator<typename TIndex::TTrees>::Type TTreesIter;

    std::cerr << "Flushing IndexSttd64." << std::endl;
    for (TTreesIter treesIt = begin(index.trees); treesIt != end(index.trees); ++treesIt) {
        flushAndFree(*treesIt);
    }
}

template <typename TText, typename TSpec>
void flushBuffers(Index<TText, IndexDigest<TSpec> > & index, Options const & /*options*/) {
    std::cerr << "Flushing IndexDigest." << std::endl;
    flushAndFree(getFibre(index, DigestSuffixTrees()));
}

template <typename TText, typename TShapeSpec, typename TSpec>
void flushBuffers(Index<TText, IndexQGram<TShapeSpec, TSpec> > & index, Options const & /*options*/) {
    std::cerr << "Flushing IndexQGram." << std::endl;
    flushAndFree(indexSA(index));
    flushAndFree(indexDir(index));
}

// ----------------------------------------------------------------------------
// Benchmark functions
// ----------------------------------------------------------------------------

// TODO(krugel) Optimal verification also for Hierarchical

// Automatically determine the best verification algorithm based on alphabet, pattern length and tolerance
// Therefore we transform "Partitioning<..., Optimum>" into e.g. "Partitioning<..., Myers>"
template <typename THaystack, typename TFinderSpec, typename TScoreValue, typename TNeedles, typename TSize, typename TPartitioningSpec, typename TScore, typename TPiecePatternSpec>
bool search(THaystack & haystack, CharString const & distanceMeasure,
        TScoreValue tolerance, typename Size<THaystack>::Type numberOfPieces, TNeedles & needles, QueryType queryType,
        TSize & matches, TSize & matchesBegin, TFinderSpec const & finderSpec, Partitioning<TPartitioningSpec, TScore, TPiecePatternSpec, Optimum> const &) {
    // bool: Myers for k/m >= 1/4 and for m=16
    // DNA: Myers for k/m >= 1/4
    // char, AminoAcid: Myers only for m=4, otherwise PEX

    typedef typename Value<THaystack>::Type                 TChar;
    enum VerificationAlgorithm {MYERS, PEX};
    VerificationAlgorithm algo = PEX;

    TSize alphabetSize = valueSize<TChar>();
    TSize patternLength = length(needles[0]);
    double errorLevel = (double) tolerance / (double) patternLength;
    if (alphabetSize == 2) { // bool
        if (patternLength <= 16 || errorLevel >= 0.25) algo = MYERS;
    } else if(alphabetSize <= 5) { // Dna
        if (errorLevel >= 0.25) algo = MYERS;
    } else if(alphabetSize <= 256) { // AminoAcid, char
        if (patternLength <= 4) algo = MYERS;
    } else if (alphabetSize == 0) { // wchar_t
        // always PEX
    } else {
        std::cerr << "Unkown alphabet size while determining optimal verification algorithm." << std::endl;
    }
    
    if (algo == MYERS) {
        std::cerr << "Verification: Myers" << std::endl;
        return search(haystack, distanceMeasure, tolerance, numberOfPieces, needles, queryType, matches, matchesBegin, finderSpec, Partitioning<TPartitioningSpec, TScore, TPiecePatternSpec, Myers<FindInfix> >());
    } else {
        std::cerr << "Verification: Pex" << std::endl;
        return search(haystack, distanceMeasure, tolerance, numberOfPieces, needles, queryType, matches, matchesBegin, finderSpec, Partitioning<TPartitioningSpec, TScore, TPiecePatternSpec, Pex<Hierarchical, WuManber> >());
    }
}

template <typename THaystack, typename TFinderSpec, typename TPatternSpecTmp, typename TScoreValue, typename TNeedles, typename TSize>
bool search(THaystack & haystack, CharString const & distanceMeasure,
        TScoreValue tolerance, typename Size<THaystack>::Type numberOfPieces, TNeedles & needles, QueryType queryType,
        TSize & matches, TSize & matchesBegin, TFinderSpec const &, TPatternSpecTmp const &) {
    typedef typename Value<TNeedles>::Type                  TNeedle;
    typedef Finder<THaystack, TFinderSpec>                  TFinder;
    typedef typename IndexPattern<TNeedle, THaystack, TPatternSpecTmp>::Type TPatternSpec;
    typedef Pattern<TNeedle, TPatternSpec>                  TPattern;
    typedef typename Iterator<TNeedles>::Type               TNeedlesIter;

    bool found;

    ignoreUnusedVariableWarning(distanceMeasure);
    ignoreUnusedVariableWarning(found);
    
    TFinder finder(haystack);
    TPattern pattern2;
    setScoreLimit(pattern2, - tolerance);
    if (numberOfPieces != 0u) setNumberOfPieces(pattern2, numberOfPieces);

    if (length(needles[0u]) < minSupportedPatternLength(finder, pattern2)) {
        std::cerr << "Pattern is too short (" << length(needles[0u]) << " < " << minSupportedPatternLength(finder, pattern2) << "). Skipping." << std::endl;
        return false;
    }
    preparePatterns(finder, pattern2, needles, tolerance);

    for (TNeedlesIter iter = begin(needles); !atEnd(iter, needles); ++iter) {
        goBegin(finder);
        clear(finder);
        
        if (empty(*iter)) continue;
        TPattern pattern(*iter, - tolerance);
        if (numberOfPieces != 0u) setNumberOfPieces(pattern, numberOfPieces);
        found = false;
        while (find(finder, pattern)) {
            found = true;
            matches++;
            while (queryType == QUERY_ALL_BEGIN_END && findBegin(finder, pattern)) {
                matchesBegin++;
                //std::cout << "found " << *iter << " at position " << position(finder) << ", \"" << infix(finder) << "\" " << tolerance << std::endl;
            }
            if (queryType == QUERY_ONE) break;
        }
    }
    
    printDebugInfo(finder);
    std::cout << std::endl;
    return true;
}

// Online
template <typename THaystack, typename TScoreValue, typename TNeedles, typename TSize>
bool search(Options const & /*options*/, THaystack & h, CharString const & dM, TScoreValue t, typename Size<THaystack>::Type nOP, TNeedles & n, QueryType qT, TSize & m, TSize & mB, CharString const & finder, False const & /*isIndex*/, False const & /*suffixTreeIteration*/) {
    if (finder == "Myers") {
        return search(h, dM, t, nOP, n, qT, m, mB, Default(), Myers<FindInfix>());
    } else if (finder == "DPSearch") {
        return search(h, dM, t, nOP, n, qT, m, mB, Default(), DPSearch<EditDistanceScore>());
    } else if (finder == "PexNonHierarchicalAhoCorasick") {
        return search(h, dM, t, nOP, n, qT, m, mB, Default(), Pex<NonHierarchical, AhoCorasick>());
    } else if (finder == "PexHierarchicalAhoCorasick") {
        return search(h, dM, t, nOP, n, qT, m, mB, Default(), Pex<Hierarchical, AhoCorasick>());
    } else if (finder == "PexNonHierarchicalWuManber") {
        return search(h, dM, t, nOP, n, qT, m, mB, Default(), Pex<NonHierarchical, WuManber>());
    } else if (finder == "PexHierarchicalWuManber") {
        return search(h, dM, t, nOP, n, qT, m, mB, Default(), Pex<Hierarchical, WuManber>());
    } else if (finder == "AbndmAlgo") {
        return search(h, dM, t, nOP, n, qT, m, mB, Default(), AbndmAlgo());
    } else {
        std::cerr << "Unsupported finder type: \"" << finder << "\". Skipping." << std::endl;
        return false;
    }
}

// IndexEsa
template <typename TString, typename TIndexSpec, typename TScoreValue, typename TNeedles, typename TSize>
bool search(Options const & options, Index<TString, IndexEsa<TIndexSpec> > & h, CharString const & dM, TScoreValue t, typename Size<TString>::Type nOP, TNeedles & n, QueryType qT, TSize & m, TSize & mB, CharString const & finder, True const & /*isIndex*/, True const & /*suffixTreeIteration*/) {
    if (options.index == "sa" && finder == "IntoExactSearch") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<EsaFindMlr>(), Partitioning<IntoExactSearch, EditDistanceScore, Default, Optimum>());
    } else if (options.index == "sa" && finder == "IntoExactSearchPrepare") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<EsaFindMlr, Default, True>(), Partitioning<IntoExactSearch, EditDistanceScore, Default, Optimum>());
    } else if (options.index == "sa" && finder == "Hierarchical") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<FinderPartitioning<EsaFindMlr> >(), Partitioning<PartitioningHierarchical, EditDistanceScore, Partitioning<IntoExactSearch>, Optimum>());
    } else if (options.index == "salcp" && finder == "IntoExactSearch") {
        // TODO(krugel) Does not work for external tables
        //return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<EsaFindLcpe>(), Partitioning<IntoExactSearch, EditDistanceScore, Default, Optimum>());
        return false;
    } else if (options.index == "salcp" && finder == "Hierarchical") {
        // TODO(krugel) Does not work for external tables
        //return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<FinderPartitioning<EsaFindLcpe> >(), Partitioning<PartitioningHierarchical, EditDistanceScore, Partitioning<IntoExactSearch>, Optimum>());
        return false;
    } else if (options.index == "salcp" && finder == "IntoExactSearchPrepare") {
        //return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<EsaFindLcpe, Default, True>(), Partitioning<IntoExactSearch, EditDistanceScore, Default, Optimum>());
        return false;
    } else if (options.index == "esa" && finder == "DPBacktracking") {
        return search(h, dM, t, nOP, n, qT, m, mB, DPBacktracking<>(), DPBacktracking<>());
    } else if (options.index == "esa" && finder == "IntoExactSearch") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<FinderSTree>(), Partitioning<IntoExactSearch, EditDistanceScore, Default, Optimum>());
    } else if (options.index == "esa" && finder == "IntoExactSearchPrepare") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<FinderSTree, Default, True>(), Partitioning<IntoExactSearch, EditDistanceScore, Default, Optimum>());
    } else if (options.index == "esa" && finder == "Intermediate") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<DPBacktracking<> >(), Partitioning<Intermediate, EditDistanceScore, DPBacktracking<>, Optimum>());
    } else if (options.index == "esa" && finder == "Hierarchical") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<FinderPartitioning<FinderSTree> >(), Partitioning<PartitioningHierarchical, EditDistanceScore, Partitioning<IntoExactSearch>, Optimum>());
    } else {
        std::cerr << "Unsupported finder type: \"" << finder << "\" for index \"" << options.index << "\". Skipping." << std::endl;
        return false;
    }
}

// Other suffix tree indexes: IndexWotd
template <typename TString, typename TIndexSpec, typename TScoreValue, typename TNeedles, typename TSize>
bool search(Options const & /*options*/, Index<TString, TIndexSpec> & h, CharString const & dM, TScoreValue t, typename Size<TString>::Type nOP, TNeedles & n, QueryType qT, TSize & m, TSize & mB, CharString const & finder, True const & /*isIndex*/, True const & /*suffixTreeIteration*/) {
    if (finder == "DPBacktracking") {
        return search(h, dM, t, nOP, n, qT, m, mB, DPBacktracking<>(), DPBacktracking<>());
    } else if (finder == "IntoExactSearch") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<>(), Partitioning<IntoExactSearch, EditDistanceScore, Default, Optimum>());
    } else if (finder == "IntoExactSearchPrepare") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<Default, Default, True>(), Partitioning<IntoExactSearch, EditDistanceScore, Default, Optimum>());
    } else if (finder == "Intermediate") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<DPBacktracking<> >(), Partitioning<Intermediate, EditDistanceScore, DPBacktracking<>, Optimum>());
    } else if (finder == "Hierarchical") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<FinderPartitioning<> >(), Partitioning<PartitioningHierarchical, EditDistanceScore, Partitioning<IntoExactSearch>, Optimum>());
    } else {
        std::cerr << "Unsupported finder type: \"" << finder << "\". Skipping." << std::endl;
        return false;
    }
}

// IndexQGram
template <typename TString, typename TShape, typename TIndexSpec, typename TScoreValue, typename TNeedles, typename TSize>
bool search(Options const & /*options*/, Index<TString, IndexQGram<TShape, TIndexSpec> > & h, CharString const & dM, TScoreValue t, typename Size<TString>::Type nOP, TNeedles & n, QueryType qT, TSize & m, TSize & mB, CharString const & finder, True const & /*isIndex*/, False const & /*suffixTreeIteration*/) {
    if (finder == "IntoExactSearch") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<FinderQGramExtended<> >(), Partitioning<IntoExactSearch, EditDistanceScore, Default, Optimum>());
    } else if (finder == "IntoExactSearchPrepare") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<FinderQGramExtended<>, Default, True>(), Partitioning<IntoExactSearch, EditDistanceScore, Default, Optimum>());
    } else if (finder == "Hierarchical") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<FinderPartitioning<FinderQGramExtended<> > >(), Partitioning<PartitioningHierarchical, EditDistanceScore, Partitioning<IntoExactSearch>, Optimum>());
    } else {
        std::cerr << "Unsupported finder type: \"" << finder << "\". Skipping." << std::endl;
        return false;
    }
}

// IndexDigest, IndexSttd64
template <typename TString, typename TIndexSpec, typename TScoreValue, typename TNeedles, typename TSize>
bool search(Options const & /*options*/, Index<TString, TIndexSpec > & h, CharString const & dM, TScoreValue t, typename Size<TString>::Type nOP, TNeedles & n, QueryType qT, TSize & m, TSize & mB, CharString const & finder, True const & /*isIndex*/, False const & /*suffixTreeIteration*/) {
    // TIndexPatternSpec is only necessary for IndexDigest, all other indexes work with "Default"
    typedef Index<TString, TIndexSpec>                      TIndex;
    typedef typename DefaultIndexPattern<TString, TIndex>::Type TIndexPatternSpec;
    
    if (finder == "IntoExactSearch") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<>(), Partitioning<IntoExactSearch, EditDistanceScore, Default, Optimum>());
    } else if (finder == "IntoExactSearchPrepare") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<Default, Default, True>(), Partitioning<IntoExactSearch, EditDistanceScore, Default, Optimum>());
    } else if (finder == "Hierarchical") {
        return search(h, dM, t, nOP, n, qT, m, mB, FinderPartitioning<FinderPartitioning<> >(), Partitioning<PartitioningHierarchical, EditDistanceScore, Partitioning<IntoExactSearch, EditDistanceScore, TIndexPatternSpec>, Optimum>());
    } else {
        std::cerr << "Unsupported finder type: \"" << finder << "\". Skipping." << std::endl;
        return false;
    }
}

template <typename TString, typename TIndexSpec>
bool runText(std::ostream & csvStreamConstruct, std::ostream & csvStreamSearch, Options const & options, TString & txt, TIndexSpec const &) {
    typedef typename Value<TString>::Type                   TChar;
    typedef Iterator<const String<CharString> >::Type       TIterator;
    typedef typename SupportsSuffixTreeIteration<TIndexSpec>::Type TSupportsSuffixTreeIteration;
    typedef typename HaystackFromIndexSpec<TString, TIndexSpec>::Type THaystack;
    typedef typename IsIndex<THaystack>::Type               TIsIndex;
    typedef Iterator<const String<CharString> >::Type       TDistanceMeasuresIter;
    typedef Iterator<const String<unsigned int> >::Type     TPatternLengthsIter;
    typedef Iterator<const String<int> >::Type              TTolerancesIter;
    typedef Iterator<const String<QueryType> >::Type        TQueryTypeIter;
    typedef String<TChar>                                   TNeedle;
    typedef StringSet<TNeedle, Owner<> >                    TNeedles;
    typedef typename Size<THaystack>::Type                  TSize;

    bool allSuccessfull = true;
    CharString constructTime = currentTime();
    
    csvStreamConstruct << constructTime << ";" << options.version << ";" << BUILD_TYPE << ";" << ALPHABET_NAMES[options.alphabet] << ";"
        << options.textFile << ";" << length(txt) << ";" << options.index << ";" << stringConcat(options.indexParameters)
        << ";" << STATIC_INDEX_PARAMETERS << ";" << options.patternFile << ";" << std::flush;

    std::cout << "CONSTRUCTION" << std::endl;

    unsigned int heapSpaceInit = SEQAN_PROVAL(SEQAN_PROMEMORY);
    Resources resConstructStart;
    double timeConstructStart = SEQAN_PROGETTIME;
    ignoreUnusedVariableWarning(timeConstructStart);

    bool successConstruction;
    CharString exceptionWhat = "";
    THaystack haystack(txt);
    try {
        indexConstruct(haystack, options); // Do the actual construction
        successConstruction = true;
    } catch (std::exception& e) {
        std::cerr << "Exception: " << e.what() << std::endl;
        exceptionWhat = e.what();
        successConstruction = false;
    }
    
    double timeConstruct = SEQAN_PROTIMEDIFF(timeConstructStart);
    unsigned int heapSpaceConstruct = SEQAN_PROVAL(SEQAN_PROMEMORY);

    Resources resConstructStop;
    Resources resConstruct = resConstructStop - resConstructStart;

    csvStreamConstruct << timeConstruct << ";" << resConstruct.userTime << ";" << resConstruct.sysTime << ";"
        << heapSpaceInit << ";" << heapSpaceConstruct << ";"
        << resConstructStart.virtMem << ";" << resConstruct.virtMem << ";" << resConstruct.virtMemPeak << ";"
        << resConstruct.physMem << ";" << resConstruct.physMemPeak << ";;;;;" << exceptionWhat << std::flush;
    
    if (! successConstruction) return false;

    for (TPatternLengthsIter lengthIt = begin(options.patternLengths); lengthIt != end(options.patternLengths); ++lengthIt) {
        for (TDistanceMeasuresIter distIt = begin(options.distanceMeasures); distIt != end(options.distanceMeasures); ++distIt) {
            for (TTolerancesIter toleranceIt = begin(options.tolerances); toleranceIt != end(options.tolerances); ++toleranceIt) {
                // Only use error levels below given threshold
                if (double(*toleranceIt) / double(*lengthIt) > options.maxErrorLevel) break;
                CharString patternFile = options.patternFile;
                append(patternFile, "-");
                append(patternFile, myToString(*lengthIt));
                append(patternFile, "_");
                append(patternFile, *distIt);
                append(patternFile, "_");
                append(patternFile, myToString(*toleranceIt));
                append(patternFile, ".txt");
                
                TNeedles needles;
                bool successNeedles = getNeedles(options, patternFile, *lengthIt, needles);
                if (! successNeedles) {
                    std::cerr << "Could not read patterns \"" << patternFile << "\"." << std::endl;
                    allSuccessfull = false;
                    continue;
                }
                    
                for (TQueryTypeIter queryTypeIt = begin(options.queryTypes); queryTypeIt != end(options.queryTypes); ++queryTypeIt) {
                    CharString queryType = *queryTypeIt == QUERY_ONE ? "one" : (*queryTypeIt == QUERY_ALL_END ? "all_end" : "all_begin_end");
                    for (TIterator finderIt = begin(options.finders); finderIt != end(options.finders); ++finderIt) {
                        unsigned int minNumberOfPieces;
                        unsigned int maxNumberOfPieces;
                        if (*finderIt == "IntoExactSearch") {
                            minNumberOfPieces = *toleranceIt + 1u;
                            maxNumberOfPieces = *toleranceIt + 1u;

                            if (options.index == "qgram" || options.index == "qsample" || options.index == "qgram2l") {
                                unsigned int stepSize = (options.index == "qsample") ? lexicalCast<unsigned int>(options.indexParameters[2u]) : 1u;
                                unsigned int q = lexicalCast<unsigned int>(options.indexParameters[1u]);
                                if (*lengthIt < (q + stepSize - 1u) * minNumberOfPieces) {
                                    std::cerr << "Pattern is too short (" << *lengthIt << " < " << (q + stepSize - 1u) * minNumberOfPieces << "). Directly skipping." << std::endl;
                                    continue;
                                }
                            }
                        } else if (*finderIt == "Intermediate") {
                            // TODO Automatically set the optimal number of pieces
                            minNumberOfPieces = 2u;
                            if (*toleranceIt == 4) minNumberOfPieces = 3;
                            if (*toleranceIt == 8) minNumberOfPieces = 5;
                            if (*toleranceIt == 16) minNumberOfPieces = 6; // 9
                            if (*toleranceIt == 32) minNumberOfPieces = 17;
                            if (*toleranceIt == 64) minNumberOfPieces = 22;
                            if (*toleranceIt == 128) minNumberOfPieces = 33;
                            if (*toleranceIt == 256) minNumberOfPieces = 129;

                            //if (*toleranceIt == 8) minNumberOfPieces = 3;
                            //if (*toleranceIt == 16) minNumberOfPieces = 6;
                            //if (*toleranceIt == 32) minNumberOfPieces = 11;
                            //if (*toleranceIt == 64) minNumberOfPieces = 22;
                            //if (*toleranceIt == 128) minNumberOfPieces = 43;
                            //if (*toleranceIt == 256) minNumberOfPieces = 86;
                            maxNumberOfPieces = *toleranceIt;
                        } else if (*finderIt == "Hierarchical") {
                            // TODO Automatically set the optimal number of pieces
                            //minNumberOfPieces = 2u;
                            //if (*toleranceIt == 8) minNumberOfPieces = 2;
                            //if (*toleranceIt == 16) minNumberOfPieces = 3;
                            //if (*toleranceIt == 32) minNumberOfPieces = 5;
                            //if (*toleranceIt == 64) minNumberOfPieces = 9;
                            //if (*toleranceIt == 128) minNumberOfPieces = 17;
                            //maxNumberOfPieces = *toleranceIt;
                            
                            // 5
                            minNumberOfPieces = 9u;
                            //if (*toleranceIt == 4) minNumberOfPieces = 3;    // 2
                            //if (*toleranceIt == 8) minNumberOfPieces = 3;    // 2
                            //if (*toleranceIt == 16) minNumberOfPieces = 9;   // 2
                            //if (*toleranceIt == 16) minNumberOfPieces = 5;   // 2
                            //if (*toleranceIt == 16) minNumberOfPieces = 3;   // 2
                            //if (*toleranceIt == 32) minNumberOfPieces = 4;                                          //11   2
                            //if (*toleranceIt == 64) minNumberOfPieces = 10;  // 4   33
                            //if (*toleranceIt == 128) minNumberOfPieces = 22; // 4   26, 33, 43, 65
                            //if (*toleranceIt == 256) minNumberOfPieces = 43;

                            maxNumberOfPieces = minNumberOfPieces;
                            //maxNumberOfPieces = *toleranceIt;  // for DNA
                            //maxNumberOfPieces = *toleranceIt;
                            //minNumberOfPieces = *toleranceIt + 1;
                            //maxNumberOfPieces = *toleranceIt + 1;

                            if (*toleranceIt == 8) maxNumberOfPieces = 5;   // 2
                            if (*toleranceIt == 16) maxNumberOfPieces = 5;   // 2
                            if (*toleranceIt == 32) maxNumberOfPieces = 10;                                          //11   2
                            if (*toleranceIt == 64) maxNumberOfPieces = 10;  // 4   33
                            if (*toleranceIt == 128) maxNumberOfPieces = 65; // 4   26, 33, 43, 65
                            if (*toleranceIt == 256) maxNumberOfPieces = 42;
                        } else {
                            minNumberOfPieces = 0u;
                            maxNumberOfPieces = 0u;
                        }
                        
                        //unsigned int maxNumberOfPieces = (*finderIt == "Intermediate" || *finderIt == "Hierarchical") ? (unsigned int) *toleranceIt + 1u : 0u;
                        for (unsigned int numberOfPieces = minNumberOfPieces; numberOfPieces <= maxNumberOfPieces; ++numberOfPieces) {
                            if (numberOfPieces > 1u && *toleranceIt / numberOfPieces == *toleranceIt / (numberOfPieces - 1)) continue;
                            
                            std::cout << "text           = " << options.textFile << std::endl;
                            std::cout << "build_type     = " << BUILD_TYPE << std::endl;
                            std::cout << "alphabet       = " << ALPHABET_NAMES[options.alphabet] << std::endl;
                            std::cout << "index          = " << options.index << std::endl;
                            std::cout << "index_params_d = " << stringConcat(options.indexParameters) << std::endl;
                            std::cout << "index_params_s = " << STATIC_INDEX_PARAMETERS << std::endl;
                            std::cout << "finder         = " << *finderIt << std::endl;
                            std::cout << "pattern_length = " << *lengthIt << std::endl;
                            std::cout << "dist_measure   = " << *distIt << std::endl;
                            std::cout << "tolerance      = " << *toleranceIt << std::endl;
                            std::cout << "query_type     = " << queryType << std::endl;
                            std::cout << "pieces         = " << numberOfPieces << std::endl;
                            std::cout << "pattern_file   = " << patternFile << std::endl;
                            
                            // Clear SeqAn buffers
                            if (options.flushSeqanCache) {
                                flushBuffers(haystack, options);
                            }
                            // Clear disk buffers
                            if (options.flushDiskCache) {
                                int ret = system("sudo -S /usr/local/bin/drop_cache < ~/.krugel"); // TODO(krugel) Remove ~/.krugel
                                if (ret != 0) std::cerr << "Dropping the cache failed: " << ret << std::endl;
                            }
                            // Clear L2 cache
                            if (options.flushL2Cache) {
                                char dummy[L2_CACHE_SIZE];
                                memset(dummy, 0, sizeof dummy);
                            }
                            
                            TSize matches = 0u;
                            TSize matchesBegin = 0u;

                            csvStreamSearch << constructTime << ";" << currentTime() << ";" << options.version << ";" << BUILD_TYPE << ";"
                                << ALPHABET_NAMES[options.alphabet] << ";" << options.textFile << ";"
                                << length(txt) << ";" << options.index << ";" << stringConcat(options.indexParameters) << ";" 
                                << STATIC_INDEX_PARAMETERS << ";" << *finderIt << ";"
                                << *lengthIt << ";" << *distIt << ";" << *toleranceIt << ";" << queryType << ";"
                                << numberOfPieces << ";" << patternFile << ";" << std::flush;
                            Resources resStart;
                            double timeSearchStart = SEQAN_PROGETTIME;
                            ignoreUnusedVariableWarning(timeSearchStart);

                            bool success = search(options, haystack, *distIt, *toleranceIt, numberOfPieces, needles, *queryTypeIt, matches, matchesBegin, *finderIt, TIsIndex(), TSupportsSuffixTreeIteration());

                            double timeSearch = SEQAN_PROTIMEDIFF(timeSearchStart);
                            unsigned int heapSpaceSearch = SEQAN_PROVAL(SEQAN_PROMEMORY);

                            Resources resStop;
                            Resources resSearch = resStop - resStart;
                            allSuccessfull = allSuccessfull && success;
                            csvStreamSearch << success << ";" << matches << ";" << matchesBegin << ";" << timeSearch << ";"
                                << resSearch.userTime << ";" << resSearch.sysTime << ";" << heapSpaceSearch << ";" << resSearch.virtMem << ";"
                                << resSearch.virtMemPeak << ";" << resSearch.physMem << ";" << resSearch.physMemPeak << ";" << std::endl;

                            std::cout << "success        = " << (success ? "yes" : "no") << std::endl;
                            std::cout << "matches        = " << matches << std::endl;
                            std::cout << "matchesBegin   = " << matchesBegin << std::endl;
                            std::cout << "SEARCHING = " << resSearch << std::endl;
                            std::cout << "====================================================================================================" << std::endl;
                            std::cout << std::endl;
                        }
                    }
                }
            }
        }
    }
    return allSuccessfull;
}

template <typename TChar>
bool runAlphabet(std::ostream & csvStreamConstruct, std::ostream & csvStreamSearch, Options & options, TChar const &) {

    // Read text file from disk into main memory
    typedef String<TChar> TText;

    TText txt;
    CharString fileFormatString = options.fileFormat == FILE_FORMAT_RAW ? "raw" : "fasta";
    CharString encodingString = options.encoding == ENCODING_DIRECT ? "direct" : "utf-8";
    bool success = readTextFile(toCString(options.textFile), txt, fileFormatString, encodingString);
    if (! success) {
        std::cerr << "Could not read text \"" << options.textFile << "\"." << std::endl;
        return false;
    }

    if (false) {
        ; // syntactic dummy
#ifdef SEQAN_BENCHMARK_Online
    } else if (options.index == "online") {
        return runText(csvStreamConstruct, csvStreamSearch, options, txt, OnlinePseudoIndex());
#endif
#ifdef SEQAN_BENCHMARK_IndexEsa
    } else if (options.index == "sa" || options.index == "salcp" || options.index == "esa") {
        // indexParameters[0] = Alloc / External
        // indexParameters[1] = suffix array construction algorithm, lcp table construction algorithm
        if (length(options.indexParameters) == 0 || options.indexParameters[0u] == "Alloc") {
            return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexEsa<>());
        } else if(options.indexParameters[0u] == "External") {
            //typedef ExternalConfig<> TExternalConfig;
            //typedef ExternalConfig<ExternalConfig<>::TFile, ExternalConfig<>::PAGESIZE, SEQAN_BENCHMARK_SUFFBUFF(SEQAN_BENCHMARK_PARAM)> TExternalConfig;
            static const size_t VALUE_SIZE = BytesPerValue<typename Size<TText>::Type>::VALUE;
            typedef ExternalConfig<ExternalConfig<>::TFile, SEQAN_BENCHMARK_PAGESIZE(SEQAN_BENCHMARK_PARAM) / VALUE_SIZE, SEQAN_BENCHMARK_SUFFBUFF(SEQAN_BENCHMARK_PARAM)> TExternalConfig;
            return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexEsa<External<TExternalConfig> >());
        } else {
            std::cerr << "Unsupported storage: \"" << options.indexParameters[0u] << "\". Skipping." << std::endl;
        }
#endif
#ifdef SEQAN_BENCHMARK_IndexStKurtz
    } else if (options.index == "stkurtz") {
        return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexStKurtz<>());
#endif
#ifdef SEQAN_BENCHMARK_IndexWotd
    } else if (options.index == "wotd") {
        // SEQAN_BENCHMARK_PARAM = 0 (Alloc), 1 (External)
        // indexParameters[0] = eager, first, lazy
        return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexWotd<>());
#endif
#ifdef SEQAN_BENCHMARK_IndexSttd64
    } else if (options.index == "sttd64") {
        // indexParameters[0u] = inMem / inExternal
#ifdef SEQAN_BENCHMARK_PARAM
        typedef Sttd64Config<SEQAN_BENCHMARK_CONFIG(SEQAN_BENCHMARK_PARAM)> TSttd64Config;
#else
        typedef Sttd64Config<> TSttd64Config;
#endif
        return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexSttd64<TSttd64Config>());
#endif
#ifdef SEQAN_BENCHMARK_IndexDigest
    } else if (options.index == "digest") {
        // Parameter SEQAN_BENCHMARK_PARAM = PARTITION_SIZE, OUTBUF_SIZE, INBUF_SIZE, TAIL_LENGTH, PREFIX_LENGTH
#ifdef SEQAN_BENCHMARK_PARAM
        typedef DigestConfig<SEQAN_BENCHMARK_CONFIG(SEQAN_BENCHMARK_PARAM)> TConfig;
#else
        typedef DigestConfig<> TConfig;
#endif

        return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexDigest<TConfig>());
#endif
#ifdef SEQAN_BENCHMARK_IndexFm
    } else if (options.index == "fm") {
        // indexParameters[0] = TOccSpec = WT / SBM
        // indexParameters[1] = compressionFactor
        SEQAN_ASSERT_GEQ(length(options.indexParameters), 1u);
        if (options.indexParameters[0u] == "WT") {
            return runText(csvStreamConstruct, csvStreamSearch, options, txt, FMIndex<WT<>, void>());
        } else if (options.indexParameters[0u] == "SBM") {
            return runText(csvStreamConstruct, csvStreamSearch, options, txt, FMIndex<SBM<>, void>());
        } else {
            std::cerr << "Unsupported FMIndex specialization: \"" << options.indexParameters[0u] << "\". Skipping." << std::endl;
        }
#endif
#ifdef SEQAN_BENCHMARK_IndexSadakane
    } else if (options.index == "sadakane") {
        // indexParameters[0] = block length
        // PSI_SAMPLE_RATE, SA_SAMPLE_RATE, ISA_SAMPLE_RATE
        return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexSadakane<unsigned char, SEQAN_BENCHMARK_CONFIG(SEQAN_BENCHMARK_PARAM)>());
#endif
#ifdef SEQAN_BENCHMARK_IndexLz
    } else if (options.index == "lz") {
        // indexParameters[0] = LZTRIE_NODE_END_TEXT_POSTION_SAMPLE_RATE
        if (length(options.indexParameters) == 0u) {
            return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexLZ<>());
        }
        unsigned int sampleRate = lexicalCast<unsigned int>(options.indexParameters[0u]);
        if (sampleRate == 1u) {
            return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexLZ<size_t, 1u>());
        } else if (sampleRate == 4u) {
            return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexLZ<size_t, 4u>());
        } else if (sampleRate == 16u) {
            return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexLZ<size_t, 16u>());
        } else if (sampleRate == 64u) {
            return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexLZ<size_t, 64u>());
        } else if (sampleRate == 256u) {
            return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexLZ<size_t, 256>());
        } else {
            std::cerr << "Unsupported LZ index sample rate: \"" << options.indexParameters[0u] << "\". Skipping." << std::endl;
        }
#endif
#ifdef SEQAN_BENCHMARK_IndexQGram
    } else if (options.index == "qgram" || options.index == "qsample") {
        // indexParameters[0] = addressing mode (direct, open)
        // indexParameters[1] = q
        // indexParameters[2] = sample step size
        // indexParameters[3] = construction algo
        
        SEQAN_ASSERT_GEQ(length(options.indexParameters), 2u);
        unsigned int q = lexicalCast<unsigned int>(options.indexParameters[1u]);
        if (options.indexParameters[0u] == "direct") {
            if (q == 2u) {
            //    return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<Shape<TChar, UngappedShape<2> > >());
            } else if (q == 3) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<Shape<TChar, UngappedShape<3> > >());
            } else if (q == 4) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<Shape<TChar, UngappedShape<4> > >());
            } else if (q == 5) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<UngappedShape<5> >());
            } else if (q == 6) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<UngappedShape<6> >());
            } else if (q == 7) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<UngappedShape<7> >());
            } else if (q == 8) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<UngappedShape<8> >());
            } else if (q == 12) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<UngappedShape<12> >());
            } else {
                std::cerr << "Unsupported qgram size: \"" << q << "\". Skipping." << std::endl;
            }
        } else if (options.indexParameters[0u] == "open") {
            if (q == 2u) {
            //    return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<UngappedShape<2>, OpenAddressing>());
            } else if (q == 3) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<UngappedShape<3>, OpenAddressing>());
            } else if (q == 4) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<UngappedShape<4>, OpenAddressing>());
            } else if (q == 5) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<UngappedShape<5>, OpenAddressing>());
            } else if (q == 6) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<UngappedShape<6>, OpenAddressing>());
            } else if (q == 7) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<UngappedShape<7>, OpenAddressing>());
            } else if (q == 8) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<UngappedShape<8>, OpenAddressing>());
            } else if (q == 12) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram<UngappedShape<12>, OpenAddressing>());
            } else {
                std::cerr << "Unsupported qgram size: \"" << q << "\". Skipping." << std::endl;
            }
        } else {
            std::cerr << "Unsupported qgram addressing: \"" << options.indexParameters[0u] << "\". Skipping." << std::endl;
        }
#endif
#ifdef SEQAN_BENCHMARK_IndexQGram2L
    } else if (options.index == "qgram2l") {
        // indexParameters[0] = addressing mode (direct, open)
        // indexParameters[1] = q
        // indexParameters[2] = subsequence length

        // IndexQGram2L in secondary memory:
        // - front-dir should be in main memory
        // - Buffers for front.dir, front.pos, back.dir, back.pos
        // - page size
        
        SEQAN_ASSERT_GEQ(length(options.indexParameters), 2u);
        unsigned int q = lexicalCast<unsigned int>(options.indexParameters[1u]);
        if (options.indexParameters[0u] == "direct") {
            if (q == 2) {
            //    return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<2> >());
            } else if (q == 3) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<3> >());
            } else if (q == 4) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<4> >());
            } else if (q == 5) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<5> >());
            } else if (q == 6) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<6> >());
            } else if (q == 7) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<7> >());
            } else if (q == 8) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<8> >());
            } else if (q == 12) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<12> >());
            } else {
                std::cerr << "Unsupported qgram size: \"" << q << "\". Skipping." << std::endl;
            }
        } else if (options.indexParameters[0u] == "open") {
            if (q == 2) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<2>, OpenAddressing>());
            } else if (q == 3) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<3>, OpenAddressing>());
            } else if (q == 4) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<4>, OpenAddressing>());
            } else if (q == 5) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<5>, OpenAddressing>());
            } else if (q == 6) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<6>, OpenAddressing>());
            } else if (q == 7) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<7>, OpenAddressing>());
            } else if (q == 8) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<8>, OpenAddressing>());
            } else if (q == 12) {
                return runText(csvStreamConstruct, csvStreamSearch, options, txt, IndexQGram2L<UngappedShape<12>, OpenAddressing>());
            } else {
                std::cerr << "Unsupported qgram size: \"" << q << "\". Skipping." << std::endl;
            }
        }
#endif
    } else {
        std::cerr << "Unsupported index type: \"" << options.index << "\". Skipping." << std::endl;
    }
    return false;
}

int mainWithOptions(Options & options) {
    //std::ostream & csvStreamConstruct = std::cout;
    std::ofstream csvStreamConstruct("results_construct.csv", std::ios::app);  // No, this is not an iOS application ;-)
    csvStreamConstruct << std::endl;
    std::ofstream csvStreamSearch("results_search.csv", std::ios::app);
    csvStreamSearch << std::endl;

    bool success = false;
    if (false) {
        ; // dummy
#ifdef SEQAN_BENCHMARK_char
    } else if (options.alphabet == ALPHABET_CHAR) {
        // We have to use unsigned char instead of char to get a correct sorting of non-ASCII characters (starting with a 1)
        unsigned char unsignedChar;
        success = runAlphabet(csvStreamConstruct, csvStreamSearch, options, unsignedChar);
#endif
#ifdef SEQAN_BENCHMARK_bool
    } else if (options.alphabet == ALPHABET_BOOL) {
        success = runAlphabet(csvStreamConstruct, csvStreamSearch, options, bool());
#endif
#ifdef SEQAN_BENCHMARK_Dna
    } else if (options.alphabet == ALPHABET_DNA) {
        success = runAlphabet(csvStreamConstruct, csvStreamSearch, options, Dna());
#endif
#ifdef SEQAN_BENCHMARK_Dna5
    } else if (options.alphabet == ALPHABET_DNA5) {
        success = runAlphabet(csvStreamConstruct, csvStreamSearch, options, Dna5());
#endif
#ifdef SEQAN_BENCHMARK_AminoAcid
    } else if (options.alphabet == ALPHABET_AMINOACID) {
        success = runAlphabet(csvStreamConstruct, csvStreamSearch, options, AminoAcid());
#endif
#ifdef SEQAN_BENCHMARK_wchar
    } else if (options.alphabet == ALPHABET_WCHAR) {
        success = runAlphabet(csvStreamConstruct, csvStreamSearch, options, wchar_t());
#endif
    } else {
        std::cerr << "Unsupported alphabet type: \"" << options.alphabet << "\"." << std::endl;
    }
    
    if (success) {
        std::cout << "Benchmark successful." << std::endl;
    }
    
    return success ? 0 : 1;
}

#endif  // #ifndef SANDBOX_TUM_APPS_FIND_INDEX_APPROX_BENCHMARK_FIND_INDEX_APPROX_BENCHMARK_H_
