 /*==========================================================================
                SeqAn - The Library for Sequence Analysis
                          http://www.seqan.de 
 ============================================================================
  Copyright (C) 2007

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

 ============================================================================
  $Id$
 ==========================================================================*/

#ifndef SEQAN_HEADER_MISC_RESOURCES_H
#define SEQAN_HEADER_MISC_RESOURCES_H

#include <ctime>

#ifdef __APPLE__
#include <mach/mach.h>
#include <mach/task.h>
#endif

#ifdef PLATFORM_WINDOWS
#include <windows.h>
#include <psapi.h>
//#pragma comment(lib,"psapi.lib")
#else
#include <sys/resource.h>
#endif

namespace SEQAN_NAMESPACE_MAIN
{

struct Resources
{
	double		userTime;
	double		sysTime;
	__uint64	virtMem;
	__uint64	virtMemPeak;
	__uint64	physMem;
	__uint64	physMemPeak;

#ifndef PLATFORM_WINDOWS
    char _line[128];
	struct rusage _r;
#endif

	Resources()
	{
		virtMemPeak = 0;
		physMemPeak = 0;
		_updateValues();
	}

	Resources(MinimalCtor)
	{
		virtMemPeak = 0;
		physMemPeak = 0;
	}

	 long parseLine(char* line)
	 {
        long i = strlen(line);
        while (*line < '0' || *line > '9') line++;
        line[i-3] = '\0';
        i = atol(line);
        return i;
    }

	void _updateValues()
	{
		// MEASURE RUNNING TIMES

#ifdef PLATFORM_WINDOWS
/*
		LARGE_INTEGER _frequency;
		QueryPerformanceFrequency(&_frequency);
		LARGE_INTEGER _timeStamp;
		QueryPerformanceCounter(&_timeStamp);
		userTime = (double)_timeStamp / (double)_frequency.QuadPart;
*/
		HANDLE procId = GetCurrentProcess();
		FILETIME cT, eT, kT, uT;
		if (GetProcessTimes(procId, &cT, &eT, &kT, &uT) != 0)
		{
			// FILETIME has a resolution of 100 nanoseconds
			userTime = (((__uint64)uT.dwHighDateTime << 32) + (__uint64)uT.dwLowDateTime) / 10000000.0;
			sysTime  = (((__uint64)kT.dwHighDateTime << 32) + (__uint64)kT.dwLowDateTime) / 10000000.0;
		} else 
		{
			userTime = 0;
			sysTime = 0;
		}

#else

		if (getrusage(RUSAGE_SELF, &_r) == 0)
		{
			userTime = _r.ru_utime.tv_sec + _r.ru_utime.tv_usec / 1000000.0;
			sysTime = _r.ru_stime.tv_sec + _r.ru_stime.tv_usec / 1000000.0;
		} else 
		{
			userTime = 0;
			sysTime = 0;
		}

#endif


		// MEASURE MEMORY CONSUMPTION

#ifdef __APPLE__

		struct task_basic_info t_info;
		mach_msg_type_number_t t_info_count = TASK_BASIC_INFO_COUNT;

		if (task_info(mach_task_self(), TASK_BASIC_INFO, (task_info_t)&t_info, &t_info_count) == KERN_SUCCESS)
		{
			physMem = t_info.resident_size / 1024ul;
			virtMem = t_info.virtual_size / 1024ul;
			physMemPeak = _max(physMem, physMemPeak);
			virtMemPeak = _max(virtMem, virtMemPeak);
		} else 
		{
			physMem = 0;
			virtMem = 0;
		}

#elif defined PLATFORM_WINDOWS

#if _MSC_VER > 1400
		PROCESS_MEMORY_COUNTERS_EX pmc;
#else
		PROCESS_MEMORY_COUNTERS pmc;
#endif
		if (GetProcessMemoryInfo(procId, reinterpret_cast<PROCESS_MEMORY_COUNTERS*>(&pmc), sizeof(pmc)) != 0)
		{
			physMem = pmc.WorkingSetSize / 1024ul;
			physMemPeak = pmc.PeakWorkingSetSize / 1024ul;
#if _MSC_VER > 1400
			virtMem = pmc.PrivateUsage / 1024ul;
			virtMemPeak = pmc.PeakPagefileUsage / 1024ul;
#else
			virtMem = 0;
			virtMemPeak = 0;
#endif
		} else
		{
			physMem = 0;
			virtMem = 0;
		}

#elif defined PLATFORM_GCC

		FILE* file = fopen("/proc/self/status", "r");
        int found = 0;
        while ((fgets(_line, 128, file) != NULL) && (found != 7))
		{
			if (strncmp(_line, "VmSize:", 7) == 0)
			{
				virtMem = parseLine(_line);
				found |= 1;
			}
			else if (strncmp(_line, "VmRSS:", 6) == 0)
			{
				physMem = parseLine(_line);
				found |= 2;
				continue;
			}
			else if (strncmp(_line, "VmPeak:", 7) == 0)
			{
				virtMemPeak = parseLine(_line);
				found |= 4;
			}
        }
        fclose(file);

		if ((found & 1) == 0) virtMem = 0;
		if ((found & 2) == 0) physMem = 0;
		physMemPeak = _max(physMem, physMemPeak);

#endif

	}
};

struct ResUserTime_;
struct ResSystemTime_;
struct ResVirtualMem_;
struct ResPhysMem_;
struct ResVirtualMemPeak_;

typedef Tag<ResUserTime_>		ResUserTime;
typedef Tag<ResSystemTime_>		ResSystemTime;
typedef Tag<ResVirtualMem_>		ResVirtualMem;
typedef Tag<ResPhysMem_>		ResPhysMem;
typedef Tag<ResVirtualMemPeak_>	ResVirtualMemPeak;


// return cpu or system times in seconds
inline double
getValue(Resources const &r, ResUserTime)
{
	return r.userTime;
}

inline double
getValue(Resources const &r, ResSystemTime)
{
	return r.sysTime;
}

// return memory consumption in bytes
inline __uint64
getValue(Resources const &r, ResPhysMem)
{
	return r.physMem;
}

inline __uint64
getValue(Resources const &r, ResVirtualMem)
{
	return r.virtMem;
}

inline __uint64
getValue(Resources const &r, ResVirtualMemPeak)
{
	return r.virtMemPeak;
}

Resources operator - (Resources const &a, Resources const &b) 
{
	Resources res = a;
	res.userTime -= b.userTime;
	res.sysTime -= b.sysTime;
	res.virtMem = _max(res.virtMem, b.virtMem);
	res.physMem = _max(res.physMem, b.physMem);
	return res;
}

Resources _max(Resources const &a, Resources const &b) 
{
	Resources res = Resources(MinimalCtor());
	res.userTime = _max(a.userTime, b.userTime);
	res.sysTime = _max(a.sysTime, b.sysTime);
	res.virtMem = _max(a.virtMem, b.virtMem);
	res.physMem = _max(a.physMem, b.physMem);
	return res;
}


std::ostream& operator << (std::ostream &out, Resources const &r) 
{
	out <<   "\"UserTime\": " << getValue(r, ResUserTime());
	out << ", \"SysTime\": " << getValue(r, ResSystemTime());
	out << ", \"PhysMem\": " << getValue(r, ResPhysMem());
	out << ", \"VirtMem\": " << getValue(r, ResVirtualMem());
	out << ", \"VirtMemPeak\": " << getValue(r, ResVirtualMemPeak());
	return out;
}

//////////////////////////////////////////////////////////////////////////////
}// namespace SEQAN_NAMESPACE_MAIN

#endif //#ifndef SEQAN_HEADER_...
