// ==========================================================================
//                         index_compressed_benchmark
// ==========================================================================
// Copyright (c) 2006-2011, Knut Reinert, FU Berlin
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * Neither the name of Knut Reinert or the FU Berlin nor the names of
//       its contributors may be used to endorse or promote products derived
//       from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL KNUT REINERT OR THE FU BERLIN BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
// DAMAGE.
//
// ==========================================================================
// Author: Tobias Stadler <stadler@in.tum.de>
// ==========================================================================
// Benchmark for the SeqAn module index_compressed.
// ==========================================================================

#include <algorithm>
#include <iostream>
#include <fstream>
#include <vector>

#ifndef PLATFORM_WINDOWS
#include <sys/time.h>
#endif

#include <seqan/basic.h>
#include <seqan/sequence.h>
#include <seqan/misc/misc_cmdparser.h>
#include <seqan/index_compressed.h>

#include "index_compressed_benchmark.h"

namespace seqan {
    template<typename TText, typename TBitBin, size_t LZTRIE_NODE_END_TEXT_POSTION_SAMPLE_RATE>
    inline bool
    indexCreate(Index<TText, IndexLZ<TBitBin, LZTRIE_NODE_END_TEXT_POSTION_SAMPLE_RATE> > & /*me*/, FibreSA const, const typename Size<TText>::Type /*blockLength*/)
    {
        return false;
    }
}

template<typename TText, typename TIndex>
static void benchmark(const String<CharString> &texts, const size_t blockLength, const size_t numberOfFindIterations, const size_t maxPatternSize, const size_t numberOfExtractIterations, const size_t maxSubstringSize, const bool displayOccurences = false)
{
    for (Iterator<const String<CharString> >::Type it = begin(texts); it != end(texts); ++it) {
        char *fileName = new char[length(*it) + 1];
        assign(fileName, *it);

        std::ifstream file(fileName);

        if (!file.is_open()) {
            std::cerr << "Error! Could not open file \""<< fileName << "\"!" << std::endl;

            continue;
        }

        std::cout << "Processing file: " << fileName << std::endl;

        TText text;
        read(file, text, Raw());

        TIndex index(text);

#ifndef PLATFORM_WINDOWS
        struct timeval _start, _stop;
        gettimeofday(&_start, 0);
#endif

        if (blockLength > 0) {
            indexCreate(index, FibreSA(), (typename Size<TText>::Type) blockLength);
        } else {
            indexCreate(index, FibreSA());
        }

#ifndef PLATFORM_WINDOWS
        gettimeofday(&_stop, 0);

        std::cout << "index (" << size(index) << "B) created in " << (_stop.tv_sec - _start.tv_sec) * 1000000 + _stop.tv_usec - _start.tv_usec << "us" << std::endl;
#else
        std::cout << "index (" << size(index) << "B) created" << std::endl;
#endif

        for (size_t i = 0; i < numberOfFindIterations; ++i) {
            const typename Position<TText>::Type patternStart = rand() % length(text), patternEnd = patternStart + (rand() % ::std::min(length(text) - patternStart, maxPatternSize)) + 1;
            const typename Infix<const TText>::Type &pattern = infix(text, patternStart, patternEnd);

            std::cout << "Searching infix(" << patternStart << ", " << patternEnd << ")" << std::endl;

            std::vector<typename SAValue<TIndex>::Type> occ;

            Finder<TIndex> finder(index);
            if (find(finder, pattern)) {
                do {
                    occ.push_back(position(finder));
                } while(find(finder));

                std::sort(occ.begin(), occ.end());
            }

            if(displayOccurences) {
                std::cout << "found " << occ.size() << " occurrences of the search pattern" << std::endl;
                std::cout << "hit at: ";
                for (std::vector<size_t>::const_iterator it = occ.begin(); it != occ.end(); ++it) {
                    std::cout << *it << " ";
                }
                std::cout << std::endl;
            }

            for (std::vector<size_t>::const_iterator it = occ.begin(); it != occ.end(); ++it) {
                if(pattern != infix(text, *it, *it + length(pattern))) {
                    std::cout << "    occurrence at " << *it << " does not match the search pattern!" << std::endl;
                }
            }
        }

        for (size_t i = 0; i < numberOfExtractIterations; ++i) {
            const typename Position<TText>::Type substringStart = rand() % length(text), substringEnd = substringStart + (rand() % ::std::min(length(text) - substringStart, maxSubstringSize)) + 1;
            const typename Infix<const TText>::Type substring = infix(text, substringStart, substringEnd);

            std::cout << "Extracting infix(" << substringStart << ", " << substringEnd << ")" << std::endl;

            if (extract<String<typename Value<TText>::Type> >(index, substringStart, substringEnd) != substring) {
                std::cout << "    extracted string does not match the substring!" << std::endl;
            }
        }
    }
}

int main(int argc, const char *argv[]) {
    typedef String<char> TCharString;
    typedef String<AminoAcid> TAminoAcidString;
    typedef String<Dna> TDnaString;
    typedef String<DnaQ> TDnaQString;
    typedef String<Dna5> TDna5String;
    typedef String<Dna5Q> TDna5QString;
    typedef String<Iupac> TIupacString;
    typedef String<Rna> TRnaString;
    typedef String<Rna5> TRna5String;

    CommandLineParser parser;
    Options options;

    setupCommandLineParser(parser, options);
    if (!parseCommandLine(options, parser, argc, argv)) {
        return EXIT_FAILURE;
    }

    switch(options.alphabet) {
        case 1:
            switch(options.index) {
                case 1:
                    benchmark<TCharString, Index<TCharString, IndexLZ<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
                case 2:
                    benchmark<TCharString, Index<TCharString, IndexSadakane<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
            }
            break;
        case 2:
            switch(options.index) {
                case 1:
                    benchmark<TAminoAcidString, Index<TAminoAcidString, IndexLZ<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
                case 2:
                    benchmark<TAminoAcidString, Index<TAminoAcidString, IndexSadakane<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
            }
            break;
        case 3:
            switch(options.index) {
                case 1:
                    benchmark<TDnaString, Index<TDnaString, IndexLZ<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
                case 2:
                    benchmark<TDnaString, Index<TDnaString, IndexSadakane<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
            }
            break;
        case 4:
            switch(options.index) {
                case 1:
                    benchmark<TDnaQString, Index<TDnaQString, IndexLZ<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
                case 2:
                    benchmark<TDnaQString, Index<TDnaQString, IndexSadakane<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
            }
            break;
        case 5:
            switch(options.index) {
                case 1:
                    benchmark<TDna5String, Index<TDna5String, IndexLZ<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
                case 2:
                    benchmark<TDna5String, Index<TDna5String, IndexSadakane<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
            }
            break;
        case 6:
            switch(options.index) {
                case 1:
                    benchmark<TDna5QString, Index<TDna5QString, IndexLZ<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
                case 2:
                    benchmark<TDna5QString, Index<TDna5QString, IndexSadakane<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
            }
            break;
        case 7:
            switch(options.index) {
                case 1:
                    benchmark<TIupacString, Index<TIupacString, IndexLZ<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
                case 2:
                    benchmark<TIupacString, Index<TIupacString, IndexSadakane<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
            }
            break;
        case 8:
            switch(options.index) {
                case 1:
                    benchmark<TRnaString, Index<TRnaString, IndexLZ<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
                case 2:
                    benchmark<TRnaString, Index<TRnaString, IndexSadakane<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
            }
            break;
        case 9:
            switch(options.index) {
                case 1:
                    benchmark<TRna5String, Index<TRna5String, IndexLZ<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
                case 2:
                    benchmark<TRna5String, Index<TRna5String, IndexSadakane<> > >(options.texts, options.sadakaneBlockLength, options.numberOfFindIterations, options.maxPatternLength, options.numberOfExtractIterations, options.maxSubstringLength, options.displayOccurences);
                    break;
            }
            break;
    }

    return EXIT_SUCCESS;
}
