// ==========================================================================
//                        index_suffix_trees_benchmark
// ==========================================================================
// Copyright (c) 2006-2010, Knut Reinert, FU Berlin
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * Neither the name of Knut Reinert or the FU Berlin nor the names of
//       its contributors may be used to endorse or promote products derived
//       from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL KNUT REINERT OR THE FU BERLIN BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
// DAMAGE.
//
// ==========================================================================
// Author: Alexander Aumann <aumann@in.tum.de>
// ==========================================================================
// Benchmark for the SeqAn module index_suffix_trees.
// ==========================================================================

#ifndef SANDBOX_TUM_APPS_INDEX_SUFFIX_TREES_BENCHMARK_INDEX_SUFFIX_TREES_BENCHMARK_H_
#define SANDBOX_TUM_APPS_INDEX_SUFFIX_TREES_BENCHMARK_INDEX_SUFFIX_TREES_BENCHMARK_H_

#include <iostream>
#include <fstream>

#include <seqan/basic.h>
#include <seqan/sequence.h>
#include <seqan/index.h>
#include <seqan/index_suffix_trees.h>

#include <seqan/misc/misc_cmdparser.h>

using namespace seqan;

// ============================================================================
// Forwards
// ============================================================================

// ============================================================================
// Tags, Classes, Enums
// ============================================================================

struct Options
{
    bool showHelp;
    bool showVersion;
    int algo;
    bool doConstruct;
    int searchNum;
    int type;
    String<CharString> texts;
    static const int STTD64 = 0;
    static const int ESA = 1;
    static const int WOTD = 2;

    static char const * ESA_PF;
    static char const * STTD64_PF;

    static const int ALGO_MAX = 2;
    static const int SEARCH_MAX = 1;
    enum TYPES_ALL
    {
         DNA5 = 1, DNA = 2, CHAR = 3
    };

    enum TYPES_STTD64
    {
        DNA5_1 = 7, DNA5_2 = 4, DNA5_4 = 5,
        DNA5_6 = 6, DNA5X_LARGE_0 = 8, DNA5Y_SMALL_0 = 9, TDNA5_LARGE_3 = 10
    };
//    static const int TYPES_ESA[];
//    static const int TYPES_STTD64[];

    Options()
    {
        // Set defaults.
        showHelp = false;
        showVersion = false;
        algo = 0;
        doConstruct = 0;
        searchNum = 0;
        type = DNA5;
    }
};

char const * Options::ESA_PF = "esa_stored";
char const * Options::STTD64_PF = "sttd64_partitionbools";
//const int Options::TYPES_ESA[] = {1,2,3};
//const int Options::TYPES_STTD64[] = {1,2,3,4};

// ============================================================================
// Metafunctions
// ============================================================================

// ============================================================================
// Functions
// ============================================================================

//forwards
template <typename TText, typename TSpec>
bool
constructSttd64(Index<TText, IndexSttd64<TSpec> > &, std::string, std::string);

template <typename TText, typename TSpec>
bool
openSttd64(Index<TText, IndexSttd64<TSpec> > &, std::string, std::string);

template <typename TText, typename TSpec>
void searchExec(Index<TText, TSpec> &, int);

void setupCommandLineParser(CommandLineParser & parser, Options const & options)
{
    addVersionLine(parser, "0.1");

    addTitleLine(parser, "**********************");
    addTitleLine(parser, "* index_suffix_trees_benchmark *");
    addTitleLine(parser, "**********************");
    addTitleLine(parser, "");
    addTitleLine(parser, "(c) 2011 by Alexander Aumann <aumann@in.tum.de>");

    addUsageLine(parser, "[OPTIONS] TEXT+");

    addSection(parser, "Main Options");
    addOption(
            parser,
            CommandLineOption("a", "algo", "set the kind of test", OptionType::Integer | OptionType::Label, options.algo));
    addOption(parser, CommandLineOption("s", "doSearches", "sets which search test shall be done", OptionType::Integer,
                                        options.searchNum));
    addOption(parser, CommandLineOption("t", "type", "chooses the index (and string) specialization", OptionType::Integer,
                                        options.type));
    addOption(parser, CommandLineOption("c", "doConstruct", "sets that the index should be constructed",
                                        OptionType::Bool, options.doConstruct));

    requiredArguments(parser, 2);
}

int parseCommandLineAndCheck(Options & options,
                             CommandLineParser & parser,
                             int argc,
                             char const ** argv)
{
    bool stop = !parse(parser, argc, argv);
    if (stop)
        return 1;
    if (isSetLong(parser, "help"))
    {
        options.showHelp = true;
        return 0;
    }
    if (isSetLong(parser, "version"))
    {
        options.showVersion = true;
        return 0;
    }

    if (isSetLong(parser, "doConstruct")) {
        options.doConstruct = true;
    }

    if (getOptionValueLong(parser, "algo", options.algo))
    {

        if (options.algo > Options::ALGO_MAX || options.algo < 0)
        {
            std::cout << "Illegal algorithm: " << options.algo << std::endl;
            std::cout << "Specify algorithm (-a= or --algo=) 0=STTD64 (default), 1=ESA, 2=Wotd, 3=Kurtz";
            return 1;
        }
    }
    int tmp;
    if (getOptionValueLong(parser, "doSearches", tmp))
    {
        options.searchNum = tmp;
    }

    if (options.searchNum < 0 || options.searchNum > Options::SEARCH_MAX)
    {
        std::cout << "Illegal search number : " << options.searchNum << std::endl;
        std::cout << "Search Number must be at most: " << Options::SEARCH_MAX << std::endl;
        return 1;
    }

    if (getOptionValueLong(parser, "type", tmp))
    {
        switch (tmp)
        {
        case Options::CHAR:
        case Options::DNA:
        case Options::DNA5:
            options.type = tmp;
            break;
        case Options::DNA5_2:
        case Options::DNA5_4:
        case Options::DNA5_6:
        case Options::DNA5X_LARGE_0:
        case Options::DNA5Y_SMALL_0:
        case Options::TDNA5_LARGE_3:
            if (options.algo == 0)
            {
                options.type = tmp;
                break;  //intended break only in if !
            }
        default:
            std::cout << "Type " << tmp << " not allowed for algo " << options.algo << "!\n";
            return 1;
        }

    }

    options.texts = getArgumentValues(parser);

    std::ifstream ifs;
    ifs.open(toCString(getValue(options.texts, 0)), std::ifstream::in);
    if (!ifs.good())
    {
        std::cerr << "The input file does not exist or cannot be read!\n";
        return 2;
    }

    CharString & dir = value(options.texts, 1);
#ifndef PLATFORM_WINDOWS
    if (getValue(dir, length(dir) - 1) != '/')
            appendValue(dir, '/');
#else
    if (getValue(dir, length(dir) - 1) != '\\')
            appendValue(dir, '\\');
#endif
    SEQAN_ASSERT_EQ(dir, getValue(options.texts, 1));

    String<char, MMap<> > testCreate;
    if (!open(testCreate, (std::string(toCString(dir)) + "zzzzz").c_str(), OPEN_CREATE | OPEN_APPEND))
    {
        std::cerr << "The output directory cannot be written";
        return 3;
    }

    if (options.doConstruct == false && options.searchNum == 0)
    {
        std::cout << "Do not construct and do not search => nothing to do!\n";
        return 1;
    }

    return 0;
}

int sttd64(std::string input, std::string outDir, bool construct, int search, int type)
{
    #define ONE_MB 1048576
    #define ONE_KB 1024

//    typedef Sttd64Config<1,
//            ONE_KB * 8,
//            256 * 128,    //250 MB string
//            1024 * 128,          //tree
//            300*128, ///512MB temp = 16384 if 4 bytes
//            300*128
//           ///1,6GB
//                    > CONFIG_UPTO_CHR1;

    typedef Sttd64Config<1,
            ONE_MB,
            256,    //250 MB string
            1024,          //tree
            300, ///512MB temp = 16384 if 4 bytes
            300
           ///1,6GB
                    > CONFIG_UPTO_CHR1;
    typedef Sttd64Config<0,
            ONE_MB,   //Page
            256,    //250 MB string
            5,          //tree
            1024, ///1GB temp
            1024    ///1GB suf
                    > SMALL_CONFIG_P0;

    typedef Sttd64Config<2,
            ONE_MB,   //Page
            1536,    //1,5 GB string
            256,          //tree
            600, ///300MB
            600///300MB
                    > VeryLARGE_CONFIG_DNA_P3;



    typedef Index<String<Dna5>, IndexSttd64<CONFIG_UPTO_CHR1> > TDna5XLargeIndex;
    TDna5XLargeIndex * indexDna5XLarge;

    typedef Index<String<char>, IndexSttd64<> > TCharIndex;

    typedef Index<String<Dna5>, IndexSttd64<SMALL_CONFIG_P0> > TDnaYSmallIndex;
    TDnaYSmallIndex * indexDnaYSmall;

    TCharIndex * esaChar = 0;
    typedef Index<String<Dna5>, IndexSttd64<Sttd64Config<1,
                                                        8192,   //Page
                                                        8640,    //250/4 MB string
                                                        5,          //tree
                                                        34560, ///250MB temp
                                                        34560///250MB suf
                                                        > > > TDna5Index;
    TDna5Index * esaDna5 = 0;
    typedef Index<String<Dna>, IndexSttd64<> > TDna4Index;
    TDna4Index * esaDna4 = 0;
    typedef Index<String<Dna>, IndexSttd64<VeryLARGE_CONFIG_DNA_P3> > TDna5_3Index;
    TDna5_3Index * indexDna5_3 = 0;
    typedef Index<String<Dna5>, IndexSttd64<Sttd64Config<4> > > TDna5_4Index;
    TDna5_4Index * esaDna5_4 = 0;
    typedef Index<String<Dna5>, IndexSttd64<Sttd64Config<6> > > TDna5_6Index;
    TDna5_6Index * esaDna5_6 = 0;
    
    // Supress warnings regarding unused variables
    (void) esaChar; (void) esaDna4; (void) esaDna5_4; (void) esaDna5_6;

    if (!construct && search <= 0) return 1;

    bool success = true;
    switch(type)
    {
    case Options::DNA5:

        esaDna5 = new TDna5Index();
        if (construct)
        {
            success = constructSttd64(*esaDna5, input, outDir);
            if (!success)
            {
                std::cout << "Could not construct!" << std::endl;
                return 1;
            }
        }

        if (search > 0)
        {
            if (!construct)
            {
                success = openSttd64(*esaDna5, input, outDir);
                if (!success)
                {
                    std::cout << "Could not reopen!" << std::endl;
                    return 1;
                }
            }
            searchExec(*esaDna5, search);
        }
        break;
    case Options::DNA5X_LARGE_0:

        indexDna5XLarge = new TDna5XLargeIndex();
        if (construct)
        {
            success = constructSttd64(*indexDna5XLarge, input, outDir);
            if (!success)
            {
                std::cout << "Could not construct!" << std::endl;
                return 1;
            }
        }

        if (search > 0)
        {
            if (!construct) {
                success = openSttd64(*indexDna5XLarge, input, outDir);
                if (!success)
                {
                    std::cout << "Could not reopen!" << std::endl;
                    return 1;
                }
            }
            searchExec(*indexDna5XLarge, search);
        }
        break;
    case Options::DNA5Y_SMALL_0:

        indexDnaYSmall = new TDnaYSmallIndex();

        if (construct) {
            success = constructSttd64(*indexDnaYSmall, input, outDir);
            if (!success)
            {
                std::cout << "Could not construct!" << std::endl;
                return 1;
            }
        }

        if (search > 0)
        {
            if (!construct) {
                success = openSttd64(*indexDnaYSmall, input, outDir);
                if (!success)
                {
                    std::cout << "Could not reopen!" << std::endl;
                    return 1;
                }
            }
            searchExec(*indexDnaYSmall, search);
        }
        break;
    case Options::TDNA5_LARGE_3:

        indexDna5_3 = new TDna5_3Index();

        if (construct) {
            success = constructSttd64(*indexDna5_3, input, outDir);
            if (!success)
            {
                std::cout << "Could not construct!" << std::endl;
                return 1;
            }
        }

        if (search > 0)
        {
            if (!construct) {
                success = openSttd64(*indexDna5_3, input, outDir);
                if (!success)
                {
                    std::cout << "Could not reopen!" << std::endl;
                    return 1;
                }
            }
            searchExec(*indexDna5_3, search);
        }
        break;
    }
    return 0;
}

template <typename TText>
bool openEsa(Index<TText, IndexEsa<> > & index, std::string outDir)
{
    return open(index, (outDir + Options::ESA_PF).c_str());
}

template <typename TSeq>
void printSorted(TSeq const & seq)
{
    std::vector<typename Value<TSeq>::Type> sortVector(length(seq));
    for (unsigned i = 0; i < length(seq); i++)
    {
        sortVector[i] = seq[i];
    }

    std::sort(sortVector.begin(), sortVector.end());
    for (unsigned i = 0; i < length(seq); i++)
    {
        std::cout << sortVector[i] << ",";
    }
}

template <typename TIndex>
void searchBasic(Iter<TIndex, VSTree<TopDown<> > > & it)
{
    typename Fibre<TIndex, FibreRawText>::Type pattern = "AGACTTGCCTAAGCTTAGCTTTA";
    bool notFound = false;

    time_t sBeg = time(NULL);
    time_t sum;
    time_t sEnd;

    while (repLength(it) < length(pattern))
    {
        // go down edge starting with the next pattern character
        if (!goDown(it, pattern[repLength(it)]))
        {
            notFound = true;
            break;
        }
        unsigned endPos = _min(repLength(it), length(pattern));
        // compare remaining edge characters with pattern
        if (infix(representative(it), parentRepLength(it) + 1, endPos) !=
                infix(pattern, parentRepLength(it) + 1, endPos))
        {
            notFound = true;
            break;
        }
    }

    sum = time(NULL);
    sum = sum - sBeg;

    if (!notFound)
    {
        String<typename SAValue<TIndex>::Type> occurrences = getOccurrences(it);
        std::cout << "All occurrences:\n[";
        for (unsigned i = 0; i < length(occurrences); i++)
        {
            printSorted(occurrences);
        }
        std::cout << "]\n";
    }
    else
    {
        std::cout << "Did not find pattern.\n";
    }

    time(&sBeg);

    clear(it);
    notFound = false;
    pattern = "CACTTAAAGGCATTCTTAAACCACAAACAATAGCT";
    while (repLength(it) < length(pattern))
    {
        // go down edge starting with the next pattern character
        if (!goDown(it, pattern[repLength(it)]))
        {
            notFound = true;
            break;
        }
        unsigned endPos = _min(repLength(it), length(pattern));

        // compare remaining edge characters with pattern
        if (infix(representative(it), parentRepLength(it) + 1, endPos) !=
                infix(pattern, parentRepLength(it) + 1, endPos))
        {
            notFound = true;
            break;
        }

    }

    time(&sEnd);
    sum += sEnd - sBeg;

    if (!notFound)
    {
        String<typename SAValue<TIndex>::Type> occurrences = getOccurrences(it);
        std::cout << "All occurrences:\n[";
        printSorted(occurrences);
        std::cout << "]\n";
    }
    else
    {
        std::cout << "Did not find pattern.\n";
    }


    time(&sBeg);

    clear(it);
    notFound = false;
    pattern = "GAGATTCCAG";
    while (repLength(it) < length(pattern))
    {
        // go down edge starting with the next pattern character
        if (!goDown(it, pattern[repLength(it)]))
        {
            notFound = true;
            break;
        }
        unsigned endPos = _min(repLength(it), length(pattern));
        // compare remaining edge characters with pattern
        if (infix(representative(it), parentRepLength(it) + 1, endPos) !=
                infix(pattern, parentRepLength(it) + 1, endPos))
        {
            notFound = true;
            break;
        }
    }

    time(&sEnd);
    sum += sEnd - sBeg;


    if (!notFound)
    {
        String<typename SAValue<TIndex>::Type> occurrences = getOccurrences(it);
        std::cout << "All occurrences:\n[";
        printSorted(occurrences);
        std::cout << "]\n";
    }
    else
    {
        std::cout << "Did not find pattern.\n";
    }
    std::cout << "Searching took: " << sum << " seconds("<< ((double) sum) / 60
            << " minutes)." << std::endl;

}

template <typename TText, typename TSpec>
void searchExec(Index<TText, TSpec> & index, int search)
{
    SEQAN_ASSERT_GT(search, 0);

    Iter<Index<TText, TSpec> , VSTree<TopDown<> > > it (index);

    switch(search)
    {
    case 1:
        searchBasic(it);
        break;
    }
}

template <typename TText, typename TSpec>
bool
constructEsa(Index<TText, IndexEsa<TSpec> > & index, std::string input, std::string outDir)
{
    String<typename Value<TText>::Type, MMap<> > inputStr;
    if (!open(inputStr, input.c_str(), OPEN_RDONLY)) {
        std::cout << "Can't open input string.\n";
        return false;
    }

    TText text;

    AutoSeqFormat format;
    guessFormat(inputStr, format);
    assignSeq(text, inputStr, format);
    text = "AGGTCCATAGACATGACAATATTAAACCCGGAGATTCCAGAGTCTTAGGATAACCTG";

    std::cout << "Constructing Esa Index...\n";

    time_t start = time(NULL);

    index = Index<TText, IndexEsa<TSpec> >(text);
    resize(index.sa, length(index.text));
    _indexRequireTopDownIteration(index);

    time_t end = time(NULL);

    std::cout << "Esa construction took " << (end - start) << " seconds("
            << ((double)(end - start)) / 60 << " minutes).\n";

    bool saveSuccess = save(index, (outDir + Options::ESA_PF).c_str());

    return saveSuccess;
}

int esa(std::string input, std::string outDir, bool construct, int search, int type)
{
    Index<String<char>, IndexEsa<> > * esaChar = 0;
    Index<String<Dna5>, IndexEsa<> > * esaDna5 = 0;
    Index<String<Dna>, IndexEsa<> > * esaDna4 = 0;
    if (construct)
    {
        bool success;
        switch(type) {
        case Options::DNA5:
            esaDna5 = new Index<String<Dna5>, IndexEsa<> >();
            success = constructEsa(*esaDna5, input, outDir);
            break;
        case Options::DNA:
            esaDna4 = new Index<String<Dna>, IndexEsa<> >();
            success = constructEsa(*esaDna4, input, outDir);
            break;

        case Options::CHAR:
        default:
            esaChar = new Index<String<char>, IndexEsa<> >();
            success = constructEsa(*esaChar, input, outDir);
            break;
        }
        if (!success)
        {
            std::cout << "Could not construct!" << std::endl;
            return 1;
        }
    }
    if (search > 0)
    {
        if (!construct) //reopen
        {
            bool success;
            switch(type)
            {
            case Options::DNA5:
                esaDna5 = new Index<String<Dna5>, IndexEsa<> >();
                success = openEsa(*esaDna5, outDir);
                break;
            case Options::DNA:
                esaDna4 = new Index<String<Dna>, IndexEsa<> >();
                success = openEsa(*esaDna4, outDir);
                break;

            case Options::CHAR:
            default:
                esaChar = new Index<String<char>, IndexEsa<> >();
                success = openEsa(*esaChar, outDir);
                break;

            }
            if (!success){
                std::cout << "Could not reopen!" << std::endl;
                return 1;
            }
        }

        switch(type)
        {
        case Options::DNA5:
            searchExec(*esaDna5, search);
            break;
        case Options::DNA:
            searchExec(*esaDna4, search);
            break;
        case Options::CHAR:
        default:
            searchExec(*esaChar, search);
            break;

        }

    }
    return 0;
}

template <typename TText, typename TSpec>
bool
constructSttd64(Index<TText, IndexSttd64<TSpec> > & index, std::string input, std::string outDir)
{
    std::cout << "Starting: copy string." << std::endl;

    // TODO(krugel) FIXME: This does not work anymore because we removed this constructor.
    // The constructor had to be removed because the text is to be stored outside the index class and the index only maintains a Holder
    //typedef Index<TText, IndexSttd64<TSpec> > TIndex;
    //index = TIndex(input.c_str(), outDir.c_str());
    ignoreUnusedVariableWarning(index);
    ignoreUnusedVariableWarning(input);
    ignoreUnusedVariableWarning(outDir);

    std::cout << "Copied string. Starting partitioning...." << std::endl;
    time_t start = time(NULL);

    if (!createPartitions(index))
    {
        std::cout << "Failed!\n";
        return false;
    }

    time_t endParts = time(NULL);

    std::cout << "Built parts, took: " << (endParts - start) << " seconds("
            << (double) (endParts - start) / 60 << " minutes)." << std::endl;
    bool success = createSuffixTrees(index);
    if (!success)
    {
        std::cout << "Failed!\n";
        return false;
    }

    time_t endTrees = time(NULL);

    std::cout << "Built trees, took: " << (endTrees- endParts) << " seconds("
            << (double) (endTrees - endParts) / 60 << " minutes)." << std::endl;

    if (!_createLeafDepths(index))
    {
        std::cout << "Failed!\n";
        return false;
    }

    time_t endD = time(NULL);

    std::cout << "Leaf depth filling took: " << (endD - endTrees) << " seconds("
            << (double) (endD - endTrees) / 60 << " minutes)." << std::endl;

    std::cout << "Construction time overall: " << (endD - start) << " seconds("
            << (double) (endD - start) / 60 << " minutes)." << std::endl;

    //the fibres are auto-persistent, but Virtual Prefix Array needs to be stored
    std::ofstream out((outDir + Options::STTD64_PF).c_str());
    std::copy(index.virtualPrefixTree.prefixesContained.begin(),
              index.virtualPrefixTree.prefixesContained.end(),
              std::ostream_iterator< bool>(out," "));
    out.close();
    if (!out.good())
    {
        std::cout << "Could not store bool vector.\n";
        return false;
    }
    return true;
}

template <typename TText, typename TSpec>
bool
openSttd64(Index<TText, IndexSttd64<TSpec> > & index, std::string inputPath, std::string outDir)
{
    typedef Index<TText, IndexSttd64<TSpec> > TIndex;

    // TODO(krugel) FIXME: This does not work anymore because we removed this constructor.
    // The constructor had to be removed because the text is to be stored outside the index class and the index only maintains a Holder
    //index = Index<TText, IndexSttd64<TSpec> >(inputPath.c_str(), outDir.c_str());
    ignoreUnusedVariableWarning(index);
    ignoreUnusedVariableWarning(inputPath);
    ignoreUnusedVariableWarning(outDir);

    resize(index.partitions, 1);
    resize(index.trees, 1);
    index.leafDepthsFilledIn = true;
    index.inMem = false;

    std::ifstream in((outDir + Options::STTD64_PF).c_str());

    if (!in.good())
    {
        std::cout << "Could not open stored bool vector!\n";
    }

    index.virtualPrefixTree.prefixesContained.clear();

    std::copy(std::istream_iterator<bool>(in),
              std::istream_iterator<bool>(),
              std::back_inserter(index.virtualPrefixTree.prefixesContained));

    in.close();

    if (in.bad())
    {
        std::cout << "Could not reload stored bool vector!\n";
        return false;
    }
    unsigned const preflenIndex = TIndex::PREFIX_LENGTH;
    if (index.virtualPrefixTree.numPartitions() != pow((double)ValueSize<typename Value<TText>::Type >::VALUE, (double) preflenIndex))
    {
        std::cout << "Partition numbers of index type and stored Virtual Tree do not match!\n";
        return false;
    }
    if (preflenIndex > 0)
        index.virtualPrefixTree.setPrefLenSuffix(suffix(value(index.txt), length(value(index.txt)) - preflenIndex + 1));
    else
        index.virtualPrefixTree.prefLenSuffix = "";
    return true;
}

int mainWithOptions(Options & options)
{
    switch(options.algo)
    {
    case 0:
        sttd64(toCString(getValue(options.texts, 0)), toCString(getValue(options.texts, 1)),
               options.doConstruct, options.searchNum, options.type);
        break;
    case 1:
        esa(toCString(getValue(options.texts, 0)), toCString(getValue(options.texts, 1)),
               options.doConstruct, options.searchNum, options.type);
        break;
    case 2:
        break;
    default:
        SEQAN_ASSERT(false);
    }
    return 0;
}

#endif  // #ifndef SANDBOX_TUM_APPS_INDEX_SUFFIX_TREES_BENCHMARK_INDEX_SUFFIX_TREES_BENCHMARK_H_
