// ==========================================================================
//                             find_index_approx
// ==========================================================================
// Copyright (c) 2006-2011, Knut Reinert, FU Berlin
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * Neither the name of Knut Reinert or the FU Berlin nor the names of
//       its contributors may be used to endorse or promote products derived
//       from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL KNUT REINERT OR THE FU BERLIN BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
// DAMAGE.
//
// ==========================================================================
// Author: Henrik Poppe <poppeh@in.tum.de>
// Author: Johannes Krugel <krugel@in.tum.de>
// ==========================================================================
// Base for module find_index_approx.
// Algorithms to find the occurences of a search pattern in a text allowing
// some errors and using an index structure. The algorithms implemented in
// this module are the classical dynamic programming algorithm performing a
// backtracking a in a suffix tree, and two variants of partitioning algorithms.
// 
// [NBYST2001] G. Navarro and R.A. Baeza-Yates and E. Sutinen and J. Tarhio
//    "Indexing Methods for Approximate String Matching",
//    IEEE Data Engineering Bulletin, 2001, 24, pp. 19-27,
//    http://sites.computer.org/debull/A01DEC-CD.pdf
// ==========================================================================

#ifndef SANDBOX_TUM_INCLUDE_SEQAN_FIND_INDEX_APPROX_FIND_INDEX_APPROX_BASE_H_
#define SANDBOX_TUM_INCLUDE_SEQAN_FIND_INDEX_APPROX_FIND_INDEX_APPROX_BASE_H_

namespace seqan {

// ============================================================================
// Tags, Classes, Enums
// ============================================================================

struct FindState_ {
    enum TState {
        STATE_EMPTY,           // Finder/pattern is empty.
        STATE_INITIAL,         // Finer/pattern has just been initialized.
        STATE_FOUND,           // Found the end position of a hit.
        STATE_NOTFOUND,        // No hit found, no more hits possible.
        STATE_BEGIN_FOUND,     // Found begin position.
        STATE_BEGIN_NOTFOUND,  // Found end but not begin, should not happen.
        STATE_NO_HIT           // Set manually to non-hit.
    };
};

// ============================================================================
// Metafunctions
// ============================================================================

// ----------------------------------------------------------------------------
// DefaultIndexPattern
// ----------------------------------------------------------------------------
// TODO(krugel) Move to find_pattern_base.h

// We use the default pattern spezialization (but e.g. for IndexDigest we need to specialize this).
template <typename TNeedle, typename TIndex>
struct DefaultIndexPattern {
    typedef typename DefaultPattern<TNeedle>::Type          Type;
};


// ----------------------------------------------------------------------------
// IsIndex<Index>
// ----------------------------------------------------------------------------
// TODO(krugel) Move to index_base.h

template <typename THaystack>
struct IsIndex {
    typedef False                                           Type;
};

template <typename TString, typename TIndexSpec>
struct IsIndex<Index<TString, TIndexSpec> > {
    typedef True                                            Type;
};

// ----------------------------------------------------------------------------
// SupportsSuffixTreeIteration<Index>
// ----------------------------------------------------------------------------
// TODO(krugel) Move to index_base.h

template <typename TIndexSpec>
struct SupportsSuffixTreeIteration {
    typedef False Type;
};

template <typename TSpec>
struct SupportsSuffixTreeIteration<IndexEsa<TSpec> > {
    typedef True Type;
};

template <typename TSpec>
struct SupportsSuffixTreeIteration<IndexWotd<TSpec> > {
    typedef True Type;
};

template <typename TOccSpec, typename TSpec>
struct SupportsSuffixTreeIteration<FMIndex<TOccSpec, TSpec> > {
    //TODO typedef True Type; as soon as backtracking is implemented for FMIndex
    typedef False Type;
};

// ----------------------------------------------------------------------------
// GetOccurrences<Index>
// ----------------------------------------------------------------------------

// TODO(krugel) Move to index_base.h ?
// Determines the return type of getOccurrences(Iterator<VSTree>), getOccurrences(index, shape) and related
// For some indexes this is an Infix<String<SAValue> > while for others it is String<SAValue>
template <typename TIndex>
struct GetOccurrences {
    //typedef String<typename SAValue<TIndex>::Type>          Type;
    typedef typename Infix<typename Fibre<TIndex, FibreSA>::Type const>::Type Type;
};

//template <typename TText, typename TShapeSpec, typename TSpec>
//struct GetOccurrences<Index<TText, IndexQGram<TShapeSpec, TSpec> > > {
//    typedef Index<TText, IndexQGram<TShapeSpec, TSpec> >    TIndex;
//    typedef typename Infix<typename Fibre<TIndex, FibreSA>::Type const>::Type Type;
//};

// ----------------------------------------------------------------------------
// SupportsApproximateSearch<Pattern>
// ----------------------------------------------------------------------------
// TODO(krugel) Move to find_pattern_base.h

template <typename TPatternSpec>
struct SupportsApproximateSearch {
    typedef False Type;
};

template <>
struct SupportsApproximateSearch<AbndmAlgo> {
    typedef True Type;
};

template <typename TScore, typename TSpec, typename TFindBeginPatternSpec>
struct SupportsApproximateSearch<DPSearch<TScore, TSpec, TFindBeginPatternSpec> > {
    typedef True Type;
};

template <typename TSpec, typename TFindBeginPatternSpec>
struct SupportsApproximateSearch<Myers<TSpec, TFindBeginPatternSpec> > {
    typedef True Type;
};

template <typename TSpec>
struct SupportsApproximateSearch<Swift<TSpec> > {
    typedef True Type;
};

//template <typename TDistance>
//struct SupportsApproximateSearch<Backtracking<TDistance> > {
//    typedef True Type;
//};
//
//template <typename TSpec>
//struct SupportsApproximateSearch<Pigeonhole<TSpec> > {
//    typedef True Type;
//};

// ----------------------------------------------------------------------------
// MatchesAreSorted<Finder>
// ----------------------------------------------------------------------------
// TODO(krugel) Move to find_base.h

template<typename TFinder>
struct MatchesAreSorted {
    typedef True Type;
};

template<typename TText, typename TIndexSpec, typename TSpec>
struct MatchesAreSorted<Finder<Index<TText, TIndexSpec>, TSpec> > {
    typedef False Type;
};


/*
    List of Finder classes:
    
    struct Default;
    struct FinderLcpe_;       	// "core/include/seqan/index\index_base.h"(86)
    struct FinderMlr_;       	// "core/include/seqan/index\index_base.h"(85)
    struct FinderQGramLookup_;  // "core/include/seqan/index\find_index_qgram.h"(44)
    QGramFindLookup
    struct FinderSTree_;       	// "core/include/seqan/index\index_base.h"(87)
    struct FinderFMIndex_;      // "core/include/seqan/index\index_fm.h"(62)
    MultipatternFinder
    PizzaChiliFinder
    EsaFindMlr
    struct Swift;
    Pigeonhole<TSpec>

    template<typename TSpec = void>
    struct FinderLz {};

    template<typename TSpec = void>
    struct FinderSadakane {};

    template<typename TVerification, typename TSpec = void>
    struct FinderDigest {};

    template<typename TSpec = void>
    struct FinderSuffixTreeGoDown {};
*/


// ============================================================================
// Functions
// ============================================================================

// ----------------------------------------------------------------------------
// minSupportedPatternLength
// ----------------------------------------------------------------------------

// TODO(krugel) Move to find_index.h?
// By default we assume the index does not have a minimal possible pattern length.
// Real definitions are in find_index_qgram_ext.h

template <typename TText, typename TIndexSpec, typename TFinderSpec>
inline typename Size<Finder<Index<TText, TIndexSpec>, TFinderSpec> >::Type
minSupportedPatternLength(Finder<Index<TText, TIndexSpec>, TFinderSpec> & finder) {
    typedef Index<TText, TIndexSpec>                        TIndex;
    typedef typename DefaultIndexPattern<TText, TIndex>::Type TPattern;
    TPattern pattern;
    return minSupportedPatternLength(finder, pattern);
}

template <typename THaystack, typename TFinderSpec, typename TPattern>
inline typename Size<Finder<THaystack, TFinderSpec> >::Type
minSupportedPatternLength(Finder<THaystack, TFinderSpec> & /*finder*/, TPattern & /*pattern*/) {
    return 0u;
}

// ----------------------------------------------------------------------------
// configureSave
// ----------------------------------------------------------------------------

// TODO(krugel) Move to index_base.h?
// For some indexes it is necessary to define the disk location already before
// the construction (if memory-mapped files are used). For compatibility we
// define the corresponding function here for all other indexes.

template <typename TIndex>
inline void
configureSave(TIndex & index, const char * filename) {
    ignoreUnusedVariableWarning(index);
    ignoreUnusedVariableWarning(filename);
    // do nothing
}

}  // namespace seqan

#endif  // SANDBOX_TUM_INCLUDE_SEQAN_FIND_INDEX_APPROX_FIND_INDEX_APPROX_BASE_H_
