// ==========================================================================
//                             index_suffix_trees
// ==========================================================================
// Copyright (c) 2006-2011, Knut Reinert, FU Berlin
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * Neither the name of Knut Reinert or the FU Berlin nor the names of
//       its contributors may be used to endorse or promote products derived
//       from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL KNUT REINERT OR THE FU BERLIN BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
// DAMAGE.
//
// ==========================================================================
// Author: Alexander Aumann <a.aumann@web.de>
// ==========================================================================
// Suffix tree interface for the space-reduced suffix tree of Kurtz.
// ==========================================================================

#ifndef SANDBOX_TUM_INCLUDE_SEQAN_INDEX_SUFFIX_TREES_INDEX_STKURTZ_STREE_H_
#define SANDBOX_TUM_INCLUDE_SEQAN_INDEX_SUFFIX_TREES_INDEX_STKURTZ_STREE_H_

namespace seqan {

// ============================================================================
// Forwards
// ============================================================================

// ============================================================================
// Tags, Classes, Enums
// ============================================================================

template<typename TIndex, typename TNodes, typename TLeaves>
struct VertexStKurtz
{
    StKurtzTarget<TNodes, TLeaves> node;
    VertexStKurtz()
    {
    }
    VertexStKurtz(MinimalCtor)
    {
    }
};

// ============================================================================
// Metafunctions
// ============================================================================

template<typename TText, typename TSpec>
struct VertexDescriptor<Index<TText, IndexStKurtz<TSpec> > >
{
    typedef typename TSpec::NODE_TYPE TNodes_;
    typedef typename TSpec::LEAF_TYPE TLeaves_;
    typedef VertexStKurtz<Index<TText, IndexStKurtz<TSpec> >, TNodes_, TLeaves_> Type;
};

template <typename TText, typename TSpec>
struct SAValue<Index<TText, IndexStKurtz<TSpec> > >
{
    typedef typename Size<TText>::Type Type;
};

// ============================================================================
// Tags, Classes, Enums
// ============================================================================

template<typename TText, typename TIndexSpec, typename TSpec>
class Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TSpec> >
{
public:
    typedef Index<TText, IndexStKurtz<TIndexSpec> > TIndex;
    typedef typename VertexDescriptor<TIndex>::Type TVertexDesc;

    TIndex const *index; // container of all necessary tables
    TVertexDesc vDesc; // current interval in suffix array and

    TVertexDesc _parentDesc;        //to be able to goUp one node

    Iter(TIndex &_index) :
            index(&_index)
    {
        _indexRequireTopDownIteration(_index);
        goRoot(*this);
    }

    Iter(TIndex const &_index, MinimalCtor) :
            index(&_index), vDesc(MinimalCtor())
    {
    }

    Iter(Iter const &_origin) :
            index(&container(_origin)), vDesc(value(_origin))
    {
    }

};

template<typename TText, typename TIndexSpec, typename TSpec>
class Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TopDown<TSpec> > >
    : public Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TSpec> >
{
    typedef Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TSpec> > TBase_;
    typedef typename TBase_::TIndex TIndex;
    typedef typename TBase_::TVertexDesc TVertexDesc;
public:
    Iter(TIndex &_index) :
            TBase_(_index)
    {
        _indexRequireTopDownIteration(_index);
    }

    Iter(TIndex const &_index, MinimalCtor) :
            TBase_(_index, MinimalCtor())
    {
    }

    Iter(Iter const &_origin) :
        TBase_(_origin)
    {
    }

};

// ============================================================================
// Functions
// ============================================================================

template<typename TText, typename TIndexSpec, class TSpec>
inline void clear(Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TSpec> > &it)
{
    SEQAN_CHECKPOINT;
    value(it).node.posNode = 0;
    value(it).node.posLeaf = StKurtzBitMasks::NoPos;
    value(it).node.nodeType = BIG_NODE;
}

template<typename TText, typename TIndexSpec, class TSpec>
inline void goRoot(Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TSpec> > &it)
{
    SEQAN_CHECKPOINT;
    clear(it);
}

template<typename TText, typename TIndexSpec, class TSpec>
inline bool isRoot(Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TSpec> > const &it)
{
    SEQAN_ASSERT_NOT(value(it).node.nodeType == SMALL_NODE && value(it).node.posNode == 0);
    return (value(it).node.nodeType == BIG_NODE && value(it).node.posNode == 0);
}

// go down the leftmost edge (including empty $-edges at least atm, but would it be this function??)
/*
template<typename TText, class TIndexSpec, class TSpec, typename TDfsOrder>
inline bool _goDown(Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TopDown<TSpec> > > &it ,
VSTreeIteratorTraits<TDfsOrder, False> const)
{
    typedef Index<TText, IndexStKurtz<TIndexSpec> > TIndex;
    TIndex const & index = container(it);
    if (isLeaf(it)) return false;

    value(it) = _followReference(index.nodeTable, index.leafTable,
                                 _getFirstChildRef(index.nodeTable, value(it).posNode));
    return true;

}
*/

template<typename TText, class TIndexSpec, class TSpec>
inline bool _goDown(Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TopDown<TSpec> > > &it , DeepestSpec<VSTree<TopDown<> > >::Type const)
{
    typedef Index<TText, IndexStKurtz<TIndexSpec> > TIndex;
    TIndex const & index = container(it);
    if (isLeaf(it)) return false;

    value(it).node = _followReference(index.nodeTable, index.leafTable,
                                 _getFirstChildRef(index.nodeTable, value(it).node.posNode));
    return true;

}

template <typename TText, typename TSpec>
inline bool _isLeaf(VertexStKurtz<Index<TText, IndexStKurtz<TSpec> > ,
                    typename TSpec::NODE_TYPE, typename TSpec::LEAF_TYPE> const & vDesc)
{
    return (vDesc.node.nodeType == LEAF);
}

template <typename TText, typename TIndexSpec, class TSpec, typename TDfsOrder >
inline bool _isLeaf(
    Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TSpec> > const &it,
    VSTreeIteratorTraits<TDfsOrder, False> const)
{
    return _isLeaf(value(it));
}

template <typename TText, typename TIndexSpec, class TSpec, typename TDfsOrder >
inline bool _isLeaf(
    Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TSpec> > const &it,
    VSTreeIteratorTraits<TDfsOrder, True> const)
{
    return _isLeaf(value(it));
}

template <typename TText, typename TIndexSpec, class TSpec, typename TSpec2>
inline bool _isLeaf(
    Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TSpec> > const &it,
    VSTree<TopDown<TSpec2> > const)
{
    return _isLeaf(value(it));
}

template <typename TText, typename TIndexSpec, typename TSpec, typename TValue>
inline bool
_getNodeByChar(
    Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TSpec> > const &it,
    TValue c,
    typename VertexDescriptor<Index<TText, IndexStKurtz<TIndexSpec> > >::Type &childDesc)
{

    typedef  StKurtzTarget<typename TIndexSpec::NODE_TYPE, typename TIndexSpec::LEAF_TYPE> TNode;
    //TNode father = value(it).node;
    TNode leftBrotherDummy;
    bool edgeFound;
    childDesc.node = _getNodeForEdge(container(it),
                    value(it).node,
                    c,
                    leftBrotherDummy,
                    edgeFound,
                    length(container(it).nodeTable));   //all nodes inserted, this should not be of importance
    return edgeFound;
}

template <typename TText, class TIndexSpec, class TSpec, typename TDfsOrder, typename THideEmptyEdges >
inline bool _goRight(
    Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TopDown<TSpec> > > &it,
    VSTreeIteratorTraits<TDfsOrder, THideEmptyEdges> const)
{
    typedef Index<TText, IndexStKurtz<TIndexSpec> > TIndex;
    TIndex const & index = container(it);
    if (_isLastChild(index.nodeTable, index.leafTable, value(it).node)) return false;

    value(it).node = _getBBOrSlinkFatherTarget(index.nodeTable, index.leafTable, value(it).node);
    return true;
}

template <typename TText, class TIndexSpec, class TSpec>
inline void
_historyPush(Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TopDown<TSpec> > > &it)
{
    it._parentDesc = value(it);
}
template <typename TText, class TIndexSpec, class TSpec>
inline void
_historyPush(Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TopDown<ParentLinks<TSpec> > > > &it)
{
    //TODO(aumann): implement for HistoryTopDown Iterator
//        typedef Iter<Index<TText, IndexEsa<TIndexSpec> >, VSTree<TopDown<ParentLinks<TSpec> > > > TIter;
//        typename HistoryStackEntry_<TIter>::Type h;
//        h.range = value(it).range;
//
//        value(it).parentRight = value(it).range.i2;
//        appendValue(it.history, h);
}

template <typename TText, typename TIndexSpec, typename TSpec>
inline typename Size<Index<TText, IndexStKurtz<TIndexSpec> > >::Type
parentRepLength(Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TopDown<TSpec> > > const &it)
{
    if (isRoot(it))
        return 0;

    return _getDepth(container(it).nodeTable, it._parentDesc.node.posNode,length(container(it).nodeTable));
}

template <typename TText,  typename TIndexSpec, typename TSpec>
inline typename Size<Index<TText, IndexStKurtz<TIndexSpec> > >::Type
repLength(Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TopDown<TSpec> > > const &it)
{
//        if (_isRoot(it)) /*superfluous*/
//            return 0;

    if (isLeaf(it))
    {
        //depth is the leaf's values
        return value(it).node.posLeaf;  //is stored at its start position, which is its depth
    }

    return _getDepth(container(it).nodeTable, value(it).node.posNode, length(container(it).nodeTable));

}

template <typename TText, typename TIndexSpec,class TSpec>
inline typename SAValue<Index<TText, IndexStKurtz<TIndexSpec> > >::Type
getOccurrence(Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TSpec> > const &it)
{
    if (isLeaf(it))
    {
        return value(it).node.posLeaf;
    }

    return _getHeadPosition(container(it).nodeTable, value(it).node.posNode, length(container(it).nodeTable));
}

template <typename TText, typename TIndexSpec, typename TSpec>
inline String<typename SAValue<Index<TText, IndexStKurtz<TIndexSpec> > >::Type const >
getOccurrences(Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TSpec> > const &it)
{
    //atm simply do a recursive search
    //typedef typename TIndexSpec::NODE_TYPE TNodes;
    //typedef typename TIndexSpec::LEAF_TYPE TLeaves;
    //typedef StKurtzBitMasks BM;

    Index<TText, IndexStKurtz<TIndexSpec> > const & index = container(it);
    String<typename SAValue<Index<TText, IndexStKurtz<TIndexSpec> > >::Type> occurrences;
    if (isLeaf(it))
    {
        appendValue(occurrences, getOccurrence(it));
        return occurrences;
    }
    _getOccurrencesRec(index, value(it).node, occurrences);

    return occurrences;

}

template <typename TText, typename TIndexSpec, typename TSpec>
inline typename Size<Index<TText, IndexStKurtz<TIndexSpec> > >::Type
countOccurrences(Iter<Index<TText, IndexStKurtz<TIndexSpec> >, VSTree<TSpec> > const &it)
{
    return length(getOccurrences(it));
}

template <typename TText, typename TIndexSpec>
inline void
_getOccurrencesRec(Index<TText, IndexStKurtz<TIndexSpec> > const & index,
                  StKurtzTarget<typename TIndexSpec::NODE_TYPE, typename TIndexSpec::LEAF_TYPE> const & currentFather,
                  String<typename SAValue<Index<TText, IndexStKurtz<TIndexSpec> > >::Type> & occs)
{
    //atm simply do a recursive search
    typedef typename TIndexSpec::NODE_TYPE TNodes;
    typedef typename TIndexSpec::LEAF_TYPE TLeaves;
    //typedef StKurtzBitMasks BM;

    StKurtzTarget<TNodes, TLeaves> currentChild;
    unsigned firstChildRef = _getFirstChildRef(index.nodeTable, currentFather.posNode);
    currentChild = _followReference(index.nodeTable, index.leafTable, firstChildRef);
    while (!_isLastChild(index.nodeTable, index.leafTable, currentChild))
    {
        if (currentChild.nodeType == LEAF)
        {
            appendValue(occs, currentChild.posLeaf);
        }
        else
        {
            _getOccurrencesRec(index, currentChild, occs);
        }
        currentChild = _getBBOrSlinkFatherTarget(index.nodeTable, index.leafTable, currentChild);
    }

    //append values for last child, not possible to do this in a do/while more elegantly, so do not bother!
    if (currentChild.nodeType == LEAF)
    {
        appendValue(occs, currentChild.posLeaf);
    }
    else
    {
        _getOccurrencesRec(index, currentChild, occs);
    }
}

}  // namespace seqan

#endif  // SANDBOX_TUM_INCLUDE_SEQAN_INDEX_SUFFIX_TREES_INDEX_STKURTZ_STREE_H_
