#ifndef _module_h_
#define _module_h_

#include <map>
#include <string>

#include "analyze_setting.h"
#include "index_map.h"

namespace papi
{
    /**
     * @class Module
     *
     * Abstract base class for a analyze module
     */
    class Module
    {
        public:
        /**
         * Initialize the module
         *
         * @param[in] settings General and module settings
         * @param[in] buf_size Size of the input buffer
         * @param[in] ind_map Index map
         */
        virtual void init(AnalyzeSetting &settings, long long buf_size,const  IndexMap<char> *ind_map)=0;
        
        /**
         * Analyze text and print results.
         *
         * @param[in] bufSize Length of the input buffer
         * @param[in] buffer Buffer containing the input file (using indices insted of the actual characters)
         * @param[in] directory Output directory, or empty if none
         * @param[in] write_stdout True, if printing to stdout
         * @param[in] file_id File ID (for result header)
         * @param[in] file_path Path to file (for result header)
         * @param[in] file_length Length of the raw (file for result header)
         */
        virtual void process(long long bufSize,unsigned char const* buffer,std::string directory,bool write_stdout, long file_id,std::string file_path,long long file_length)=0;
        
        /**
         * @return Name of the module
         */
        virtual const std::string getId()=0;
        
        /**
         * Destructor
         */
        virtual ~Module(){}
    };
        
    /**
     * @struct module_thread
     *
     * Helper class to start a module in a thread
     */
    struct module_thread
    {
        /// The module
        Module *module;
        
        /// The buffer
        unsigned char *buffer;
        
        /// Buffer size
        long long buffer_size;
        
        /// Module settings
        AnalyzeSetting *settings;
        
        /// Output directory
        string *output_directory;
        
        /// True, if printing st stdout
        bool write_stdout;
        
        /// File id
        long file_id;
        
        /// Path to input file
        string *absolute_input_path;
        
        /// Index map
        IndexMap<char> *ind_map;
        
        /// Length of raw file
        long long file_length;
        
        
        /**
         * Constructor
         *
         * @param module The module
         * @param file_length Length of raw input file
         * @param buffer_size Buffer size
         * @param buffer The buffer
         * @param settings Module settings
         * @param output_directory Ouput directory
         * @param write_stdout True, if printing to stdout
         * @param file_id File ID
         * @param absolute_input_path Path to input file
         * @param ind_map Index map
         */
        module_thread(Module *module,long long file_length,  long long buffer_size,unsigned char* buffer, AnalyzeSetting *settings, 
                      string *output_directory,bool write_stdout, long file_id,string *absolute_input_path, IndexMap<char> *ind_map):
        module(module),buffer(buffer),buffer_size(buffer_size), settings(settings),output_directory(output_directory),
        write_stdout(write_stdout), file_id(file_id), absolute_input_path(absolute_input_path), ind_map(ind_map), file_length(file_length)
        {}
        
        /**
         * Starts the thread
         */
        void operator() ()
        {
            module->init(*settings ,buffer_size, ind_map);
            module->process(buffer_size, buffer,*output_directory,write_stdout,file_id,*absolute_input_path,file_length);
        }
        
    };
}

#endif

