/*******************************************************************************
 * This program was created in the context of the PAPI-project:                *
 * "Practical Approximate Pattern Matching with Index Structures"              *
 * Project website: http://www14.in.tum.de/spp1307/                            *
 * Author: Andre Dau <dau#in.tum.de>                                           *
 ******************************************************************************/

/**
 *******************************************************************************
 * @file tt-analyze.cpp
 *
 * <tt>tt-analyze</tt> calculates efficiently some statistical properties of texts
 * (such as entropy, mutual information function, qgram distributions, word
 * distributions) and estimates parameters for probability models implemented by
 * <tt>tt-generate</tt> (such as discrete autoregressive process, markov chain,
 * approximate repeats model by Allison et al. (1998)).
 *
 * The <tt>tt-generate</tt> and <tt>tt-analyze</tt> tools are intended to be
 * used together. More details on the tools can be found in <i> Andre Dau (2010):
 * Analysis of the structure and statistical properties of texts and generation 
 * of random texts</i>.
 *
 * <tt>tt-analyze</tt> needs configuration settings which can be specified in 
 * a settings file or directly in the command line (see <i>Usage</i>). The 
 * <i>sample_config.ini</i>, which comes with the source code, contains all available 
 * settings and a short explanation for each setting. If a setting is not set
 * a default value is used.
 *
 * In order to further faciliate the estimation of parameters and the generation of
 * new random texts, <tt>tt-analyze</tt> allows the usage of presets. Presets define a
 * minimal set of settings for <tt>tt-analyze</tt> which are sufficient to 
 * estimate all necessary parameters for a specific probability model of 
 * <tt>tt-generate</tt>. 
 *
 * If a setting is defined in multiple places, the precedence is as follow:
 * Command line argument    over    settings file   over    preset.
 *
 * The results can be written to a result directory as csv-files. Along with the 
 * results, a file <i>settings.ini</i> will be generated. It contains all used settings
 * as well as general information such as the complete path to the source file,
 * the date of execution, the file length and the command line string.
 * The naming convention for the output files is <i>[module_name]_[content_type].csv</i>
 *
 * The contents of the files can also be written to <tt>stdout</tt>. This is especially
 * useful when combining <tt>tt-analyze</tt> and <tt>tt-generate</tt> since this 
 * allows direct piping from <tt>tt-analyze</tt> to <tt>tt-generate</tt>.
 *
 * WARNING: If printing to <tt>stdout</tt> all modules must be executed one after 
 * another. Otherwise the modules will print to <tt>stdout</tt> simultaneously 
 * resulting in a corrupt stream. This means every module needs a unique thread 
 * group number in <tt>[module_selection]</tt> (see <i>sample_config.ini</i> for 
 * more details).
 *
 *******************************************************************************
 * @b Modules:
 *  @code
 * FrequencyDistribution   --   character distribution
 *                              bigram distribution
 *                              word distribution
 * Correlation             --   character distribution
 *                              mutual information function
 *                              estimated autocorrelation parameters of a Discrete Autoregressive Process  
 *                              (see: Jacobs, P. A. & Lewis, P. A. W.: 
 *                                    Stationary Discrete Autoregressive-Moving Average Time Series 
 *                                    Generated By Mixtures
 *                                    In: Journal of Time Series Analysis 4 (1983), Nr. 1, pp. 19-36
 *                                    http://dx.doi.org/10.1111/j.1467-9892.1983.tb00354.x)
 *                              (see: Dehnert, M. & Helm, W. E. & Huett, M.-Th.: 
 *                                    A Discrete Autoregressive Process as a model for short-range 
 *                                    correlations in DNA sequences
 *                                    In: Physica A 327 (2003), pp. 535-553
 *                                    http://dx.doi.org/10.1016/S0378-4371(03)00399-6)
 *                              (see: Huett, M.-Th. & Dehnert, M. : 
 *                                    Methoden der Bioinformatik: Eine Einfuehrung
 *                                    Springer, 2006)
 * Entropy                 --   character distribution
 *                              qgram distribution
 *                              block entropy
 *                              conditional entropy
 *                              (The entropy estimator uses a correction term by Miller; 
 *                              see: Schuermann, T. & Grassberger, P.: 
 *                                   Entropy estimation of symbol sequences 
 *                                   In: CHAOS 6 (1996), Nr. 3, pp. 414-427
 *                                   http://dx.doi.org/10.1063/1.166191s)
 * ApproximateRepeats      --   character distribution
 *                              qgram distribution
 *                              approximate repeat model parameters:
 *                                  direct repeat
 *                                  inverted repeat
 *                                  mirror repeat
 *                              (see: Allison, L. & Edgoose, T. & Dix, T. I.: 
 *                                    Compression of Strings with Approximate Repeats
 *                                    In: Intelligent Systems in Mol. Biol. (1998), pp. 8-16)
 *  @endcode
 *
 *******************************************************************************
 * @b Usage:
 *   @code
 *     tt-analyze [preset] [arguments]
 *   @endcode
 *
 *******************************************************************************
 * @param preset            Zero or one of the setting presets from the list below.
 * @param arguments         Zero or more arguments from the list below.
 *
 *******************************************************************************
 * @b Presets:
 *  @code
 *      markov <markov_order>                           --  Preset to estimate parameters for a 
 *                                                          markov chain
 *                                                          Enables the Entropy module and activates 
 *                                                          all submodules except entropy calculation
 *      dar <dar_process_order>                         --  Preset to estimate parameters for a 
 *                                                          Discrete Autoregressive Process
 *                                                          Enables the Correlation module and disables 
 *                                                          the mutual information submodule
 *      repeatsDna <number_iterations>      \
 *                 <markov_order>           \
 *                 <minimum_hit_length>     \
 *                 <minimum_region_size>    \
 *                 <minimum_relative_probability>     --    Preset to estimate parameters for the repeat 
 *                                                          model (by Allison et. al) on dna sequences
 *                                                          Alphabet is set to [ACGT], A is inverse 
 *                                                          to T and C to G
 *                                                          Character case is ignored
 *                                                          Enables ApproximateRepeats modul and all 
 *                                                          three repeat models
 *  @endcode
 *
 *******************************************************************************
  * @b Arguments:
 *  @code
 *      -o <directory>              -  output_directory (default: stdout)
 *      -s <file>                   -  settings file (default: default settings)
 *      -i <file>                   -  input file (default: read from stdin)
 *                                     NOTE: Because of implementation details -i <file> should always 
 *                                     be preferred to stdin for large files
 *      -id <integer>               -  file id which will be printed in result headers (default: -1)
 *      --stdout                    -  print to stdout (default: only print to stdout if -o <directory> 
 *                                     is not specified)
 *      -Dsection::setting=value    -  set 'value' for 'setting' in 'section'
 *  @endcode
 *
 *******************************************************************************
* @b Examples:
 * - Piping the results from tt-analyze to tt-generator using preset <tt>markov</tt>:
 *   @code
 *     $ tt-analyze markov 5 -i test.txt | tt-generate 500 markov > generated.txt
 *   @endcode
 *
 * - Complex example with settings file, disable approximate repeats module via command line, use 
 *   output directory and stdout, specify file id in result file headers:
 *   @code
 *     $ tt-analyze -s sample_config.ini --stdout -Dmodule_selection::ApproximateRepeats=0  \
 *                  -o ~/result_dir -i dna.fasta -id 123 
 *   @endcode
 *
 **************************************************************************************************
 * @return 0 on success, something else on error
 *
 **************************************************************************************************
 * @b Download:
 * - The newest version of this tool can be downloaded from http://www14.in.tum.de/spp1307/downloads.html
 *
 **************************************************************************************************
 */

#include <climits>
#include <cstdlib>
#include <ctime>
#include <fstream>
#include <iostream>
#include <map>
#include <string>

#define BOOST_THREAD_USE_LIB
#include <boost/filesystem.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/regex.hpp>
#include <boost/thread/thread.hpp>

#include "index_map.h"
#include "register_modules.h"
#include "string_utils.h"

using namespace std;
using namespace papi;

/**
 * Print usage message
 */
void printUsage()
{
    cerr << endl << "USAGE:" <<endl;
    cerr << "tt-analyze [preset] [arguments]" << endl << endl;

    cerr << "ARGUMENTS (all optional):" << endl;
    cerr << "   -s  <file>                -   settings_file (default: default settings)"<<endl;
    cerr << "   -o  <directory>           -   output_directory (default: stdout)" << endl;
    cerr << "   -i  <file>                -   input_file (default: stdin)" << endl;
    cerr << "                                 NOTE: Because of implementation details" << endl;
    cerr << "                                 -i <file> should always be preferred to stdin" << endl;
    cerr << "                                 and piping, especially for large files" << endl;
    cerr << "   -id <number>              -   file_id (default: -1) " << endl;
    cerr << "                                 (only used in result headers)" << endl;
    cerr << "   --stdout                  -   print to stdout (default: only print to stdout"<< endl;
    cerr << "                                 if output directory not set)" << endl;
    cerr << "   -Dsection::setting=value  -   set a value for a setting" << endl << endl;
    cerr << "PRESETS (optional): " << endl;
    cerr << "   markov      <markov_order>" << endl;
    cerr << "   dar         <dar_process_order>" << endl;
    cerr << "   repeatsDna  <number_iterations> <markov_order> <minimum_hit_length>" << endl;
    cerr << "               <minimum_region_size> <minimum_relative_probability>" << endl;
    cerr << "       repeatsDna is configured to work with dna files on the alphabet [ACGT]" << endl << endl;
    cerr << "EXAMPLE 1:" << endl;
    cerr << "Piping the results from tt-analyze to tt-generator using preset 'markov'" <<endl << endl;
    cerr << "   tt-analyze markov 5 -i test.txt | tt-generate 500 markov > generated.txt" << endl << endl;
    cerr << "EXAMPLE 2" << endl;
    cerr << "Settings file, repeats disabled by setting the appropriate setting," << endl;
    cerr << "output to a directory and to stdout, set the id for the result header"<< endl << endl;
    cerr << "   tt-analyze  -s sample.ini --stdout -Dmodule_selection::ApproximateRepeats=0 \\" << endl;
    cerr << "               -o ~/result_dir -i dna.fasta -id 123" << endl << endl << endl;
    cerr << "There are three ways to set settings. If a setting is not specified " << endl;
    cerr << "a default value will be used." << endl<<endl;
    cerr << "1.) Preset: A preset is intended for use with the generator. It specifies" << endl;
    cerr << "a minimal set of settings needed to create the input files for " << endl;
    cerr << "the generator. The preset argument is optional." << endl << endl;
    cerr << "2.) Settings file: An ini file which specifies the settings to be used. " << endl;
    cerr << "For a list of all settings take a look at sample_config.ini." << endl << endl;
    cerr << "3.) -Dsection::setting=value: Sets 'setting' in 'section' to 'value'." << endl << endl;
    cerr << "Precedence: '-Dsection::setting=value' over 'settings file' over 'preset'." << endl << endl;
    cerr << "WARNING:   You must execute modules sequentially if printing to stdout." << endl;
    cerr << "           This means: Do not put them into the same thread group." << endl;
    cerr << "           (see: [module_selection] in the sample_config.ini file). " << endl;
    cerr << "           Otherwise they will print their results simultaneously to stdout!" << endl; 
}


/**
 * Read configuration file and store settings in a map
 * @param[out] settings Map where to put the settings
 * @param[in] iFile_settings Stream to the config file
 */
void initSettings(map<string,map<string,string> > &settings,ifstream &iFile_settings)
{
    string line,part1,part2;
    map<string, string> *current_map = NULL;

    while(!(iFile_settings.eof()))
    {
        getline(iFile_settings, line);
        trimSpaces(line);
        if(!line.empty() && line[0]!='%')
        {
            if(line[0] == '[' && line[line.length()-1]==']')
            {
                part1 = line.substr(1, line.length()-2);
                trimSpaces(part1);
                current_map = &(settings[part1]);
            }
            else 
            {
                int delim = line.find_first_of('=');
                part1= line.substr(0, delim);
                trimSpaces(part1);
                part2 = line.substr(delim+1,line.length());
                trimSpaces(part2);
                (*current_map)[part1] = part2;
            }
        }
    }
}

/**
 * Set settings from command line
 * @param[out] settings Map where to put the settings
 * @param[in] single_settings List of settings which were passed in the command line
 */
void initSettings(map<string,map<string,string> > &settings, vector<pair<string,pair<string,string> > > single_settings) {
    for(size_t i = 0; i< single_settings.size(); ++i) {
        settings[single_settings[i].first][single_settings[i].second.first] = single_settings[i].second.second;
    }
}

/**
 * Set settings from preset
 * @param[out] settings Map where to put the settings
 * @param[in] preset Preset name
 * @param[in] preset_options Preset parameters
 */
void initSettings(map<string,map<string,string> > &settings, string preset, vector<string> preset_options) {
    if(preset.empty()) {
        return;
    }
    if(preset == "markov") {
        if(preset_options.size()!=1) {
            cerr << "Wrong number of parameters for preset: " << preset << endl;
            exit(EXIT_FAILURE);
        }
        settings["module_selection"]["Entropy"] = "1";
        settings["Entropy"]["disable_entropytropytropy"] = "1";        
        settings["Entropy"]["disable_character_distribution"] = "0";        
        settings["Entropy"]["disable_qgram_distribution"] = "0";        
        settings["Entropy"]["qgram_length"] = convertToString(atoi(preset_options[0].c_str())+1) ;        
    } else if(preset == "dar") {
        if(preset_options.size()!=1) {
            cerr << "Wrong number of parameters for preset: " << preset << endl;
            exit(EXIT_FAILURE);
        }
        settings["module_selection"]["Correlation"] = "1";
        settings["Correlation"]["disable_mutual_information"] = "1";
        settings["Correlation"]["disable_autocorrelation_dar"] = "0";
        settings["Correlation"]["disable_character_distribution"] = "0";
        settings["Correlation"]["autocorrelation_dar_max"] = preset_options[0] ;        
    } else if(preset == "repeatsDna") {
        if(preset_options.size()!=5) {
            cerr << "Wrong number of parameters for preset: " << preset << endl;
            exit(EXIT_FAILURE);
        }
        settings["module_selection"]["ApproximateRepeats"] = "1";
        settings["general"]["alphabet"] = "[ACGT]";
        settings["general"]["ignore_case"] = "1";
        settings["ApproximateRepeats"]["markov_order"] = preset_options[0];
        settings["ApproximateRepeats"]["number_of_iterations"] = preset_options[1];
        settings["ApproximateRepeats"]["minimum_hit_length"] = preset_options[2];
        settings["ApproximateRepeats"]["minimum_region_size"] = preset_options[3];
        settings["ApproximateRepeats"]["minimum_relative_probability"] = preset_options[4];
        settings["ApproximateRepeats"]["C"] = "G";
        settings["ApproximateRepeats"]["G"] = "C";
        settings["ApproximateRepeats"]["A"] = "T";
        settings["ApproximateRepeats"]["T"] = "A";
    } else {
        cerr << "Unknown preset: " << preset << endl;
        exit(EXIT_FAILURE);
    }
}

/**
 * Read bool setting value from general settings map and set default value if 
 * setting not already set.
 *
 * @param[in,out] map_settings General settings
 * @param[in] default_value Default value used when setting not defined yet
 *
 * @return Settings value
 */
bool getBool(map<string,string>  &map_settings, string setting, bool default_value) {
    if( map_settings[setting] == "0" || map_settings[setting] == "false") {
        return false;
    } else if(!map_settings[setting].empty()) {
        return true;
    } else {
        map_settings[setting] = default_value ? "1" : "0";
        return default_value;
    }
}

/**
 * Main method
 */
int main (int argc, char * const argv[]) {
    clock_t start,end;
    start = clock();
    
    // Stream to the settings file
    ifstream iFile_settings;
    
    // Stream to the text to analyze
    istream *iFile_text = NULL;
    filebuf fb;
    
    // Settings
    map<string,map<string,string> > map_settings;
    
    //settings from the command line
    vector<pair<string,pair<string,string> > > single_settings;
    
    // Activated modules and their thread group
    list<pair<int,Module*> > list_modules_selected;
    
    // Buffer into which the text is copied (after indexing)
    unsigned char *buffer = NULL;
    
    // Buffer to read stdin
    vector<unsigned char> buffer_stdin;
    
    // read from stdin
    bool read_stdin = true;
    // write results to stdout
    bool write_stdout = false;
    
    // input file length
    long long file_length = 0;
    
    //buffer length
    long long buffer_size = 0;
    
    // absolute path to input file
    string absolute_input_path;
    
    // result directory
    string output_directory;
    
    // path to settings file
    string settings_file;

    // file id for result headers
    long file_id = -1;
    
    // chosen preset
    string preset;
    
    // preset options
    vector<string> preset_options;

    if(argc <2 )
    {
        cerr << "ERROR: Invalid number of arguments" << endl;
        printUsage();
        exit(EXIT_FAILURE);
    }
    

    // read command line arguments
    for(int i=1; i<argc;)
    {
        if( i==1 && argv[i][0]!='-') {
            preset = argv[i];
            ++i;
            while(i<argc && argv[i][0] !='-') {
                preset_options.push_back(argv[i]);
                ++i;
            }
        }
        else if ( strcmp(argv[i],"-s") == 0)
        {
            settings_file = string(argv[i+1]);
            i+=2;
        }
        else if(strcmp(argv[i], "-o") == 0)
        {
            output_directory = string(argv[i+1]);
            i+=2;
        }
        else if(strcmp(argv[i], "-i") == 0) 
        {
            read_stdin = false;
            absolute_input_path = boost::filesystem::system_complete(argv[i+1]).string();
            i+=2;
        }
        else if(strcmp(argv[i], "-id") == 0)
        {
            file_id = atol(argv[i+1]);
            i+=2;
        }
        else if(strcmp(argv[i], "--stdout") == 0) 
        {
            write_stdout = true;
            ++i;
        } else if(strncmp(argv[i], "-D", 2) == 0) {
            string p(argv[i]+2);
            int delim = p.find_first_of("::");
            string section = p.substr(0,delim);
            int delim2 = p.find_first_of('=');
            string key = p.substr(delim+2,delim2-delim-2);
            string val = p.substr(delim2+1,p.size());
            single_settings.push_back(make_pair(section,make_pair(key,val)));
            ++i;
        }
        else {
            cerr << "ERROR: Unkown flag: " << argv[i]<<endl;
            printUsage();
            exit(EXIT_FAILURE);
        }
    }
    
    // Check input
    if(!read_stdin && absolute_input_path.empty())
    {
        cerr << "ERROR: Input file not specified" <<endl;
        printUsage();
        exit(EXIT_FAILURE);
    }
    
    // If no ouput folder specified use stdout
    if(output_directory.empty())
    {
        write_stdout = true;
    }

    // if read_stdin use stdin as input stream
    if(read_stdin)
    {
        iFile_text = &cin;
    }
    // else open file
    else 
    {
        boost::filesystem::path p(absolute_input_path);
        absolute_input_path =  p.string();
        if(!fb.open(absolute_input_path.c_str(),ios::in))
        {
            cerr << "ERROR: Could not open input file "<<absolute_input_path<<endl;
            exit(EXIT_FAILURE);
        }
        iFile_text = new istream(&fb);
    }

    
    // if output folder speficied create directory structure
    if(!output_directory.empty()) {
        if(!boost::filesystem::exists(output_directory.c_str()))
        {
            boost::filesystem::create_directory(output_directory.c_str());
            if(!boost::filesystem::exists(output_directory.c_str()))
            {
                cerr << "ERROR: Could not create directory "<<output_directory<<endl;
                exit(EXIT_FAILURE);
            }
        }
    }
    
    // initialize settings from preset
    initSettings(map_settings, preset, preset_options);
    
    // initialize settings from settings file
    if (!settings_file.empty()) {
        iFile_settings.open(settings_file.c_str());
        if(!iFile_settings)
        {
            cerr << "ERROR: Could not open settings file "<<settings_file<<endl;
            exit(EXIT_FAILURE);
        }
        initSettings(map_settings, iFile_settings);        
        iFile_settings.close();
    }
    
    //initialize settings from command line
    initSettings(map_settings,  single_settings);

    
    // activate selected modules and assign thread group
    map<string, string> *map_selected_modules;
    map_selected_modules = &( map_settings["module_selection"]);
    
    for(int i=0; i<(int)(sizeof(list_modules_all)/sizeof(list_modules_all[0])); ++i)
    {
        if(getBool(*map_selected_modules,(list_modules_all[i])->getId(),false))
        {
            list_modules_selected.push_back(pair<int,Module*>(  atoi(  ((*map_selected_modules)[(list_modules_all[i])->getId()]).c_str()),list_modules_all[i]));
        }
        else
            delete list_modules_all[i];
    }

    list_modules_selected.sort();
    
    // read general settings
    boost::regex re;
    map<string,string> *map_general = &map_settings["general"];

    bool one_whitespace_type = getBool(*map_general,"one_whitespace_type",false);
    bool ignore_case = getBool(*map_general,"ignore_case",false);
    
    bool valid_characters[CHAR_ANZ];
    
    map<string,string>::iterator pref = map_general->find("alphabet");
    
    if(pref!=map_general->end())
    {
        memset(valid_characters,false,sizeof(valid_characters));
        if(ignore_case)
            re = boost::regex (pref->second,boost::regex_constants::normal|boost::regex_constants::icase);
        else
            re = boost::regex (pref->second);
        
        for(int i = CHAR_MIN; i<=CHAR_MAX;++i)
        {
            char c = (char)i;
            if( boost::regex_match(&c, (&c)+1, re))
                valid_characters[c-CHAR_MIN]=true;
        }
    }
    else    
        memset(valid_characters,true,sizeof(valid_characters));
    
    
    pref = map_general->find("skip_characters");

    if(pref!=map_general->end())
    {
        if(ignore_case)
            re = boost::regex (pref->second,boost::regex_constants::normal|boost::regex_constants::icase);
        else
            re = boost::regex (pref->second);
        
        for(int i = CHAR_MIN; i<=CHAR_MAX;++i)
        {
            char c = (char)i;
            if( boost::regex_match(&c, (&c)+1, re))
                valid_characters[c-CHAR_MIN]=false;
        }
    }
    
    if(getBool(*map_general,"skip_non_print_chars",false))
    {
        for(int i = CHAR_MIN; i<=CHAR_MAX;++i)
        {
            char c = (char)i;
            if( !isprint(c))
                valid_characters[c-CHAR_MIN]=false;
        }
    }
    
    if(getBool(*map_general, "skip_first_line", false)) {
        iFile_text->ignore(UINT_MAX,'\n');
    }
    
    IndexMap<char> ind_map(CHAR_MIN,CHAR_MAX);

    cerr << "Started copying file into buffer"<<endl;
    
    // determine file length
    if(!read_stdin)
    {
        iFile_text->seekg (0, ios::end);    
        file_length = iFile_text->tellg();
        iFile_text->seekg (0, ios::beg);
        buffer = new unsigned char[file_length];
    }
    
    // copy file to buffer using indices instead of characters
    while(iFile_text->good())
    {
        char c;
        iFile_text->get(c);
        if(iFile_text->good())
        {
            if(ignore_case)
                c = toupper(c);
            
            if(one_whitespace_type && isspace(c))
                c=' ';

            if(valid_characters[c-CHAR_MIN])
            {
                
                if(!read_stdin)
                    buffer[buffer_size++] = (unsigned char)ind_map.addIndex(c);
                else
                    buffer_stdin.push_back((unsigned char)ind_map.addIndex(c));
            }
            if(read_stdin)
                ++file_length;
        }
    }
    
    // resize buffer if reading from stdin
    if(read_stdin)
    {

        buffer_size = buffer_stdin.size();
        buffer_stdin.resize(buffer_size);
        buffer = &buffer_stdin[0];
    }
    cerr << "Finished copying file into buffer "<<endl;
        
    
    // start modules in order
    
    boost::thread_group *proc_threads =  new boost::thread_group();
        
    for(list<pair<int,Module*> >::iterator it = list_modules_selected.begin(); it != list_modules_selected.end();)
    {
        int group_id = it->first;
        cerr << "Starting module group "<< group_id << endl;
        do
        {
            proc_threads->create_thread(module_thread(it->second,file_length,buffer_size,buffer,new AnalyzeSetting( &map_settings["general"],&map_settings[((it->second))->getId()]), &output_directory, write_stdout, file_id, &absolute_input_path, &ind_map));
        }while( ++it!= list_modules_selected.end() && it->first == group_id);
        proc_threads->join_all();
        for(list<pair<int,Module*> >::iterator it2 = list_modules_selected.begin(); it2!=it; it2 = list_modules_selected.erase(it2))
        {
            delete it2->second;
        }
    }

    // write settings file
    if(!output_directory.empty()) {
        ofstream oFile_settings;
        oFile_settings.open((output_directory+"/settings.ini").c_str(), ios::trunc);
        
        if(oFile_settings)
        {
            string line;
            time_t t;
            time(&t);
            oFile_settings << "% Date: " << ctime(&t);
            oFile_settings << "% Absoulte file path: " << absolute_input_path <<endl;
            oFile_settings << "% File-ID: " << file_id <<endl;
            oFile_settings << "% File length: " << file_length <<endl;
            oFile_settings << "% Command line: ";
            for(int i=0; i<argc; ++i) {
                oFile_settings << '"' << argv[i] << "\" ";
            }
            oFile_settings <<endl;
            
            for(map<string,map<string,string> >::iterator it1 = map_settings.begin();
                it1 != map_settings.end();
                ++it1) {
                oFile_settings << "[" << it1->first << "]" << endl;
                for(map<string,string>::iterator it2 = it1->second.begin();
                    it2 != it1->second.end();
                    ++it2) {
                    oFile_settings << it2->first << " = " << it2->second << endl;                    
                }
            }
            
            oFile_settings.close();
        }
        else {
            cerr<<"ERROR: Cant write file "<<output_directory<<"/settings.ini"<<endl;
            exit(EXIT_FAILURE);
        }
    }    
    

    
    if(proc_threads)
        delete proc_threads;
    
    if(buffer)
        delete buffer;
    
    if(!read_stdin) {
        fb.close();
        if(iFile_text) {
            delete iFile_text;
        }
    }

    end = clock();
    
    cerr<< "Execution time: "<<(double)(end-start)/CLOCKS_PER_SEC<<endl;
    
    exit(EXIT_SUCCESS);
}

