#ifndef _markov_h_
#define _markov_h_

#include <cmath>
#include <fstream>
#include <iostream>
#include <string>

#include "csv_in_stream.h"
#include "format_helper_in.h"
#include "generator.h"
#include "index_map.h"
#include "matrix.h"
#include "queue_fixed_size.h"
#include "random.h"

using namespace std;

namespace papi
{
    /**
     * @class Markov
     * Markov chain generator
     */
    class Markov: public Generator
    {
    public:
        /// Overrides Generator::generate
        void generate( long long length,  map<string,string> &parameters);
        /**
         * Get next character. Calls generateCharacter(-1).
         * @return The next character or -1 on error
         */
        int generateCharacter();
        
        /**
         * Generating single character with the possibitlity to exclude a single
         * character. If no character should be excluded pass -1.
         * @param[in] excludedCharacter The character to exclude or -1 if none.
         * @return Next character or -1 on error
         */
        int generateCharacter(int excludedCharacter);
        
        /**
         * Get probability for a character that it will be generated next
         * @param[in] c The character
         */
        double getProbability(char c);

        /**
         * Update the state of the chain with the last generated character
         * @param[in] c The last generated character
         */
        void updateHistory(char c);
        
        /**
         * Initialize markov chain parameters.
         * Calls preInit, starts the parser and calls postInit.
         * @param[in] file_character_prob Path to the character distribution file
         * @param[in] file_cond_prob Path to the qgram distribution file
         */
        void init(const char *file_character_prob,const char *file_cond_prob);
        
        /**
         * Initialize the parameter parser.
         * @param[in] parser The parameter parser
         * @param[in] file_character_prob Path to the character distribution file
         * @param[in] file_cond_prob Path to the qgram distribution file
         * @param[out] start_word Start word chosen by the parser.
         */
        void preInit(ParameterParser &parser, 
                     const char *file_character_prob,
                     const char *file_cond_prob,
                     char *&start_word);
        
        /**
         * Update the markov chain state with the start word.
         * @param[in] start_word The start word (first k characters)
         */
        void postInit(const char *start_word);
        
        /**
         * Initialize a zero order markov chain with uniform distribution.
         * @param[in] alphabet All symbols of the alphabet as a string.
         */
        void initUniform(string alphabet);
        
        /**
         * Constructor.
         */
        Markov();
        
        /**
         * Destructor.
         */
        ~Markov();
    protected:
        ///Character distribution
        double *character_prob_rand_interval;
        ///Index map
        IndexMap<char> ind;
        ///Last k generated characters
        QueueFixedSizeRandomAccess<int> *last_k;
        ///Conditional probabilities
        NDimMat<double *> *conditional_probability;
        ///Order of the markov chain
        int cp_order;
        /**
         * Generate sequence
         * @param[in] length Length of the sequence
         */
        void generateSequence(long long length);
        

        /// Flag indicating whether this is a zero order uniformaly distributed markov chain.
        bool uniform;
        
        ///Weighted die
        Die die;
    };
	
}

#endif

