#include "approximate_repeats.h"

using namespace std;


namespace papi
{

    ApproximateRepeats::ApproximateRepeats():
        disable_direct_repeat(false),disable_mirror_repeat(false), disable_inverted_repeat(false),  randomDouble(0,1)
    {
    }

    ApproximateRepeats::~ApproximateRepeats()
    {
        if(buffer)
            delete [] buffer;
        for(list<RepeatGenerator*>::iterator it = generators.begin();it!=generators.end();++it)
            delete *it;
    }


    void ApproximateRepeats::generate(long long length,  map<string,string> &parameters)
    {
        const char *file_prob = NULL;
        const char *file_cond_prob = NULL;
        const char *file_direct_repeat = NULL;
        const char *file_inverted_repeat = NULL;
        const char *file_mirror_repeat = NULL;
        
        buffer = new char[length];
        
        map<string,string>::iterator it;
        
        if( (it = parameters.find("character_distribution"))!=parameters.end()) {
            file_prob = it->second.c_str();
        }

        if( (it = parameters.find("qgram_distribution"))!=parameters.end()) {
            file_cond_prob = it->second.c_str();
        }
        
        if( (it = parameters.find("direct_repeat"))!=parameters.end()) {
            file_direct_repeat = it->second.c_str();
        }

        if( (it = parameters.find("mirror_repeat"))!=parameters.end()) {
            file_mirror_repeat = it->second.c_str();
        }

        if( (it = parameters.find("inverted_repeat"))!=parameters.end()) {
            file_inverted_repeat = it->second.c_str();
        }

        init(file_direct_repeat, file_mirror_repeat, file_inverted_repeat, file_prob,file_cond_prob);
        
        generateSequence(length);

        
    }

    void ApproximateRepeats::generateSequence(long long length)
    {
        long long l = 0;
        
        while(l<length)
        {
            double p = randomDouble.generate();
            list<RepeatGenerator*>::iterator it;
            for(it = generators.begin(); it!=generators.end() && (*it)->prob_start<=p; p-=(*it++)->prob_start);
            if(l==0 || it == generators.end())
            {
                buffer[l] = (char)markov_model.generateCharacter();
                markov_model.updateHistory(buffer[l++]);
            }
            else {
                RepeatGenerator* g = *it;
                g->startGenerator(l);
                do
                {
                    l+=g->performOperation();
                }while(l<length && !g->finished);
            }

        }
        
        write(buffer, length);
    }


    void ApproximateRepeats::init(const char *file_direct_repeat,  
                                  const char *file_mirror_repeat,
                                  const char *file_inverted_repeat,
                                  const char *file_prob,
                                  const char *file_cond_prob)
    {
        ParameterParser parser;
        char *start_word_markov;
        markov_model.preInit(parser, file_prob, file_cond_prob, start_word_markov);
        double direct_prob_start = -1, direct_prob_end = -1, direct_prob_match = -1, 
                direct_prob_change = -1, direct_prob_insert = -1, direct_prob_delete = -1;
        double inverted_prob_start = -1, inverted_prob_end = -1, inverted_prob_match = -1, 
                inverted_prob_change = -1, inverted_prob_insert = -1, inverted_prob_delete = -1;
        double mirror_prob_start = -1, mirror_prob_end = -1, mirror_prob_match = -1, 
                mirror_prob_change = -1, mirror_prob_insert = -1, mirror_prob_delete = -1;
        char *complement;
        
        parser.requestRepeatMachine(file_inverted_repeat, "inverted_repeat",
                                    inverted_prob_start, inverted_prob_end,
                                    inverted_prob_match, inverted_prob_change,
                                    inverted_prob_insert, inverted_prob_delete,
                                    &complement);
        
        parser.requestRepeatMachine(file_direct_repeat, "direct_repeat",
                                    direct_prob_start, direct_prob_end,
                                    direct_prob_match, direct_prob_change,
                                    direct_prob_insert, direct_prob_delete,
                                    NULL);
        parser.requestRepeatMachine(file_mirror_repeat, "mirror_repeat",
                                    mirror_prob_start, mirror_prob_end,
                                    mirror_prob_match, mirror_prob_change,
                                    mirror_prob_insert, mirror_prob_delete,
                                    NULL);
       
        parser.start();
        
        markov_model.postInit(start_word_markov);
        
        if(direct_prob_start == -1) {
            disable_direct_repeat = true;
        } else {
            generators.push_back(new ForwardRepeatGenerator(buffer, 
                                                            &markov_model, 
                                                            direct_prob_start,
                                                            direct_prob_end,
                                                            direct_prob_match,
                                                            direct_prob_change,
                                                            direct_prob_insert,
                                                            direct_prob_delete));            
        }
        
        if(mirror_prob_start == -1) {
            disable_mirror_repeat = true;
        } else {
            generators.push_back(new MirrorRepeatGenerator(buffer, 
                                                           &markov_model, 
                                                           mirror_prob_start,
                                                           mirror_prob_end,
                                                           mirror_prob_match,
                                                           mirror_prob_change,
                                                           mirror_prob_insert,
                                                           mirror_prob_delete));            
        }
        if(inverted_prob_start == -1) {
            disable_inverted_repeat = true;
        } else {
            generators.push_back(new InvertedRepeatGenerator(buffer, 
                                                             &markov_model, 
                                                             inverted_prob_start,
                                                             inverted_prob_end,
                                                             inverted_prob_match,
                                                             inverted_prob_change,
                                                             inverted_prob_insert,
                                                             inverted_prob_delete,
                                                             complement));            
        }

    }

    ApproximateRepeats::RepeatGenerator::RepeatGenerator(char * buffer, Markov *markov_model, double prob_start, double prob_end, double prob_match, double prob_change, double prob_insert, double prob_delete):
    finished(false), prob_start(prob_start), randomDouble(0,1),buffer(buffer),markov_model(markov_model)
    {
        prob_continue_end[CONTINUE] = 1-prob_end;
        prob_continue_end[END] = prob_end;
        
        prob_delete_nodelete[DELETE] = prob_delete;
        prob_delete_nodelete[NODELETE] = 1-prob_delete;
        
        prob_match_change_insert[MATCH-1] = prob_match;
        prob_match_change_insert[CHANGE-1] = prob_change;
        prob_match_change_insert[INSERT-1] = prob_insert;
    }

    
    void ApproximateRepeats::RepeatGenerator::startGenerator(long long cur_pos)
    {
        pos_buffer = cur_pos;
        pos_repeat = (long long)(randomDouble.generate()*cur_pos);
        finished = false;
        empty = true;
    }

    
   
    
    ApproximateRepeats::ForwardRepeatGenerator::ForwardRepeatGenerator(char * buffer, Markov *markov_model, double prob_start, double prob_end, double prob_match, double prob_change, double prob_insert, double prob_delete):
    RepeatGenerator(buffer,markov_model,prob_start, prob_end, prob_match, prob_change, prob_insert, prob_delete)
    {}



    int ApproximateRepeats::ForwardRepeatGenerator::performOperation()
    {
        
        
        if(finished)
            return -1;
        
        int operation = DELETE;
        if(pos_repeat+1 >= pos_buffer || die.roll(2,prob_delete_nodelete) == NODELETE)
        {
            if(markov_model->getProbability(buffer[pos_repeat]) > 0.9999)
                operation = MATCH;
            else
                operation = die.roll(NUM_OPERATIONS-1,prob_match_change_insert)+1;
        }
        if(operation < 0 )
        {
            cerr << "ERROR: ForwardRepeatGenerator: Unable to choose next operation at position "<< pos_buffer << "(repeat position " <<pos_repeat<< ")" <<endl;
            exit(EXIT_FAILURE);
        }
        switch(operation)
        {
            case MATCH:
                buffer[pos_buffer] = buffer[pos_repeat];
                ++pos_repeat;
                empty = false;
                break;
            case CHANGE:
                buffer[pos_buffer] = (char)markov_model->generateCharacter(buffer[pos_repeat]);
                ++pos_repeat;
                empty = false;
                break;
            case INSERT:
                buffer[pos_buffer] =(char)markov_model->generateCharacter();
                empty = false;
                break;
            case DELETE:
                ++pos_repeat;
                break;
        }

        if(!empty && die.roll(2, prob_continue_end) == END)
            finished = true;

        if(operation!=DELETE)
        {
            markov_model->updateHistory(buffer[pos_buffer++]);
            return 1;
        }
        
        
        return 0;
    }
    
    ApproximateRepeats::MirrorRepeatGenerator::MirrorRepeatGenerator(char * buffer, Markov *markov_model, double prob_start, double prob_end, double prob_match, double prob_change, double prob_insert, double prob_delete):
    RepeatGenerator(buffer,markov_model,prob_start, prob_end, prob_match, prob_change, prob_insert, prob_delete)
    {}


    int ApproximateRepeats::MirrorRepeatGenerator::performOperation()
    {
        if(finished)
            return -1;

        int operation = DELETE;
        if(pos_repeat == 0 || die.roll(2,prob_delete_nodelete) == NODELETE)
        {
            if(markov_model->getProbability(buffer[pos_repeat]) > 0.9999)
                operation = MATCH;
            else
                operation = die.roll(NUM_OPERATIONS-1,prob_match_change_insert)+1;
        }
        if(operation < 0 )
        {
            cerr << "ERROR: MirrorRepeatGenerator: Unable to choose next operation at position "<< pos_buffer << "(repeat position " <<pos_repeat<< ")" <<endl;
            exit(EXIT_FAILURE);
        }
        switch(operation)
        {
            case MATCH:
                buffer[pos_buffer] = buffer[pos_repeat];
                --pos_repeat;
                empty = false;
                break;
            case CHANGE:
                buffer[pos_buffer] = (char)markov_model->generateCharacter(buffer[pos_repeat]);
                --pos_repeat;
                empty = false;
                break;
            case INSERT:
                buffer[pos_buffer] = (char)markov_model->generateCharacter();
                empty = false;
                break;
            case DELETE:
                --pos_repeat;
                break;
        }

        if(!empty && (pos_repeat<0 || die.roll(2, prob_continue_end) == END))
            finished = true;

        
        if(operation!=DELETE)
        {

            markov_model->updateHistory(buffer[pos_buffer++]);

            return 1;
        }
        
        return 0;
    }
    
    ApproximateRepeats::InvertedRepeatGenerator::InvertedRepeatGenerator(char * buffer, Markov *markov_model, double prob_start, double prob_end, double prob_match, double prob_change, double prob_insert, double prob_delete, char *complement):
    RepeatGenerator(buffer,markov_model,prob_start, prob_end, prob_match, prob_change, prob_insert, prob_delete), complement(complement)
    {}


    int ApproximateRepeats::InvertedRepeatGenerator::performOperation()
    {
        if(finished)
            return -1;
        int operation = DELETE;
        if(pos_repeat == 0 || die.roll(2,prob_delete_nodelete) == NODELETE)
        {
            if(markov_model->getProbability(complement[(unsigned char)buffer[pos_repeat]]) > 0.9999)
                operation = MATCH;
            else
                operation = die.roll(NUM_OPERATIONS-1,prob_match_change_insert)+1;
        }
        if(operation < 0 )
        {
            cerr << "ERROR: InvertedRepeatGenerator: Unable to choose next operation at position "<< pos_buffer << "(repeat position " <<pos_repeat<< ")" <<endl;
            exit(EXIT_FAILURE);
        }
        switch(operation)
        {
            case MATCH:
                buffer[pos_buffer] = complement[(unsigned char)buffer[pos_repeat]];
                --pos_repeat;
                empty = false;
                break;
            case CHANGE:
                buffer[pos_buffer] = (char)markov_model->generateCharacter(complement[(unsigned char)buffer[pos_repeat]]);
                --pos_repeat;
                empty = false;
                break;
            case INSERT:
                buffer[pos_buffer] = (char)markov_model->generateCharacter();
                empty= false;
                break;
            case DELETE:
                --pos_repeat;
                break;
        }
        
        if(!empty && (pos_repeat<0 || die.roll(2, prob_continue_end) == END))
            finished = true;
        
        
        if(operation!=DELETE)
        {
            markov_model->updateHistory(buffer[pos_buffer++]);

            return 1;
        }
        
        return 0;
    }

}

