#ifndef _repeats_h_
#define _repeats_h_

#include <cmath>
#include <cstdlib>
#include <ctime>
#include <fstream>
#include <iostream>
#include <list>
#include <string>

#include "format_helper_in.h"
#include "generator.h"
#include "markov/markov.h"
#include "string_utils.h"

#define CONTINUE 0
#define END 1
#define DELETE 0
#define NODELETE 1
#define MATCH 1
#define CHANGE 2
#define INSERT 3
#define NUM_OPERATIONS 4
#define DIRECT_REPEAT 0
#define MIRROR_REPEAT 1
#define INVERTED_REPEAT 2

using namespace std;


namespace papi
{
    /**
     * @class ApproximateRepeats
     * Approximate repeats generator according to Allison et al.(1998): Compression of Strings with Approximate Repeats
     */
    class ApproximateRepeats: public Generator
    {
    public:
        /// Overrides Generator::generate
        void generate(long long length,  map<string,string> &parameters);
        /**
         * Initialize generator.
         * @param[in] file_direct_repeat Path to direct repeat parameter file
         * @param[in] file_mirror_repeat Path to mirror repeat parameter file
         * @param[in] file_inverted_repeat Path to inverted repeat parameter file
         * @param[in] file_prob Stream zur Path to character distribution file
         * @param[in] file_cond_prob Path to qgram distribution file
         */
        void init(const char *file_direct_repeat,  
                  const char *file_mirror_repeat,
                  const char *file_inverted_repeat,
                  const char *file_prob,
                  const char *file_cond_prob);
        /**
         * Constructor.
         */
        ApproximateRepeats();
        /**
         * Destructor.
         */
        ~ApproximateRepeats();
   
    protected:
        /**
         * Base class of a repeat machine.
         */
        class RepeatGenerator
        {
        public:
            /**
             * Perform transition within repeat graph
             * @return Number of generated characters
             */
            virtual int performOperation() = 0;
            
            /**
             * Start new repeat
             * @param[in] cur_pos Start position of the repeat
             */
            void startGenerator(long long cur_pos);
            
            /**
             * Constructor
             * @param buffer The text buffer
             * @param markov_model Underlying markov chain
             * @param prob_start Probability for a repeat start
             * @param prob_end Probability for a repeat end
             * @param prob_match Probability for a match operation within the repeat
             * @param prob_change Probability for a change operation within the repeat
             * @param prob_insert Probability for an insert operation within the repeat
             * @param prob_delete Probability for a delete operation within the repeat
             */
            RepeatGenerator(char * buffer, Markov *markov_model, double prob_start, double prob_end, double prob_match, double prob_change, double prob_insert, double prob_delete);
            
            /**
             * Virtual destructor
             */
            virtual ~RepeatGenerator(){}

            /// True, if current repeat has finished
            bool finished;

            /// Probability for a repeat start
            double prob_start;
            
        protected:
            /// Weighted die
            Die die;
            /// Random number generator
            RandomDouble randomDouble;
            /// Distribution: continue repeat and end repeat
            double prob_continue_end[2];
            /// Distribution: deletion and no deletion
            double prob_delete_nodelete[2];
            /// Distribution of operation 'no deletion': match, change insert
            double prob_match_change_insert[NUM_OPERATIONS-1];
            /// Position of the repeat from which to copy
            long long pos_repeat;
            /// Current position at which to write
            long long pos_buffer;
            /// The buffer
            char *buffer;
            /// Underlying markov model
            Markov *markov_model;
            /// true, if last repeat was empty (only delete operations)
            bool empty;
            
            
        };
        
        /**
         * @class ForwardRepeatGenerator
         * Generator for direct repeats
         */
        class ForwardRepeatGenerator: public RepeatGenerator
        {
        public:
            /// Overrides RepeatGenerator::performOperation()
            int performOperation();
            /**
             * Constructor
             * @param buffer The text buffer
             * @param markov_model Underlying markov chain
             * @param prob_start Probability for a repeat start
             * @param prob_end Probability for a repeat end
             * @param prob_match Probability for a match operation within the repeat
             * @param prob_change Probability for a change operation within the repeat
             * @param prob_insert Probability for an insert operation within the repeat
             * @param prob_delete Probability for a delete operation within the repeat
             */
            ForwardRepeatGenerator(char * buffer, Markov *markov_model, double prob_start, double prob_end, double prob_match, double prob_change, double prob_insert, double prob_delete);
            
        };
        ///Generiert umgekehrte Repeats
        class MirrorRepeatGenerator: public RepeatGenerator
        {
        public:
            /// Overrides RepeatGenerator::performOperation()
            int performOperation();
            /**
             * Constructor
             * @param buffer The text buffer
             * @param markov_model Underlying markov chain
             * @param prob_start Probability for a repeat start
             * @param prob_end Probability for a repeat end
             * @param prob_match Probability for a match operation within the repeat
             * @param prob_change Probability for a change operation within the repeat
             * @param prob_insert Probability for an insert operation within the repeat
             * @param prob_delete Probability for a delete operation within the repeat
             */
            MirrorRepeatGenerator(char * buffer, Markov *markov_model, double prob_start, double prob_end, double prob_match, double prob_change, double prob_insert, double prob_delete);
        };
        
        ///Generiert umgedrehte komplementäre Repeats
        class InvertedRepeatGenerator: public RepeatGenerator
        {
        public:
            ~InvertedRepeatGenerator()
            {
                if(complement)
                    delete [] complement;
            }
            /// Overrides RepeatGenerator::performOperation()
            int performOperation();
            /**
             * Constructor
             * @param buffer The text buffer
             * @param markov_model Underlying markov chain
             * @param prob_start Probability for a repeat start
             * @param prob_end Probability for a repeat end
             * @param prob_match Probability for a match operation within the repeat
             * @param prob_change Probability for a change operation within the repeat
             * @param prob_insert Probability for an insert operation within the repeat
             * @param prob_delete Probability for a delete operation within the repeat
             * @param complement Field containing the complement for each character
             */
            InvertedRepeatGenerator(char * buffer, Markov *markov_model, double prob_start, double prob_end, double prob_match, double prob_change, double prob_insert, double prob_delete, char *complement);
        protected:
            /// Field containing the complement for each character
            char *complement;
            
        };

        
        /// Generate sequence
        /**
         * @param[in] length Length of the sequence
         */
        void generateSequence(long long length);
        
        /// The text
        char *buffer;
        
        /// Underlying markov model
        Markov markov_model;
        
        /// True, if direct repeats are disabled
        bool disable_direct_repeat;
        /// True, if mirror repeats are disabled
        bool disable_mirror_repeat;
        /// True, if inverted repeats are disabled
        bool disable_inverted_repeat;
        /// List of all active repeat machines
        list<RepeatGenerator *> generators;
        /// random number generator
        RandomDouble randomDouble;
        /// weighted die
        Die die;
        
        
    };

	
}

#endif

