#ifndef _analyze_setting_h_
#define _analyze_setting_h_

#include <cstdlib>
#include <iostream>
#include <map>
#include <string>

#include "string_utils.h"


namespace papi
{
    /**
     * @struct AnalyzeSetting
     *
     * Allows convenient access to module settings
     */
    struct AnalyzeSetting
    {
        /// general settings
        std::map<std::string,std::string> *settings_general;
        /// module specific settings
        std::map<std::string,std::string> *settings_module;
        
        /**
         * Constructor
         *
         * @param[in,out] settings_general General settings
         * @param[in,out] settings_module Module specific settings
         */
        AnalyzeSetting(std::map<std::string,std::string> *settings_general, std::map<std::string,std::string> *settings_module)
        : settings_general(settings_general),settings_module(settings_module)
        {
        }
        
        /**
         * Try to find setting and return value formatted as a char.
         * If setting is not set, use default value and set default value in the map
         *
         * @param[in] setting Name of the setting
         * @param[in] default_value Default value if setting is not set
         * @return The setting value
         */
        char getChar(std::string setting, char default_value);
        /**
         * Try to find setting and return value formatted as a bool.
         * If setting is not set, use default value and set default value in the map
         *
         * @param[in] setting Name of the setting
         * @param[in] default_value Default value if setting is not set
         * @return The setting value
         */
        bool getBool(std::string setting, bool default_value);
        /**
         * Try to find setting and return value formatted as a int.
         * If setting is not set, use default value and set default value in the map
         *
         * @param[in] setting Name of the setting
         * @param[in] default_value Default value if setting is not set
         * @return The setting value
         */
        int getInt(std::string setting, int default_value);
        /**
         * Try to find setting and return value formatted as a double.
         * If setting is not set, use default value and set default value in the map
         *
         * @param[in] setting Name of the setting
         * @param[in] default_value Default value if setting is not set
         * @return The setting value
         */
        double getDouble(std::string setting, double default_value);
        /**
         * Try to find setting and return value formatted as a long.
         * If setting is not set, use default value and set default value in the map
         *
         * @param[in] setting Name of the setting
         * @param[in] default_value Default value if setting is not set
         * @return The setting value
         */
        long int getLong(std::string setting, long int default_value);
        /**
         * Try to find setting and return value formatted as a string.
         * If setting is not set, use default value and set default value in the map
         *
         * @param[in] setting Name of the setting
         * @param[in] default_value Default value if setting is not set
         * @return The setting value
         */
        std::string getString(std::string setting, std::string default_value);
    };

}

#endif

