#ifndef _csv_out_stream_h_
#define _csv_out_stream_h_

#include <fstream>
#include <iostream>
#include <list>
#include <locale>
#include <sstream>
#include <string>

#include "string_utils.h"

#define COMMA true
#define POINT false

using namespace std;

namespace papi
{

    /**
     * @class CommaAsDecimalPoint
     * 
     * Derived from std::numpunct_byname, implementing a comma as a decimal separator
     */
    class CommaAsDecimalPoint : public std::numpunct_byname<char>
    {
    public:
        /**
         * Constructor
         *
         * @param[in] name
         */
        CommaAsDecimalPoint(char const* name)
        : std::numpunct_byname<char>(name)
        {}
    protected:
        /// Use a comma as a decimal point
        char do_decimal_point() const
        {
            return ',';
        }
    }; 


    /**
     * @class CsvOutStream
     *
     * Class to faciliate the creation of csv formatted files
     */
    class CsvOutStream
    {
    protected:	
        /// index of the last written column
        int last_col;
        /// locale object used for stream formatting
        locale l;
        /**
         * Writes formatted data into the file
         *
         * @param[in] data The data to be written
         */
        template<class T>
        void write(T data) {
            for(list<ostream*>::iterator it = streams.begin();it!=streams.end();++it) {
                ostream *stream = *it;
                (*stream) << data;
            }
        }
        
    public:
        /// List of outpit stream to write to
        list<ostream*> streams;
        
        /**
         * Constructor
         * @param decimalPoint #COMMA or #POINT as decimal separator
         */
        CsvOutStream(bool decimalPoint);
        
        /**
         * Constructor. Does nothing. #init must be called before first usage
         */
        CsvOutStream();
        
        
        /**
         * Add stream to the list of output streams.
         *
         * @param[in] s The stream 
         */
        void addStream(ostream *s);
        
        /**
         * (Re-)Initialize csv output
         *
         * @param decimalPoint #COMMA or #POINT as decimal separator
         */
        void init(bool decimalPoint);

    
        /**
         * Get string representation of character
         * Special characters may be formatted before printing them.
         * 
         * @param[in] c The character
         * @return String representation of the character
         */
        string getFormattedChar(char c);
        
        /**
         * Write C-string to streams. Special characters will be escaped properly.
         *
         * @param[in] s C-string
         */
        void printFormatted(char const* s);
        
        
        /**
         * Write string to streams. Special characters will be escaped properly.
         *
         * @param[in] s The string
         */
        void printFormatted(string s);
        
        /**
         * Write character to streams. Special characters will be escaped properly.
         *
         * @param[in] c The character
         */
        void printFormatted(char c);
        
        /**
         * Fill the current line with empty cells and begin new line.
         */
        void newline();
        
        /**
         * Write emtpy cells.
         * @param[in] num Number of empty cells to write
         */
        void skipCell(int num);
        
        /**
         * Write formatted string and begin new cell 
         *
         * @param[in] s The string
         */
        void addCell(string s);
        /**
         * Write formatted C-string and begin new cell 
         *
         * @param[in] s The C-string
         */
        void addCell(char const * s);
        /**
         * Write formatted char and begin new cell 
         *
         * @param[in] c The character
         */
        void addCell(char c);
        
        /**
         * Write formatted data and begin new cell 
         *
         * @param[in] data The data
         */
        template<class T>
        void addCell(T data)
        {
            ++last_col;
            if(last_col > 0)
                write(";");
            
            write(data);
        }
        
        
    };

}

#endif

