#ifndef _format_helper_out_h_
#define _format_helper_out_h_

#include <fstream>
#include <iostream>
#include <string>

#include "csv_out_stream.h"
#include "hash_map.h"
#include "index_map.h"


namespace papi {
    /**
     * Initialize csv out stream
     *
     * @param[out] csv the csv stream
     * @param[out] oFile the underlying file stream
     * @param[in] dir the output directory
     * @param[in] write_stdout True, if result should be printed to stdout
     * @param[in] prefix The module prefix for the output file name
     * @param[in] name Base file name
     */
    void initCsvOut(CsvOutStream &csv, ofstream &oFile, 
                    string &dir, bool write_stdout, 
                    const string &prefix, const char *name);
    
    /**
     * Print csv header.
     *
     * @param[in] csv The csv out stream
     * @param[in] file_id File id which will be printed in the header
     * @param[in] file_path Path to the file which was analyzed
     * @param[in] file_length Length of the analyzed file
     * @param[in] type_identifier Result type to which content_type will be set (important for generator)
     */
    void printCsvHeader(CsvOutStream &csv, long file_id, string file_path, 
                        long long file_length, string type_identifier);
    
    /**
     * Print character distribution file
     *
     * @param[in] dir Output directory
     * @param[in] write_stdout True, if write to stdout
     * @param[in] prefix Module name prefix for result files
     * @param[in] file_id File id which will be printed in the header
     * @param[in] file_path Path to the file which was analyzed
     * @param[in] file_length Length of the analyzed file
     * @param[in] ind_map The index amp
     * @param[in] distribution Distribution of the characters
     */
   template <class T>
    void printCsvCharacterDistribution(string &dir, bool write_stdout, const string &prefix, 
                                      long file_id, string &file_path, long long file_length,
                                      const IndexMap<char> *ind_map, T *distribution) {
        CsvOutStream csv(POINT);
        ofstream oFile;
        
        initCsvOut(csv, oFile, dir, write_stdout, prefix, "character_distribution");
        
        printCsvHeader(csv,file_id, file_path, file_length,"character_distribution");
        csv.addCell("alphabet_size");
        csv.addCell(ind_map->size);
        csv.newline();
        csv.addCell("symbol");
        csv.addCell("relative frequency density");
        csv.newline();           
        for(int i=0;i<ind_map->size;++i)
        {
            csv.addCell(ind_map->getValue(i));
            csv.addCell(distribution[i]);
            csv.newline();
        }
        if(oFile.is_open()) {
            oFile.close();
        }        
    }
    
    
    /**
     * Print qgram distribution file header
     *
     * @param[in] csv Csv output stream
     * @param[in] oFile Underlying output file stream
     * @param[in] dir Output directory
     * @param[in] write_stdout True, if write to stdout
     * @param[in] prefix Module name prefix for result files
     * @param[in] file_id File id which will be printed in the header
     * @param[in] file_path Path to the file which was analyzed
     * @param[in] file_length Length of the analyzed file
     * @param[in] qgram_length Length of qgrams
     * @param[in] alphabet_size Alphabet size
     */
    void printCsvQgramDistributionHeader(CsvOutStream &csv,ofstream &oFile,
                                         string &dir, bool write_stdout, const string &prefix, 
                                         long file_id, string &file_path, long long file_length,
                                         int qgram_length, int alphabet_size );
    /**
     * Print autocorrelation parameters for dar process
     *
     * @param[in] dir Output directory
     * @param[in] write_stdout True, if write to stdout
     * @param[in] prefix Module name prefix for result files
     * @param[in] file_id File id which will be printed in the header
     * @param[in] file_path Path to the file which was analyzed
     * @param[in] file_length Length of the analyzed file
     * @param[in] autocorrelation_dar The autocorrelation parameters
     * @param[in] autocorrelation_dar_max Order of the dar process
     */
    void printCsvAutocorrelationDar(string &dir, bool write_stdout, const string &prefix, 
                                    long file_id, string &file_path, long long file_length,
                                    long double *autocorrelation_dar, int autocorrelation_dar_max);
    
    
    /**
     * Print mutual information function file
     *
     * @param[in] dir Output directory
     * @param[in] write_stdout True, if write to stdout
     * @param[in] prefix Module name prefix for result files
     * @param[in] file_id File id which will be printed in the header
     * @param[in] file_path Path to the file which was analyzed
     * @param[in] file_length Length of the analyzed file
     * @param[in] mutual_information Mutual information function
     * @param[in] mutual_information_min Minimal position at which the function is evaluated
     * @param[in] mutual_information_max Maximal position at which the function is evaluated
     * @param[in] mutual_information_step Distance between evaluated positions
     */
    void printCsvMutualInformation(string &dir, bool write_stdout, const string &prefix, 
                                   long file_id, string &file_path, long long file_length,
                                   long double *mutual_information, int mutual_information_min, 
                                   int mutual_information_max, int mutual_information_step);

    /**
     * Print entropy file
     *
     * @param[in] dir Output directory
     * @param[in] write_stdout True, if write to stdout
     * @param[in] prefix Module name prefix for result files
     * @param[in] file_id File id which will be printed in the header
     * @param[in] file_path Path to the file which was analyzed
     * @param[in] file_length Length of the analyzed file
     * @param[in] block_entropy Block entropy
     * @param[in] cond_entropy Conditional entropy
     * @param[in] entropy_entries Size of the entropy arrays
     */
    void printCsvEntropy(string &dir, bool write_stdout, const string &prefix, 
                         long file_id, string &file_path, long long file_length,
                         long double *block_entropy, long double *cond_entropy,
                         int entropy_entries);
    
    /**
     * Print bigram distribution file
     *
     * @param[in] dir Output directory
     * @param[in] write_stdout True, if write to stdout
     * @param[in] prefix Module name prefix for result files
     * @param[in] file_id File id which will be printed in the header
     * @param[in] file_path Path to the file which was analyzed
     * @param[in] file_length Length of the analyzed file
     * @param[in] ind_map Index map
     * @param[in] count_bigram Bigram count
     * @param[in] num_distinct_bigrams Number of different bigrams
     * @param[in] all_bigrams Total number of (not necessarily different) bigrams
     */
    void printCsvBigramDistribution(string &dir, bool write_stdout, const string &prefix, 
                                    long file_id, string &file_path, long long file_length,
                                    const IndexMap<char> *ind_map, long long **count_bigram,
                                    long long num_distinct_bigrams, long long all_bigrams);
    
    
    /**
     * Print repeat machine iterations header
     *
     * @param[in] csv Csv output stream
     * @param[in] oFile Underlying output file stream
     * @param[in] dir Output directory
     * @param[in] write_stdout True, if write to stdout
     * @param[in] prefix Module name prefix for result files
     * @param[in] file_id File id which will be printed in the header
     * @param[in] file_path Path to the file which was analyzed
     * @param[in] file_length Length of the analyzed file
     * @param[in] machine_name Name of the repeat machine type
     * @param[in] params Array of all repeat machine parameter names
     * @param[in] num_params Number of repeat machine parameters
     */
    void printCsvRepeatsIterationsHeader(CsvOutStream &csv,ofstream &oFile,
                                         string &dir, bool write_stdout, const string &prefix, 
                                         long file_id, string &file_path, long long file_length,
                                         const char *machine_name, const char **params,
                                         int num_params);
    
    /**
     * Print single repeat machine iteration 
     *
     * @param[in] csv Csv output stream
     * @param[in] distribution Distribution of repeat machine parameters
     * @param[in] num_params Number of repeat machine parameters
     * @param[in] iteration Number of the iteration
     */
    void printCsvRepeatsIteration(CsvOutStream &csv, float *distribution, int num_params, int iteration);
    
    /**
     * Print repeat machine file
     *
     * @param[in] dir Output directory
     * @param[in] write_stdout True, if write to stdout
     * @param[in] prefix Module name prefix for result files
     * @param[in] file_id File id which will be printed in the header
     * @param[in] file_path Path to the file which was analyzed
     * @param[in] file_length Length of the analyzed file
     * @param[in] machine_name Name of the repeat machine type
     * @param[in] params Array of parameter names
     * @param[in] distribution Distribution of repeat machine parameters
     * @param[in] num_params Number of repeat machine parameters
     * @param[in] ind_map Index map
     * @param[in] complement Array containing complement of each character or NULL if not inverted repeat
     */
    void printCsvRepeats(string &dir, bool write_stdout, const string &prefix, 
                               long file_id, string &file_path, long long file_length,
                               const char *machine_name, const char **params, 
                               float *distribution, int num_params,
                               const IndexMap<char> *ind_map, unsigned char* complement);
    
    /**
     * Print word distribution file
     *
     * @param[in] dir Output directory
     * @param[in] write_stdout True, if write to stdout
     * @param[in] prefix Module name prefix for result files
     * @param[in] file_id File id which will be printed in the header
     * @param[in] file_path Path to the file which was analyzed
     * @param[in] file_length Length of the analyzed file
     * @param[in] ind_map Index map
     * @param[in] word_delimiters List of all word delimiters
     * @param[in] count_word Conter for each word
     * @param[in] count_word_sum Number of (not necessarily different) different words
     */
    void printCsvWordDistribution(string &dir, bool write_stdout, const string &prefix, 
                                  long file_id, string &file_path, long long file_length,
                                  const IndexMap<char> *ind_map, bool *word_delimiters,
                                  pair_unsigned_cstring_int_long_long_hash_map &count_word,
                                  long long count_word_sum);
}

#endif
