#ifndef _index_map_h_
#define _index_map_h_

#include <cstdlib>
#include <cstring>
#include <iostream>

using namespace std;

namespace papi
{
    
    /**
     * @class IndexMap
     *
     * This class assigns each symbol of the used alphabet an index which can be
     * used to index arrays which have exactly the size of the alphabet.
     */
    template <class T>
    class IndexMap
    {
    protected:
        /// Stores the index for each value
        int *index;
        /// Stores the value for each index
        T *value;
        /// Minimal value
        T min_val;
        /// Maximal value
        T max_val;
    public:
        /// Number of assigned indices
        int size;
        
        /**
         * Constructor
         * @param[in] min_value Minimal value
         * @param[in] max_value Maximal value
         */
        IndexMap(T min_value, T max_value):
        min_val(min_value), max_val(max_value), size(0)
        {
            int max_size = (int)(max_val-min_val+1);
            index = new int[max_size];
            value = new T[max_size];
            memset(index,-1,sizeof(int)*max_size);
        }
        /**
         * Destructor
         */
        ~IndexMap()
        {
            if(index)
                delete [] index;
            if(value)
                delete [] value;
        }
        
        /**
         * Add the given value to the index map
         * @param[in] c The value
         * @return Index of the value
         */
        int addIndex(T c)
        {
            if(c>max_val || c<min_val)
            {
                cerr << "IndexMap: Value out of bound: "<<c<<endl;
                exit(EXIT_FAILURE);
            }
            
            int tmp = c-min_val;
            if(index[tmp]==-1)
            {
                index[tmp]=size;
                value[size++]=c;
            }
            return index[tmp];
        }
        
        /**
         * Get index of the value
         * @param[in] c The value
         * @return Index of the value or -1 if value was not added
         */
        int getIndex(T c) const
        {
            
            if(c>max_val || c<min_val)
            {
                cerr << "IndexMap: Value out of bound: "<<c<<endl;
                exit(EXIT_FAILURE);
            }
            
            int tmp = c-min_val;
            return index[tmp];
        }
        
        /**
         * Return the value of an index
         * @param[in] c The index
         * @return Value of the index
         */
        T getValue(int c) const
        {
            if((int)c>=size || c<0)
            {
                cerr << "IndexMap: Index out of bound "<<c<<endl;
                exit(EXIT_FAILURE);
            }
            return value[c];
        }
    };

}

#endif

