#ifndef _stack_fixed_size_h_
#define _stack_fixed_size_h_
/**
 * @class StackFixedSizeRandomAccess
 *
 * Stack of fixed size which stores its content sequentially making it 
 * possible to treat it as an ordinary array
 *
 * It is only possible to add elements. If the stack hits its maximum size
 * old elements are deleted.
 */
template <class T>
class StackFixedSizeRandomAccess
{
protected:
    /// Buffer containing the content
    T *buffer;
public:
    /// Pointer to the start of the stack (last inserted element)
    T *start;
    /// Pointer to the end of the stack (delimiter after the first inserted element)
    T *end;
    /// Delimiter which always stands at the end of the stack making it possible to use the stack as a C-String
    T delimiter;
    /// Current size of the stack
    int size;
    /// Maximum size of the stack
    int max_size;
    
    /**
     * Constructor
     * 
     * @param max_size Maximum size of the stack
     * @param delimiter Delimiter at the end of the stack
     */
    StackFixedSizeRandomAccess(int max_size, T delimiter):
    delimiter(delimiter),size(0),max_size(max_size)
    {
        buffer = new T[max_size<<1];
        start = end = buffer+(max_size<<1)-1;
        
        memset(buffer, delimiter, sizeof(T)*(max_size<<1));
    }
    
    /**
     * Insert element at the beginning of the stack 
     *
     * @param[in] el The element
     */
    void push_front(T el)
    {
        if(max_size==0)
            return;
        
        if(size == max_size)
        {
            if(end==buffer+max_size)
            {
                memcpy(buffer+max_size, start, sizeof(T)*(max_size-1));
                buffer[max_size-1] = el;
                start = buffer+max_size-1;
                end = buffer+(max_size<<1)-1;
            }
            else {
                *--start = el;
                *--end = delimiter;
            }
            
        }
        else {
            ++size;
            *--start = el;
        }
    }
    
    /**
     * Random access to the stack
     *
     * @param[in] i The index
     * @return The value at index i
     */
    T& operator[](const int i)
    {
        return start[i];
    }
};

#endif
