// ==========================================================================
//                       index_compressed_benchmark
// ==========================================================================
// Copyright (c) 2006-2010, Knut Reinert, FU Berlin
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * Neither the name of Knut Reinert or the FU Berlin nor the names of
//       its contributors may be used to endorse or promote products derived
//       from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL KNUT REINERT OR THE FU BERLIN BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
// DAMAGE.
//
// ==========================================================================
// Author: Tobias Stadler <stadler@in.tum.de>
// ==========================================================================
// Benchmark for the SeqAn module index_compressed.
// ==========================================================================

#ifndef SANDBOX_TUM_APPS_INDEX_COMPRESSED_BENCHMARK_INDEX_COMPRESSED_BENCHMARK_H_
#define SANDBOX_TUM_APPS_INDEX_COMPRESSED_BENCHMARK_INDEX_COMPRESSED_BENCHMARK_H_

using namespace seqan;

// ============================================================================
// Forwards
// ============================================================================

// ============================================================================
// Tags, Classes, Enums
// ============================================================================

struct Options
{
    int alphabet;
    int index;

    int sadakaneBlockLength;

    int numberOfFindIterations;
    int maxPatternLength;
    bool displayOccurences;

    int numberOfExtractIterations;
    int maxSubstringLength;

    seqan::String<seqan::CharString> texts;

    Options()
    {
        // Set defaults.
        alphabet = 1;
        index = 2;

        sadakaneBlockLength = 0;

        numberOfFindIterations = 1000;
        maxPatternLength = 10000;
        displayOccurences = false;

        numberOfExtractIterations = 1000;
        maxSubstringLength = 10000;
    }
};

// ============================================================================
// Metafunctions
// ============================================================================

// ============================================================================
// Functions
// ============================================================================

void
setupCommandLineParser(seqan::CommandLineParser & parser, Options const & /*options*/)
{
    addVersionLine(parser, "0.1");

    addTitleLine(parser, "******************************");
    addTitleLine(parser, "* index_compressed_benchmark *");
    addTitleLine(parser, "******************************");
    addTitleLine(parser, "");
    addTitleLine(parser, "(c) 2011 by Tobias Stadler <stadler@in.tum.de>");

    addUsageLine(parser, "[OPTIONS] TEXT+");

    addSection(parser, "Index Options");
    addOption(parser, CommandLineOption('a', "alphabet", "set the alphabet (default: Char)", OptionType::String));
    addOption(parser, CommandLineOption('i', "index", "set the compressed index (default: Sadakane)", OptionType::String));

    addSection(parser, "IndexSadakane Options");
    addOption(parser, CommandLineOption('b', "blockLength", "set the block length for the incremental construction algorithm (default: 0)", OptionType::Int));

    addSection(parser, "Search Options");
    addOption(parser, CommandLineOption('f', "numberOfFindIterations", "set the number of find iterations (default: 1000)", OptionType::Int));
    addOption(parser, CommandLineOption('p', "maxPatternLength", "set the maximum length of a pattern (default: 10000)", OptionType::Int));
    addOption(parser, CommandLineOption('o', "displayOccurences", "display the occurrence positions", OptionType::Bool));

    addSection(parser, "Extract Options");
    addOption(parser, CommandLineOption('e', "numberOfExtractIterations", "set the number of extract iterations (default: 1000)", OptionType::Int));
    addOption(parser, CommandLineOption('s', "maxSubstringLength", "set the maximum length of a substring (default: 1000)", OptionType::Int));

    requiredArguments(parser, 1);
}

bool
parseCommandLine(Options & options,
                                                         CommandLineParser & parser,
                                                         int argc,
                                                         char const ** argv)
{
    if (!parse(parser, argc, argv)) {
        return false;
    }

    if (isSetShort(parser, 'a')) {
        CharString str;
        getOptionValueShort(parser, 'a', str);

        toLower(str);

        if (str == "char") {
            options.alphabet = 1;
        } else if (str == "aminoacid") {
            options.alphabet = 2;
        } else if (str == "dna") {
            options.alphabet = 3;
        } else if (str == "dnaq") {
            options.alphabet = 4;
        } else if (str == "dna5") {
            options.alphabet = 5;
        } else if (str == "dna5q") {
            options.alphabet = 6;
        } else if (str == "iupac") {
            options.alphabet = 7;
        } else if (str == "rna") {
            options.alphabet = 8;
        } else if (str == "rna5") {
            options.alphabet = 9;
        } else {
            std::cerr << "Error! " << options.alphabet << " is not a valid alphabet!" << std::endl;
            std::cerr << "Possible values are: Char, AminoAcid, Dna, DnaQ, Dna5, Dna5Q, Iupac, Rna, Rna5." << std::endl;

            return false;
        }
    }
    if (isSetShort(parser, 'i')) {
        CharString str;
        getOptionValueShort(parser, 'i', str);

        toLower(str);

        if (str == "lz") {
            options.index = 1;
        } else if (str == "sadakane") {
            options.index = 2;
        } else {
            std::cerr << "Error! " << str << " is not a valid compressed index!" << std::endl;
            std::cerr << "Possible values are: lz, sadakane." << std::endl;

            return false;
        }
    }

    if (isSetShort(parser, 'b')) {
        if (options.index == 2) {
            getOptionValueShort(parser, 'b', options.sadakaneBlockLength);

            if(options.sadakaneBlockLength < 0) {
                std::cerr << "Error! The number of find iterations must be greater than zero!" << std::endl;

                return false;
            }
        } else {
            std::cerr << "-b is only available if -i sadakane is supplied!" << std::endl;
        }
    }

    if (isSetShort(parser, 'f')) {
        getOptionValueShort(parser, 'f', options.numberOfFindIterations);

        if(options.numberOfFindIterations <= 0) {
            std::cerr << "Error! The number of find iterations must be greater than zero!" << std::endl;

            return false;
        }
    }
    if (isSetShort(parser, 'p')) {
        getOptionValueShort(parser, 'p', options.maxPatternLength);

        if(options.maxPatternLength <= 0) {
            std::cerr << "Error! The maximum length of a patternn must be greater than zero!" << std::endl;

            return false;
        }
    }     if (isSetShort(parser, 'o')) {
        options.displayOccurences = true;
    }

    if (isSetShort(parser, 'e')) {
        getOptionValueShort(parser, 'e', options.numberOfExtractIterations);

        if(options.numberOfExtractIterations <= 0) {
            std::cerr << "Error! The number of extract iterations must be greater than zero!" << std::endl;

            return false;
        }
    }
    if (isSetShort(parser, 's')) {
        getOptionValueShort(parser, 's', options.maxSubstringLength);

        if(options.maxSubstringLength <= 0) {
            std::cerr << "Error! The maximum length of a substring must be greater than zero!" << std::endl;

            return false;
        }
    }

    options.texts = getArgumentValues(parser);

    return true;
}

#endif    // #ifndef SANDBOX_TUM_APPS_INDEX_COMPRESSED_BENCHMARK_INDEX_COMPRESSED_BENCHMARK_H_
