// ==========================================================================
//                             index_suffix_trees
// ==========================================================================
// Copyright (c) 2006-2011, Knut Reinert, FU Berlin
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * Neither the name of Knut Reinert or the FU Berlin nor the names of
//       its contributors may be used to endorse or promote products derived
//       from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL KNUT REINERT OR THE FU BERLIN BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
// DAMAGE.
//
// ==========================================================================
// Author: Alexander Aumann <a.aumann@web.de>
// ==========================================================================
// Space-reduced suffix tree of Kurtz.
//
// [Kurtz1999] S. Kurtz "Reducing the space requirement of suffix trees"
//   Software - Practice and Experience, John Wiley & Sons, Inc., 1999, 29, 1149-1171,
//   http://dx.doi.org/10.1002/(SICI)1097-024X(199911)29:13<1149::AID-SPE274>3.0.CO;2-O
// ==========================================================================

#ifndef SANDBOX_TUM_INCLUDE_SEQAN_INDEX_SUFFIX_TREES_INDEX_STKURTZ_BASE_H_
#define SANDBOX_TUM_INCLUDE_SEQAN_INDEX_SUFFIX_TREES_INDEX_STKURTZ_BASE_H_

namespace seqan {

// ============================================================================
// Forwards
// ============================================================================

// ============================================================================
// Tags, Classes, Enums
// ============================================================================

// ----------------------------------------------------------------------------
// Fibres
// ----------------------------------------------------------------------------

struct StKurtzNodeTab_;
struct StKurtzLeafTab_;

typedef FibreText StKurtzText;
typedef FibreRawText StKurtzRawText;
typedef Tag<StKurtzNodeTab_> const StKurtzNodeTab;
typedef Tag<StKurtzLeafTab_> const StKurtzLeafTab;

// ----------------------------------------------------------------------------
// Bit mask
// ----------------------------------------------------------------------------

struct StKurtzBitMasks
{
    static const unsigned int AllBitsSet = 0xFFFFFFFF;

    static const unsigned int ShiftDistanceBitsInt1 = 27;
    static const unsigned int DistanceInt1 = 31 << ShiftDistanceBitsInt1;           //5 highest bits
    static const unsigned int FirstChildRefInt1 = AllBitsSet - DistanceInt1;        //27 lowest bits

    static const unsigned int FirstChildRefInt2 = 3 << 30;                          //2 highest bits
    static const unsigned int NilBitInt2 = 1 << 29;                                 //3rd highest bit
    static const unsigned int BBRefOrSlinkFatherInt2 =
            AllBitsSet - FirstChildRefInt2 - NilBitInt2;                            //29 lowest bits

    static const unsigned int NotDeepBitInt3 = 1 << 31;                             //highest bit
    static const unsigned int DepthFlatInt3 = 1023;                                 //10 lowest bits
    static const unsigned int SlinkInt3 =
            AllBitsSet - NotDeepBitInt3 - DepthFlatInt3;                            //Bits 2 to 22
    static const unsigned int DepthDeepInt3 = FirstChildRefInt1;                    //27 lowes bits
    static const unsigned int ShiftSlinkBitsInt3 = 10;                              //will be to the very right

    static const unsigned int SlinkInt4 = 31 << 27;                                 //5 highest bits
    static const unsigned int HeadPosInt4 = AllBitsSet - SlinkInt4;                 //27 lowest bits
    static const unsigned int ShiftSlinkBitsInt4 = 6;                               //will be to the left of int3 slink bits

    static const unsigned int SlinkLeaf = 3 << 30;                                  //2 highest bits
    static const unsigned int NotNilBitLeaf = 1 << 29;                              //3rd highest bit
    static const unsigned int SlinkFatherOrBBRefLeaf =
            AllBitsSet - SlinkLeaf - NotNilBitLeaf;                                 //29 lowest bits
    static const unsigned int ShiftSlinkBitsLeaf = 4;                               //will be bits 28 and 27 (very left)

    static const unsigned int RawRefsLeafBit = 1 << 28;                             //29th lowest bit
    static const unsigned int TwoHighestRawRefBits = 3 << 27;                       //bits 29 and 28 from the right
    static const unsigned int ShiftTwoHighestRawRefBitsToInt2 = 3;                  //shift val to align to the very left of integer2
    static const unsigned int NoPos = AllBitsSet;                                   //signals an illegal position
    static const unsigned int PosGreaterThanString = AllBitsSet - 1;                //may be useful
};

// -----------------------------------------------------------------------------
// Internal Nodes Definitions
// There are three types: BigNodes, SmallNodes and LeafNodes
// LeafNodes consist of just one integer pointing to the right neighbor, or null if there is none
// BigNodes and SmallNodes are inner nodes, containing either 2 or 4 integer values
// The meaning of the bits inside of the integers depends on (bits numbered from 1 to 32)
// 1) in a BigNode the first 5 bits of Integer 1 are 0, otherwise they store a distance number
// 2) bit 3 of Integer 2 determines whether the last 29 bits store
//         a) the suffix link
//         b) the branch brother reference
// 3) bit 1 of Integer 3 determines if the node depth is at most 2^10-1 (bit set) or greater
//         a) if the depth is at most 2^10 the last 10 bits store the depth
//             and bits 2-22 the first 21 bits of the suffix link,
//             the first 5 Bits of Integer 4 store the remaining suffix link bits
//         b) if the depth is at least 2^10 then the lower 27 bits store the depth
// 4) the last 27 bits of Integer 4 always store the headposition
// -----------------------------------------------------------------------------

struct StKurtzBigNode
{
    //5 zero bits, 27 first child bits
    unsigned integer1;

    //2 first child bits, 1 nil bit(=last child bit), 29 branch brother(BB) or s(uffix)link bits
    unsigned integer2;

    //1 isDeep bit: (21 slink bits, 10 depth bits) or (27 lowest bits = depth bits)
    unsigned integer3;

    //5 slink bits, 27 head position bits (hp bits always!)
    unsigned integer4;
};

struct StKurtzSmallNode
{
    //5 distance bits, 27 first child bits
    unsigned integer1;

    //2 first child bits, 1 nil bit, 29 BB or slink bits
    unsigned integer2;
};

struct StKurtzLeafNode
{
    unsigned rightBrother;
};

enum StKurtzNodeType
{
    BIG_NODE, SMALL_NODE, LEAF
};

template<typename TNodes, typename TLeaves>
struct StKurtzTarget
{
    typename Position<TNodes>::Type posNode;
    typename Position<TLeaves>::Type posLeaf;
    StKurtzNodeType nodeType;
    typedef StKurtzBitMasks BM;
    
    // TODO(krugel) Correctly added this constructor?
    StKurtzTarget() :
        posNode(BM::NoPos),
        posLeaf(BM::NoPos),
        nodeType(LEAF) {};

    bool operator==(StKurtzTarget<TNodes, TLeaves> & rhs)
    {
        SEQAN_ASSERT(this->nodeType == LEAF ? this->posNode == BM::NoPos : this->posLeaf == BM::NoPos);
        SEQAN_ASSERT(rhs.nodeType == LEAF ? rhs.posNode == BM::NoPos : rhs.posLeaf == BM::NoPos);
        return (this->nodeType == rhs.nodeType && this->posNode == rhs.posNode
                && this->posLeaf == rhs.posLeaf);
    }
    bool operator !=(StKurtzTarget<TNodes, TLeaves> & rhs)
    {
        return !((*this) == rhs);
    }

};

template<typename TText, typename TNodes, typename TLeaves>
struct StKurtzHeadLoc
{
    typename Position<TNodes>::Type fatherPos;
    typename Infix<TText>::Type rest;
    StKurtzTarget<TNodes, TLeaves> leftBrother;    //to get the exact edge position (node creation)
};

template<typename TText>
struct StKurtzTailPtr
{
    typename Suffix<TText>::Type tail;
};

template<typename TText, typename TNodes, typename TLeaves>
struct StKurtzHLTP
{
    StKurtzHeadLoc<TText, TNodes, TLeaves> headLoc;
    StKurtzTailPtr<TText> tailPtr;
};

template<typename TNodes_ = String<unsigned>, typename TLeaves_ = String<unsigned> >
struct StKurtzTableConfig
{
    typedef TNodes_ NODE_TYPE;
    typedef TLeaves_ LEAF_TYPE;
};

template<typename Config = StKurtzTableConfig<>, typename TSpec = void>
struct IndexStKurtz {};

template<typename TObj, typename TConfig, typename TSpec>
class Index<TObj, IndexStKurtz<TConfig, TSpec> >
{
public:
    typedef typename Cargo<Index>::Type TCargo;
    typedef typename Fibre<Index, StKurtzText>::Type TText;
    //typedef typename Fibre<Index, StKurtzBranchNodes>::Type TNodes;
    //typedef typename Fibre<Index, StKurtzLeafNodes>::Type  TLeaves;
    typedef typename TConfig::NODE_TYPE TNodes;
    typedef typename TConfig::LEAF_TYPE TLeaves;
    typedef Holder<TText> THolder;
    typedef typename Value<TText>::Type TValue;
    typedef typename Size<TText>::Type TSize;
    typedef typename Position<TText>::Type TPosition;

    ////The suffix tree consists of two tables TLeafs containing the leaves
    ////and TNodes which contains the Inner Nodes (=SmallNode or BigNode, arbitrarily mixed)
    //typedef String<unsigned>                                Table;

    THolder text;           //the text to index
    //TText text;             //the text to index
    TCargo cargo;           //user defined cargo
    TLeaves leafTable;      //the leaf table
    TNodes nodeTable;       //the branch node table
    TSize textSize;

    Index()
    {
    }
    Index(Index & other) :
            text(other.text),
                    cargo(other.cargo),
                    leafTable(other.leafTable),
                    nodeTable(other.nodeTable)
    {
    }
    //Index(Index const& other) :
    //        text(other.text),
    //                cargo(other.cargo),
    //                leafTable(other.leafTable),
    //                nodeTable(other.nodeTable)
    //{
    //}
    template<typename TText_>
    Index(TText_ &_text) :
            text(_text)
    {
    }

    //template<typename TText_>
    //Index(TText_ const &_text) :
    //        text(_text)
    //{
    //}

};

// ============================================================================
// Metafunctions
// ============================================================================

template<typename TText, typename TSpec>
struct Fibre<Index<TText, TSpec>, StKurtzNodeTab>
{
    typedef String<unsigned> Type;
};

template<typename TText, typename TSpec>
struct Fibre<Index<TText, TSpec>, StKurtzLeafTab>
{
    typedef String<unsigned> Type;
};

// TODO(krugel) Enable as soon as suffix tree iterators are implemented
template <typename TSpec>
struct SupportsSuffixTreeIteration<IndexStKurtz<TSpec> > {
    typedef False Type;
};

// ============================================================================
// Functions
// ============================================================================

template<typename TText, typename TConfig, typename TSpec>
inline bool
_createIndex(Index<TText, IndexStKurtz<TConfig, TSpec> > & tree)
{
    SEQAN_CHECKPOINT;
    _initIndexStKurtz(tree);
    _buildIndexStKurtz(tree);
    return true;
}

template<typename TText, typename TConfig, typename TSpec>
inline void
_initIndexStKurtz(Index<TText, IndexStKurtz<TConfig, TSpec> > & tree)
{
    SEQAN_CHECKPOINT;
    //first of all, we create the tables for the leaf nodes (exactly size of n)
    //and for the inner nodes. as the average size in bytes for a suffix tree for string of size n
    //is ~10n bytes, we start with 2.5 * n integers (one int takes 4 bytes)
    //TODO(aumann): more sophisticated allocation
    tree.textSize = length(tree.text);
    if (tree.textSize <= 0)
        return;

    resize(tree.leafTable, tree.textSize, 0, Exact());
    reserve(tree.nodeTable, (int) (tree.textSize * 2.5));

    //insert empty root
    appendValue(tree.nodeTable, 0);
    appendValue(tree.nodeTable, 0);
    appendValue(tree.nodeTable, 0 + StKurtzBitMasks::NotDeepBitInt3);
    appendValue(tree.nodeTable, 0);

}

template<typename TText, typename TConfig, typename TSpec>
inline void
_buildIndexStKurtz(Index<TText, IndexStKurtz<TConfig, TSpec> > & index)
{
    SEQAN_CHECKPOINT;
    typedef typename TConfig::NODE_TYPE TNodes;
    typedef typename TConfig::LEAF_TYPE TLeaves;
    typedef StKurtzBitMasks BM;

    SEQAN_ASSERT_EQ(length(index.nodeTable), 4u);    //the root is there
    StKurtzHeadLoc<TText, TNodes, TLeaves> headloc;
    headloc.fatherPos = 0;
    headloc.rest = infix(indexText(index), 0, 0);
    StKurtzTailPtr<TText> tailptr;
    StKurtzTarget<TNodes, TLeaves> tempNode;
    typename Iterator<TNodes, Rooted>::Type nodeIt = begin(index.nodeTable);
    tailptr.tail = suffix(indexText(index), 0);
    unsigned baseAddressQ = 4;    //node insertion address
    unsigned chainLengthP = 0;
    _insertLeaf(index, headloc, tailptr);

    appendValue(index.nodeTable, 0);    //always two trailing node ints necessary
    appendValue(index.nodeTable, 0);
    unsigned i = 0;
    unsigned lastNodeInsertionAddress = 0;

    //tail begin is at most the position of the last real character (the one before e.g. $)
    while (beginPosition(tailptr.tail) <= length(indexText(index)) - 2    //still characters || headloc != root
    || !(headloc.fatherPos == 0 && length(headloc.rest) == 0))
    {
        if (headloc.fatherPos == 0 && length(headloc.rest) == 0)
        {    //root
            setBeginPosition(tailptr.tail, beginPosition(tailptr.tail) + 1);
            _scanPrefix(index, headloc, tailptr, lastNodeInsertionAddress);
        }
        else
        {
            if (length(headloc.rest) == 0)
            {    //is a node with suffix link already complete
                unsigned slink = _getOwnSlink(index.nodeTable, index.leafTable, headloc.fatherPos);
                tempNode = _followReference(index.nodeTable, index.leafTable, slink);
                SEQAN_ASSERT(_nodeValid(tempNode));
                headloc.fatherPos = tempNode.posNode;
                _scanPrefix(index, headloc, tailptr, lastNodeInsertionAddress);
            }
            else
            {
                if (headloc.fatherPos == 0)
                { //the root is the father (intended use of no && length(...) addition!!
                    setBeginPosition(headloc.rest, beginPosition(headloc.rest) + 1);    //chop of the first char
                    _rescan(index, headloc, lastNodeInsertionAddress);

                }
                else
                {
                    unsigned slink = _getOwnSlink(index.nodeTable, index.leafTable, headloc.fatherPos);
                    tempNode = _followReference(index.nodeTable, index.leafTable, slink);
                    SEQAN_ASSERT(_nodeValid(tempNode));
                    headloc.fatherPos = tempNode.posNode;
                    _rescan(index, headloc, lastNodeInsertionAddress);
                }
                chainLengthP++;
                //TODO(aumann): when chain becomes to big, insert artificial big node!
                if (length(headloc.rest) == 0)
                {    //is a node
                    //set own distance to zero
                    setPosition(nodeIt, baseAddressQ);
                    unsigned int1 = *nodeIt;
                    int1 &= (BM::AllBitsSet - BM::DistanceInt1);
                    assignValue(nodeIt, int1);

                    _setOwnSlink(index.nodeTable, index.leafTable, baseAddressQ, headloc.fatherPos);
                    for (unsigned d = 1; d < chainLengthP; d++)
                    {    //set distances
                        setPosition(nodeIt, baseAddressQ - 2 * d);
                        unsigned int1 = *nodeIt;
                        int1 &= (BM::AllBitsSet - BM::DistanceInt1);
                        int1 += (d << BM::ShiftDistanceBitsInt1);
                        assignValue(nodeIt, int1);
                    }

                    chainLengthP = 0;
                    baseAddressQ += 4;
                    appendValue(index.nodeTable, 0);    //always two trailing node ints necessary
                    appendValue(index.nodeTable, 0);
                    /***CS1***/
                    //missing scanprefix in contrast to paper
                    _scanPrefix(index, headloc, tailptr, lastNodeInsertionAddress);//seems to change little

                    /*********/

                }
                else
                {
                    //SEQAN_ASSERT_EQ(_getHeadPosition(index.nodeTable, baseAddressQ - 2, BM::NoPos), i);
                    baseAddressQ += 2;
                }

            }
        }

        if (length(headloc.rest) == 0)
        {    //is a node
            _insertLeaf(index, headloc, tailptr, lastNodeInsertionAddress);
        }
        else
        {
            _insertNode(index, baseAddressQ, headloc, tailptr);
            if (lastNodeInsertionAddress + 2 == baseAddressQ)
            {
                //set maximum distance of last node until real distance is set lateron
                setPosition(nodeIt, lastNodeInsertionAddress);
                unsigned int1 = *nodeIt;
                int1 &= (BM::AllBitsSet - BM::DistanceInt1);
                int1 += (31 << BM::ShiftDistanceBitsInt1);
                assignValue(nodeIt, int1);
            }
            lastNodeInsertionAddress = baseAddressQ;
        }
        i++;
    }
}

template<typename TText, typename TConfig, typename TSpec>
void
_printTree(const Index<TText, IndexStKurtz<TConfig, TSpec> > & index,
StKurtzTarget<typename TConfig::NODE_TYPE, typename TConfig::LEAF_TYPE> const & currentFather)
{
    //print a preorder representation
    typedef typename TConfig::NODE_TYPE TNodes;
    typedef typename TConfig::LEAF_TYPE TLeaves;
    //typedef StKurtzBitMasks BM;

    StKurtzTarget<TNodes, TLeaves> currentChild;
    unsigned firstChildRef = _getFirstChildRef(index.nodeTable, currentFather.posNode);
    currentChild = _followReference(index.nodeTable, index.leafTable, firstChildRef);
    std::cout << "(";
    while (!_isLastChild(index.nodeTable, index.leafTable, currentChild))
    {
        if (currentChild.nodeType == LEAF)
        {
            std::cout << "(" << currentChild.posLeaf << ")";
        }
        else
        {
            _printTree(index, currentChild);
        }
        std::cout << " ";
        currentChild = _getBBOrSlinkFatherTarget(index.nodeTable, index.leafTable, currentChild);
    }
    //print last child
    if (currentChild.nodeType == LEAF)
    {
        std::cout << "(" << currentChild.posLeaf << ")";
    }
    else
    {
        _printTree(index, currentChild);
    }
    std::cout << ")";

    if (currentFather.posNode == 0)
    {
        std::cout << std::endl;
    }

}

//Determine node type at table base address i
template<typename TNodes>
inline unsigned int
_getNodeDistance(const TNodes & nodeTable,
                 typename Position<TNodes>::Type pos,
                 unsigned lastInsertedNodePos = StKurtzBitMasks::NoPos)
{
    SEQAN_CHECKPOINT;
    SEQAN_ASSERT_GEQ(length(nodeTable), (pos + 1) + 1);    //must be at least 1 more integer

    typedef StKurtzBitMasks BM;
    typename Iterator<TNodes const, Rooted>::Type it = begin(nodeTable);
    setPosition(it, pos);
    unsigned int nodeHead = *it;
    unsigned dist = ((nodeHead & StKurtzBitMasks::DistanceInt1) >> StKurtzBitMasks::ShiftDistanceBitsInt1);
    //special case, node distance 31, the chain is not yet completed!
    if (length(nodeTable) < pos + dist)
    {
        SEQAN_ASSERT_EQ(dist, 31u);
        //SEQAN_ASSERT_NEQ(lastInsertedNodePos, BM::NoPos);//allowed when called by followReference
        if (lastInsertedNodePos == BM::NoPos)
            return BM::NoPos;

        unsigned dif = lastInsertedNodePos - pos;
        SEQAN_ASSERT_EQ(dif % 2, 0u);
        dif /= 2;    //dif is now the virtual distance (= distance to the last inserted node)
        SEQAN_ASSERT_EQ(length(nodeTable), (pos + 2 * dif + 3) + 1);
        dist = dif;
    }
    return dist;

}

template<typename TNodes>
inline unsigned int
_getFirstChildRef(TNodes const & nodeTable, typename Position<TNodes>::Type pos)
{
    SEQAN_CHECKPOINT;
    typename Iterator<TNodes const, Rooted>::Type it = begin(nodeTable);
    typedef StKurtzBitMasks BM;
    SEQAN_ASSERT_GEQ(length(nodeTable), pos + 1);
    setPosition(it, pos);
    unsigned int nodeInt1 = *it;
    unsigned int nodeInt2 = *(++it);

    return (nodeInt1 & BM::FirstChildRefInt1) +
            ((nodeInt2 & BM::FirstChildRefInt2) >> BM::ShiftTwoHighestRawRefBitsToInt2);
}

template<typename TNodes>
inline unsigned int
_getNodeBBRefOrSlinkFather(TNodes const & nodeTable, typename Position<TNodes>::Type pos)
{
    SEQAN_CHECKPOINT;
    typename Iterator<TNodes const, Rooted>::Type it = begin(nodeTable);
    SEQAN_ASSERT_GEQ(length(nodeTable), (pos + 1) + 1);
    setPosition(it, pos + 1);
    unsigned int nodeInt2 = *it;
    return (nodeInt2 & StKurtzBitMasks::BBRefOrSlinkFatherInt2);
}

template<typename TLeaves>
inline unsigned int
_getLeafBBRefOrSlinkFather(TLeaves const & leafTable, typename Position<TLeaves>::Type pos)
{
    SEQAN_CHECKPOINT;
    typename Iterator<TLeaves const, Rooted>::Type it = begin(leafTable);
    SEQAN_ASSERT_GEQ(length(leafTable), (pos) + 1);     //leaf exists (i.e. its memory storage)
    setPosition(it, pos);
    unsigned int leaf = *it;
    return (leaf & StKurtzBitMasks::SlinkFatherOrBBRefLeaf);
}
//TODO(aumann): AAAAA very important: check for rawReference Nil-Bit consistency!

template<typename TNodes>
inline unsigned int
_getDepth(TNodes const & nodeTable,
          typename Position<TNodes>::Type pos,
          unsigned lastInsertedNodePos)
{
    SEQAN_CHECKPOINT;
    if (pos == 0)    //root
        return 0;

    typename Iterator<TNodes const, Rooted>::Type it;
    typedef StKurtzBitMasks BM;
    SEQAN_ASSERT_GEQ(length(nodeTable), (pos + 1) + 1);
    //determine whether node is big or small
    unsigned int distance = _getNodeDistance(nodeTable, pos, lastInsertedNodePos);
    if (distance == 0)
    {    //Big Node
        //the depth depends on whether the node is deep or not
        SEQAN_ASSERT_GEQ(length(nodeTable), (pos + 3) + 1);    //must be at least 3 integers following
        it = begin(nodeTable);
        setPosition(it, pos + 2);
        unsigned int nodeInt3 = *(it);
        if ((nodeInt3 & BM::NotDeepBitInt3) == 0)
        {    //not deep bit not set =>deep
            return (nodeInt3 & BM::DepthDeepInt3);
        }
        else
        {
            return (nodeInt3 & BM::DepthFlatInt3);
        }
    }
    else
    {    //small node
        //the depth needs to be calculated from the distance of the small node to the big node
        //and the depth of the BigNode at the end of the chain
        //assert that there is a big node at the right address
        SEQAN_ASSERT_GEQ(length(nodeTable), (pos + distance * 2) + 1);
        SEQAN_ASSERT_EQ(_getNodeDistance(nodeTable, pos + (distance * 2), lastInsertedNodePos), 0u);    //big node

        //calculate depth
        unsigned int depth;
        depth = _getDepth(nodeTable, pos + (distance * 2), BM::NoPos) + distance;//no last inserted node pos necessary
        return depth;
    }

}

template<typename TNodes>
inline unsigned int
_getHeadPosition(TNodes const & nodeTable, typename Position<TNodes>::Type pos, unsigned lastInsertedNodePos)
{
    SEQAN_CHECKPOINT;
    //the head position of a small node is calculated from its distance and the head position
    //of the big node at the end of its chain
    //the head number of a big node is stored in integer4
    typedef StKurtzBitMasks BM;

    SEQAN_ASSERT_GEQ(length(nodeTable), (pos + 1) + 1);
    typename Iterator<TNodes const, Rooted>::Type it = begin(nodeTable);
    setPosition(it, pos);

    unsigned int distance = _getNodeDistance(nodeTable, pos, lastInsertedNodePos);
    if (distance == 0)
    {    //big node
        unsigned int integer4 = *(it + 3);
        return (integer4 & StKurtzBitMasks::HeadPosInt4);
    }
    else
    {    //small node
        SEQAN_ASSERT_GEQ(length(nodeTable), (pos + 2 * distance) + 1);
        SEQAN_ASSERT_EQ(_getNodeDistance(nodeTable, pos + 2 * distance, lastInsertedNodePos), 0u);    //big node
        return _getHeadPosition(nodeTable, pos + 2 * distance, BM::NoPos) - distance;
    }

}

template<typename TNodes, typename TLeaves>
inline StKurtzTarget<TNodes, TLeaves>
_followReference(const TNodes & nodeTable, const TLeaves & leafTable, unsigned int rawReference)
{
    SEQAN_CHECKPOINT;
    ignoreUnusedVariableWarning(leafTable);
    
    //reference is an unsigned integer with only the least significant 29 bits mattering
    //if the first bit is set, the reference goes to a leaf, otherwise to a branch node
    //as all base addresses in the branch table are even, the node is found at address
    //nodeTable(rawReference * 2)

    typedef StKurtzBitMasks BM;

    //assert that the three highest bits are 0, would otherwise indicate an error elsewhere
    SEQAN_ASSERT_EQ((rawReference & (BM::AllBitsSet - BM::BBRefOrSlinkFatherInt2)), 0u);

    StKurtzTarget<TNodes, TLeaves> target;
    if ((rawReference & BM::RawRefsLeafBit) == 0)
    {    //Reference to a node
        target.posNode = rawReference * 2;
        target.posLeaf = BM::NoPos;
        //at least a small node must be in the table
        SEQAN_ASSERT_GEQ(length(nodeTable), (target.posNode + 1) + 1);
        if (_getNodeDistance(nodeTable, target.posNode, BM::NoPos) == 0)
        {
            SEQAN_ASSERT_GEQ(length(nodeTable), (target.posNode + 3) + 1);
            target.nodeType = BIG_NODE;
        }
        else
        {
            target.nodeType = SMALL_NODE;
        }

    }
    else
    {    //reference to a leaf
        target.posNode = BM::NoPos;
        rawReference -= BM::RawRefsLeafBit;    //remove the leaf indicator
        target.posLeaf = rawReference;
        SEQAN_ASSERT_GEQ(length(leafTable), (target.posLeaf) + 1);
        target.nodeType = LEAF;
    }
    return target;

}

template<typename TNodes, typename TLeaves>
inline bool
_isLastChild(const TNodes & nodeTable, const TLeaves & leafTable, StKurtzTarget<TNodes, TLeaves> node)
{
    SEQAN_CHECKPOINT;
    //if (node.nodeType == SMALL_NODE) {    //bollocks, very old assumption of mine :-/
    //    return false;    //small node can't be last child
    //}
    if (node.nodeType != LEAF)
    {
        typename Iterator<TNodes const, Rooted>::Type it = begin(nodeTable);
        SEQAN_ASSERT_GEQ(length(nodeTable), (node.posNode + 3) + 1);
        setPosition(it, node.posNode);
        unsigned integer2 = *(it + 1);
        if ((integer2 & StKurtzBitMasks::NilBitInt2) == StKurtzBitMasks::NilBitInt2)
        {
            //the nil bit is set, which indicates that this is the last child
            return true;
        }
        else
        {
            return false;
        }
    }
    else
    {    //leaf
        typename Iterator<TLeaves const, Rooted>::Type it = begin(leafTable);
        SEQAN_ASSERT_GEQ(length(leafTable), (node.posLeaf) + 1);
        setPosition(it, node.posLeaf);
        unsigned leaf = *it;
        if ((leaf & StKurtzBitMasks::NotNilBitLeaf) == StKurtzBitMasks::NotNilBitLeaf)
        {
            //not nil bit set => not the last child
            return false;
        }
        else
        {
            return true;
        }
    }

    return true;
}

template<typename TNodes, typename TLeaves>
inline unsigned int
_getOwnSlink(const TNodes & nodeTable, const TLeaves & leafTable, const typename Position<TNodes>::Type & pos)
{
    SEQAN_CHECKPOINT;
    //the own suffix link of big node w is either in the rightmost childnode if the node depth is
    //at least 2 ^ 10, or it is stored in integer3, integer4 and and TLeaf[w.headposition]
    //for a small node it's the next node in the node chain
    typename Iterator<TNodes const, Rooted>::Type it = begin(nodeTable);
    typedef StKurtzBitMasks BM;
    SEQAN_ASSERT_GEQ(length(nodeTable), (pos + 1) + 1);
    SEQAN_ASSERT_NEQ(pos, 0u);    //the root has no suffix link
    unsigned distance = _getNodeDistance(nodeTable, pos);//no lastInsertedPos, as _getOwnSlink should never be called before
    if (distance == 0)
    {    //big node
        SEQAN_ASSERT_GEQ(length(nodeTable), (pos + 3) + 1);
        setPosition(it, pos + 2);
        unsigned integer3 = *(it);
        unsigned integer4 = *(it + 1);

        if ((integer3 & BM::NotDeepBitInt3) == 0)
        {    //not deep bit not set => deep node
            //the suffix link is stored in the rightmost child
            StKurtzTarget<TNodes, TLeaves> target =
                    _followReference(nodeTable, leafTable,
                                     _getFirstChildRef(nodeTable, pos));
            while (!_isLastChild(nodeTable, leafTable, target))
            {
                if (target.nodeType == LEAF)
                {
                    unsigned leafBBRef
                    = _getLeafBBRefOrSlinkFather(leafTable, target.posLeaf);
                    target = _followReference(nodeTable, leafTable, leafBBRef);
                }
                else /*if (target.nodeType == BIG_NODE)*/
                {
                    unsigned nodeBBRef
                    = _getNodeBBRefOrSlinkFather(nodeTable, target.posNode);
                    target = _followReference(nodeTable, leafTable, nodeBBRef);
                } /*else {    //small node
                 //skip nodes and go directly to the end of the chain
                 //THIS DOES NOT WORK!!!
                 //   !a chain is stored in suffix link order, not in branch brother order
                 SEQAN_ASSERT_GEQ(length(nodeTable), 1 +
                 target.posNode + 2 * _getNodeDistance(nodeTable, target.posNode));
                 target.posNode = target.posNode + 2 * _getNodeDistance(nodeTable, target.posNode);
                 target.nodeType = BIG_NODE;
                 SEQAN_ASSERT_EQ(_getNodeDistance(nodeTable, target.posNode), 0); //big node
                 }*/
            }                //while (!_isLastChild(..))
                            //target now points to the last child, slink can be extracted
            SEQAN_ASSERT(target.nodeType == BIG_NODE || target.nodeType == LEAF);
            unsigned slink;
            if (target.nodeType == BIG_NODE)
            {
                slink = _getNodeBBRefOrSlinkFather(nodeTable, target.posNode);
            }
            else
            {    //LEAF
                slink = _getLeafBBRefOrSlinkFather(leafTable, target.posLeaf);
            }
            //a suffix link can never ever point to a leaf
            SEQAN_ASSERT_NEQ(_followReference(nodeTable, leafTable, slink).nodeType, LEAF);
            return slink;

        }
        else
        {    //flat node
            //suffix link is stored in integer3, 4 and the two highest bits of TLeaf[w.headposition]
            unsigned int slink = (integer3 & BM::SlinkInt3) >> BM::ShiftSlinkBitsInt3;
            slink += (integer4 & BM::SlinkInt4) >> BM::ShiftSlinkBitsInt4;
            typename Iterator<TLeaves const, Rooted>::Type leafIt = begin(leafTable);
            SEQAN_ASSERT_GEQ(length(leafTable), _getHeadPosition(nodeTable, pos, BM::NoPos) + 1);//leaf table long enough
            setPosition(leafIt, _getHeadPosition(nodeTable, pos, BM::NoPos));
            unsigned leafInteger = *leafIt;
            slink += (leafInteger & BM::SlinkLeaf) >> BM::ShiftSlinkBitsLeaf;
            SEQAN_ASSERT_EQ(((15 << 28) & slink), 0u);
            //4 highest Bits should be 0
            SEQAN_ASSERT_NEQ(_followReference(nodeTable, leafTable, slink).nodeType, LEAF);
            return slink;
        }

    }
    else
    { //small node
        //THIS IS INCORRECT: suffix link = branch brother
        //RATHER: suffix link = next node in node chain
        /*return (integer2 & BM::BBRefOrSlinkFatherInt2);*///incorrect
        unsigned int ownSlink = pos + 2;
        SEQAN_ASSERT_EQ(((BM::AllBitsSet - BM::BBRefOrSlinkFatherInt2) & ownSlink), 0u);
        SEQAN_ASSERT_EQ(_getHeadPosition(nodeTable, pos, BM::NoPos),
                        _getHeadPosition(nodeTable, pos + 2, BM::NoPos) - 1);
        SEQAN_ASSERT_EQ(pos % 2, 0u);
        ownSlink /= 2;    //return the rawReference to position

        return ownSlink;
    }

}

template<typename TText, typename TConfig, typename TSpec>
inline void
_insertLeaf(Index<TText, IndexStKurtz<TConfig, TSpec> > & index,
    const StKurtzHeadLoc<TText, typename TConfig::NODE_TYPE, typename TConfig::LEAF_TYPE> & headLoc,
    const StKurtzTailPtr<TText> & tailPtr,
    unsigned lastInsertedNodePos = StKurtzBitMasks::NoPos)
{

    //std::cout << "insertLeaf: headLoc.fp: " << headLoc.fatherPos << ". headLoc.beginR: "
    //    << beginPosition(headLoc.rest) << ". headLoc.endR: " << endPosition(headLoc.rest)
    //    << std::endl << ". headLoc.left.np: "
    //    << headLoc.leftBrother.posNode << ". headLoc.left.lp: " << headLoc.leftBrother.posLeaf
    //    << "." << "tailbegin: " << beginPosition(tailPtr.tail) << "." << std::endl;

    SEQAN_CHECKPOINT;
    SEQAN_ASSERT_EQ(length(headLoc.rest), 0u);
    typedef StKurtzBitMasks BM;
    typedef typename TConfig::NODE_TYPE TNodes;
    typedef typename TConfig::LEAF_TYPE TLeaves;
    //adds a leaf to the current node, at the right position
    //typename Iterator<TText, Rooted>::Type textIt = begin(indexText(index));
    //Iterator<TNodes, Rooted>::Type nodeIt = begin(index.nodeTable);
    //typename Iterator<TLeaves, Rooted>::Type leafIt = begin(index.leafTable);

    StKurtzTarget<TNodes, TLeaves> currentChild;
    StKurtzTarget<TNodes, TLeaves> leftChild;
    StKurtzTarget<TNodes, TLeaves> rightChild;
    StKurtzTarget<TNodes, TLeaves> leaf;

    //unsigned leafStart = beginPosition(tailPtr.tail);

    //setPosition(nodeIt, headLoc.fatherPos);
    unsigned fatherDepth = _getDepth(index.nodeTable, headLoc.fatherPos, lastInsertedNodePos);
    unsigned nextNodeRef = _getFirstChildRef(index.nodeTable, headLoc.fatherPos);

    leaf.nodeType = LEAF;
    leaf.posNode = BM::NoPos;
    leaf.posLeaf = beginPosition(tailPtr.tail) - fatherDepth;
    SEQAN_ASSERT_GEQ(leaf.posLeaf, 0u);

    typename Value<TText>::Type firstCharChild;
    typename Value<TText>::Type firstCharLeafEdge = value(indexText(index), beginPosition(tailPtr.tail));
    bool lastChild = false;

    if (nextNodeRef == 0)
    {    //this leaf is the first child of the node
        _setFirstChildRef(index.nodeTable, headLoc.fatherPos,
                          leaf.posLeaf + BM::RawRefsLeafBit); //leaf nil, do not set notNilBit
        //the leaf table need not be changed, as the newly inserted leaf has no branch brother!
    }
    else
    {
        //first determine where the leaf needs to be inserted
        currentChild = _followReference(index.nodeTable, index.leafTable, nextNodeRef);
        lastChild = _isLastChild(index.nodeTable, index.leafTable, currentChild);

        unsigned childHeadPosition;

        if (currentChild.nodeType == BIG_NODE || currentChild.nodeType == SMALL_NODE)
        {
            //the overridenChild... variable is set in the case this leaf insertion was in
            //an insertNode call, as the headposition might not yet available in the node table
            //but can be determined from the newly inserted node's headLoc.rest
            childHeadPosition = _getHeadPosition(index.nodeTable, currentChild.posNode, lastInsertedNodePos);

            firstCharChild = value(indexText(index), fatherDepth + childHeadPosition);
        }
        else
        {
            SEQAN_ASSERT_EQ(nextNodeRef & BM::RawRefsLeafBit, (unsigned) BM::RawRefsLeafBit);
            firstCharChild = value(indexText(index), (nextNodeRef - BM::RawRefsLeafBit) + fatherDepth);
        }

        SEQAN_ASSERT_NOT(firstCharLeafEdge == firstCharChild);
        if (firstCharLeafEdge < firstCharChild)
        {
            //the leaf will be the leftmost child => special case, as no "leftChild" exists
            //first, set the fathers first child link to the new leaf
            _setFirstChildRef(index.nodeTable, headLoc.fatherPos, leaf.posLeaf + BM::RawRefsLeafBit);
            //now set the leaf's branch brother reference to the old "firstChild" (== currentChild)
            _setBBRefOrSlinkFather(index.nodeTable, index.leafTable, leaf, currentChild);
        }
        else
        {
            while (firstCharLeafEdge > firstCharChild
                    && !lastChild)
            {
                leftChild = currentChild;
                if (leftChild.nodeType == BIG_NODE || leftChild.nodeType == SMALL_NODE)
                {
                    nextNodeRef = _getNodeBBRefOrSlinkFather(index.nodeTable, leftChild.posNode);
                }
                else
                {
                    nextNodeRef = _getLeafBBRefOrSlinkFather(index.leafTable, leftChild.posLeaf);
                }
                currentChild = _followReference(index.nodeTable, index.leafTable, nextNodeRef);
                if (currentChild.nodeType == BIG_NODE || currentChild.nodeType == SMALL_NODE)
                {
                    childHeadPosition = _getHeadPosition(index.nodeTable, currentChild.posNode, lastInsertedNodePos);
                    firstCharChild = value(indexText(index), fatherDepth + childHeadPosition);
                }
                else
                {
                    //if the current node is a leaf, the reference to the leaf equals its start
                    //position in the Text            the (unsigned) is a workaround for the EQ-macro
                    SEQAN_ASSERT_EQ(nextNodeRef & BM::RawRefsLeafBit, (unsigned) (BM::RawRefsLeafBit));
                    firstCharChild = value(indexText(index), (nextNodeRef - BM::RawRefsLeafBit) + fatherDepth);
                }
                lastChild = _isLastChild(index.nodeTable, index.leafTable, currentChild);
            }
            if (lastChild)
            {    //the currentChild is the last of the father's children
                //need to preserve the suffixlink of the foremost last child
                unsigned referenceLastChild;
                if (currentChild.nodeType == LEAF)
                {
                    referenceLastChild = _getLeafBBRefOrSlinkFather(index.leafTable, currentChild.posLeaf);
                }
                else
                {
                    referenceLastChild = _getNodeBBRefOrSlinkFather(index.nodeTable, currentChild.posNode);
                }
                _setBBRefOrSlinkFather(index.nodeTable, index.leafTable, currentChild, leaf);
                rightChild = _followReference(index.nodeTable, index.leafTable, referenceLastChild);
                _setBBRefOrSlinkFather(index.nodeTable, index.leafTable, leaf, rightChild, true);
                SEQAN_ASSERT(_isLastChild(index.nodeTable, index.leafTable, leaf));

            }
            else
            {
                _setBBRefOrSlinkFather(index.nodeTable, index.leafTable, leftChild, leaf);
                _setBBRefOrSlinkFather(index.nodeTable, index.leafTable, leaf, currentChild);
            }
        }
    }
}

template<typename TText, typename TConfig, typename TSpec>
inline void
_insertNode(Index<TText, IndexStKurtz<TConfig, TSpec> > & index, unsigned baseAddressQ,
StKurtzHeadLoc<TText, typename TConfig::NODE_TYPE, typename TConfig::LEAF_TYPE> & headLoc,
    const StKurtzTailPtr<TText> & tailPtr)
{
    SEQAN_ASSERT_GEQ(length(headLoc.rest), 1u);
    //std::cout << "insertNode: headLoc.fp: " << headLoc.fatherPos << ". headLoc.beginR: "
    //    << beginPosition(headLoc.rest) << ". headLoc.endR: " << endPosition(headLoc.rest)
    //    << std::endl  << ". headLoc.left.np: "
    //    << headLoc.leftBrother.posNode << ". headLoc.left.lp: " << headLoc.leftBrother.posLeaf
    //    << "." << "tailbegin: " << beginPosition(tailPtr.tail) << "." << std::endl;
    //headLoc contains a pointer to the father of the newly inserted node, as well as the edge (rest) from
    //the father to the new node, and if available (if onlyChild == false) also the left brother
    //the tailPtr contains the rest of the suffix that is inserted into the string
    //this means, the inserted node has one edge outgoing to a new leaf which edge is labeld tailPtr.tail
    //and a second edge pointing to an already existing node (or leaf).

    //a newly inserted node is always a "small" node, which might later be expanded, if necessary
    //however, a big node is created, so that the depth can temporarily be stored in the node
    //this is important for the _insertLeaf function which is called on the new node (see step 2)
    //TODO(aumann): VERIFY
    typedef typename TConfig::NODE_TYPE TNodes;
    typedef typename TConfig::LEAF_TYPE TLeaves;
    typedef StKurtzBitMasks BM;
    typename Iterator<TNodes, Rooted>::Type nodeIt = begin(index.nodeTable);

    appendValue(index.nodeTable, 0);
    appendValue(index.nodeTable, 0);
    SEQAN_ASSERT_EQ(length(index.nodeTable), (baseAddressQ + 3) + 1);    //only as long as nodes exist
    //it is assumed, that the first two node integers already exist
    //either because the last created node was small
    //or because when a large node was found, the two ints are artificially appended
    setPosition(nodeIt, baseAddressQ);
    assignValue(nodeIt, 0);    //clear values, there might have been a small node's "big node parts"
    assignValue(nodeIt + 1, 0);

    StKurtzTarget<TNodes, TLeaves> newNode;
    StKurtzTarget<TNodes, TLeaves> oldChild;    //the node or leaf whose edge will be split
    StKurtzTarget<TNodes, TLeaves> rightBrother;
    newNode.nodeType = BIG_NODE;
    newNode.posNode = baseAddressQ;
    newNode.posLeaf = BM::NoPos;

    //first step: split old edge father-(rest)-(something)->existingChild
    //into                        father-(rest)->newNode-(something)->existingChild
    //the depth and headLoc of the old node remain unchanged

    SEQAN_ASSERT_EQ(baseAddressQ % 2, 0u);

    //first of all, the insertion position has to be found (which is already done by the scan-methods)

    //THIS IF STATEMENT SEEMS UNNECCESSARY (and its execution is wrong, either way (suffix link father!)
    //if (headLoc.onlyChild) {
    //    //the father's child becomes the child of the new node
    //    unsigned fathersOldChildRef = _getFirstChildRef(index.nodeTable, headLoc.fatherPos);
    //    _setFirstChildRef(index.nodeTable, baseAddressQ, fathersOldChildRef);
    //    //the father's child is set to the new node
    //    _setFirstChildRef(index.nodeTable, headLoc.fatherPos, baseAddressQ / 2);

    //    oldChild =  _followReference(index.nodeTable, index.leafTable, fathersOldChildRef);
    //    SEQAN_ASSERT(_followReference(index.nodeTable, index.leafTable,
    //            _getFirstChildRef(index.nodeTable, baseAddressQ)) == oldChild);
    //} else {
    unsigned refToOldChild;
    if (headLoc.fatherPos == headLoc.leftBrother.posNode)
    {
        //the father's first child owns the edge that has to be split
        SEQAN_ASSERT_NEQ(headLoc.leftBrother.nodeType, LEAF);
        refToOldChild = _getFirstChildRef(index.nodeTable, headLoc.fatherPos);
        oldChild = _followReference(index.nodeTable, index.leafTable, refToOldChild);
    }
    else
    {

        if (headLoc.leftBrother.nodeType == LEAF)
        {
            refToOldChild = _getLeafBBRefOrSlinkFather(index.leafTable, headLoc.leftBrother.posLeaf);
        }
        else
        {
            refToOldChild = _getNodeBBRefOrSlinkFather(index.nodeTable, headLoc.leftBrother.posNode);
        }
        oldChild = _followReference(index.nodeTable, index.leafTable, refToOldChild);
    }
    unsigned rawBBRefOrSlinkFatherOldChild;
    if (oldChild.nodeType == BIG_NODE || oldChild.nodeType == SMALL_NODE)
    {
        rawBBRefOrSlinkFatherOldChild = _getNodeBBRefOrSlinkFather(index.nodeTable, oldChild.posNode);
    }
    else
    {
        rawBBRefOrSlinkFatherOldChild = _getLeafBBRefOrSlinkFather(index.leafTable, oldChild.posLeaf);
    }
    rightBrother = _followReference(index.nodeTable, index.leafTable, rawBBRefOrSlinkFatherOldChild);
    if (_isLastChild(index.nodeTable, index.leafTable, oldChild))
    {
        _setBBRefOrSlinkFather(index.nodeTable, index.leafTable, newNode, rightBrother, true);
    }
    else
    {
        //the branch brother of the old child becomes the branch brother of the new node!
        _setBBRefOrSlinkFather(index.nodeTable, index.leafTable, newNode, rightBrother);
    }

    _setFirstChildRef(index.nodeTable, baseAddressQ, refToOldChild);

    if (headLoc.fatherPos == headLoc.leftBrother.posNode)
    {
        //the father's first child owns the edge that has to be split
        SEQAN_ASSERT_NEQ(headLoc.leftBrother.nodeType, LEAF);
        _setFirstChildRef(index.nodeTable, headLoc.fatherPos, baseAddressQ / 2);

    }
    else
    {
        _setBBRefOrSlinkFather(index.nodeTable, index.leafTable, headLoc.leftBrother, newNode);
    }

    //}

    //the branchbrother reference of the child has to be cleared
    _setBBRefOrSlinkFather(index.nodeTable, index.leafTable, oldChild, oldChild, true, true);

    //now set the depth of the newly inserted node
    unsigned nodeInt3 = _getDepth(index.nodeTable, headLoc.fatherPos, baseAddressQ) + length(headLoc.rest);
    if (nodeInt3 <= BM::DepthFlatInt3)
    {
        nodeInt3 += BM::NotDeepBitInt3;
    }
    assignValue(nodeIt + 2, nodeInt3);

    //set the head position of the node
    unsigned nodeInt4 = beginPosition(headLoc.rest);
    assignValue(nodeIt + 3, nodeInt4);

    //step 2
    //first step done! the second step is to insert the new leaf
    StKurtzHeadLoc<TText, TNodes, TLeaves> headLocLeafInsertion;
    headLocLeafInsertion.fatherPos = baseAddressQ;
    headLocLeafInsertion.rest = infix(indexText(index), endPosition(headLoc.rest), endPosition(headLoc.rest));        //empty
    //the misMatchCharCausingNodeInsertion is given because some small node headPos might not be set
    //so a firstCharEdge detection of the insertLeaf leads to wrong results
    _insertLeaf(index, headLocLeafInsertion, tailPtr, baseAddressQ);

    ////set the headloc to the new node
    //headLoc.fatherPos = baseAddressQ;
    //setBeginPosition(headLoc.rest, endPosition(headLoc.rest));

}

template<typename TNodes>
inline void
_setFirstChildRef(TNodes & nodeTable, const typename Position<TNodes>::Type & nodePos, unsigned firstChild)
{
    SEQAN_CHECKPOINT;
    SEQAN_ASSERT_GEQ(length(nodeTable), (nodePos + 1) + 1);    //table at least large enough for small node
    typedef StKurtzBitMasks BM;
    typename Iterator<TNodes, Rooted>::Type it = begin(nodeTable);
    setPosition(it, nodePos);
    unsigned integer1 = *it;
    unsigned integer2 = *(it + 1);
    //the highest 2 reference bits go to int2, the remaining ones go to integer1
    integer1 &= (BM::AllBitsSet - BM::FirstChildRefInt1);    //clear old refernce
    integer1 += (BM::FirstChildRefInt1 & firstChild);

    integer2 &= (BM::AllBitsSet - BM::FirstChildRefInt2);
    integer2 += (BM::TwoHighestRawRefBits & firstChild) << BM::ShiftTwoHighestRawRefBitsToInt2;
    ;
    assignValue(it, integer1);
    assignValue(it + 1, integer2);

}

template<typename TNodes, typename TLeaves>
inline void
_setBBRefOrSlinkFather(TNodes & nodeTable, TLeaves & leafTable,
                       const StKurtzTarget<TNodes, TLeaves> & node,
                       const StKurtzTarget<TNodes, TLeaves> & branchBrother,
                       bool setSlinkFather = false,
                       bool clear = false)
{
    typedef StKurtzBitMasks BM;
    unsigned rawReference;
    SEQAN_ASSERT(_nodeValid(branchBrother));
    if (clear)
    {
        SEQAN_ASSERT(setSlinkFather);    //convention and neccessity (nil bit = signal for slink)
        SEQAN_ASSERT(&node == &branchBrother);    //just a convention
        rawReference = 0;
    }
    else if (branchBrother.nodeType == LEAF)
    {
        SEQAN_ASSERT_GEQ(branchBrother.posLeaf, 0u);
        SEQAN_ASSERT_LEQ(branchBrother.posLeaf, length(leafTable));
        rawReference = branchBrother.posLeaf;
        rawReference += BM::RawRefsLeafBit;
        SEQAN_ASSERT_EQ(rawReference & (BM::AllBitsSet - BM::SlinkFatherOrBBRefLeaf), 0u); //not to high
    }
    else
    {    //set reference to a node
        SEQAN_ASSERT_EQ(branchBrother.posNode % 2, 0u);
        rawReference = branchBrother.posNode / 2;
        SEQAN_ASSERT_EQ(rawReference & (BM::AllBitsSet - BM::BBRefOrSlinkFatherInt2), 0u);
    }

    SEQAN_ASSERT(_nodeValid(node));

    if (node.nodeType == LEAF)
    {
        typename Iterator<TLeaves, Rooted>::Type leafIt = begin(leafTable);
        SEQAN_ASSERT_GEQ(node.posLeaf, 0u);
        SEQAN_ASSERT_NOT(leafIt == 0);
        setPosition(leafIt, node.posLeaf);
        unsigned leafValue = value(leafIt);
        leafValue &= BM::AllBitsSet - BM::SlinkFatherOrBBRefLeaf - BM::NotNilBitLeaf;
        leafValue += rawReference;
        if (!setSlinkFather)
        {
            leafValue += BM::NotNilBitLeaf;
        }
        assignValue(leafIt, leafValue);
    }
    else
    {    //set the reference of a node
        typename Iterator<TLeaves, Rooted>::Type nodeIt = begin(nodeTable);
        setPosition(nodeIt, node.posNode + 1);
        unsigned nodeInt2 = value(nodeIt);
        nodeInt2 &= BM::AllBitsSet - BM::BBRefOrSlinkFatherInt2 - BM::NilBitInt2;
        nodeInt2 += rawReference;
        if (setSlinkFather)
        {
            nodeInt2 += BM::NilBitInt2;
        }
        assignValue(nodeIt, nodeInt2);
    }

}

template<typename TNodes, typename TLeaves>
inline void _setOwnSlink(TNodes & nodeTable, TLeaves & leafTable,
                         typename Position<TNodes>::Type posNode,
                         typename Position<TNodes>::Type posTarget)
{
    //this function ever only called on the last inserted node of a chain
    SEQAN_ASSERT_EQ(length(nodeTable), (posNode + 3) + 1);
    typedef StKurtzBitMasks BM;
    typename Iterator<TNodes, Rooted>::Type nodeIt = begin(nodeTable);
    StKurtzTarget<TNodes, TLeaves> node;
    setPosition(nodeIt, posNode);
    unsigned int3 = *(nodeIt + 2);
    unsigned int4 = *(nodeIt + 3);
    SEQAN_ASSERT_EQ(posTarget % 2, 0u);
    unsigned slink = posTarget / 2;
    SEQAN_ASSERT_LEQ(slink, (unsigned) BM::SlinkFatherOrBBRefLeaf);
    if ((int3 & BM::NotDeepBitInt3) == 0)
    {    //not deep bit not set => deep
        unsigned firstChildRef = _getFirstChildRef(nodeTable, posNode);
        node = _followReference(nodeTable, leafTable, firstChildRef);
        while (!_isLastChild(nodeTable, leafTable, node))
        {
            node = _getBBOrSlinkFatherTarget(nodeTable, leafTable, node);
        }
        SEQAN_ASSERT(_nodeValid(node));
        StKurtzTarget<TNodes, TLeaves> targetNode;
        targetNode = _followReference(nodeTable, leafTable, slink);
        _setBBRefOrSlinkFather(nodeTable, leafTable, node, targetNode, true, false);

    }
    else
    {    //flat node
        int3 &= (BM::AllBitsSet - BM::SlinkInt3);    //delete any old slinks //TODO(aumann): necessary or check that not set?
        int3 += ((BM::SlinkInt3 >> BM::ShiftSlinkBitsInt3) & slink) << BM::ShiftSlinkBitsInt3;
        int4 &= (BM::AllBitsSet - BM::SlinkInt4);
        int4 += ((BM::SlinkInt4 >> BM::ShiftSlinkBitsInt4) & slink) << BM::ShiftSlinkBitsInt4;
        //the rest goes to TLeav[headpos]
        typename Iterator<TLeaves, Rooted>::Type leafIt = begin(leafTable);
        setPosition(leafIt, _getHeadPosition(nodeTable, posNode, BM::NoPos));

        unsigned intLeaf = *leafIt;
        intLeaf &= (BM::AllBitsSet - BM::SlinkLeaf);
        intLeaf += ((BM::SlinkLeaf >> BM::ShiftSlinkBitsLeaf) & slink) << BM::ShiftSlinkBitsLeaf;

        assignValue(nodeIt + 2, int3);
        assignValue(nodeIt + 3, int4);
        assignValue(leafIt, intLeaf);

    }
    SEQAN_ASSERT_EQ(_followReference(nodeTable, leafTable, _getOwnSlink(nodeTable, leafTable, posNode)).posNode,
                    posTarget);

}

template<typename TText, typename TConfig, typename TSpec>
inline StKurtzTarget<typename TConfig::NODE_TYPE, typename TConfig::LEAF_TYPE>
_getNodeForEdge(const Index<TText, IndexStKurtz<TConfig, TSpec> > & index,
const StKurtzTarget<typename TConfig::NODE_TYPE, typename TConfig::LEAF_TYPE>& father,
                typename Value<TText>::Type searchChar,
                StKurtzTarget<typename TConfig::NODE_TYPE, typename TConfig::LEAF_TYPE> & leftBrother,
                bool & edgeFound,
                unsigned lastInsertedNodePos = StKurtzBitMasks::NoPos)
{
    SEQAN_CHECKPOINT;
    SEQAN_ASSERT(_nodeValid(father));

    typedef typename TConfig::NODE_TYPE TNodes;
    typedef typename TConfig::LEAF_TYPE TLeaves;
    typedef StKurtzBitMasks BM;

    StKurtzTarget<TNodes, TLeaves> currentChild;
    typename Iterator<TText const, Rooted>::Type textIt = begin(indexText(index));
    typename Value<TText>::Type firstChar;
    currentChild = _followReference(index.nodeTable, index.leafTable,
                                    _getFirstChildRef(index.nodeTable, father.posNode));
    //TEST CONST ZEIGER
    //leftBrother = &currentChild;

    leftBrother = father;    //this would sign that no real leftBrother yet exists
    bool mismatchFinal = false;
    edgeFound = false;
    while (!_isLastChild(index.nodeTable, index.leafTable, currentChild))
    {    //still nodes
        if (currentChild.nodeType == LEAF)
        {
            setPosition(textIt, currentChild.posLeaf
                                + _getDepth(index.nodeTable, father.posNode, lastInsertedNodePos));
        }
        else
        {
            setPosition(textIt, _getHeadPosition(index.nodeTable, currentChild.posNode, lastInsertedNodePos)
                                + _getDepth(index.nodeTable, father.posNode, lastInsertedNodePos));
        }

        firstChar = *textIt;

        if (searchChar > firstChar)
        {
            leftBrother = currentChild;
            currentChild = _getBBOrSlinkFatherTarget(index.nodeTable, index.leafTable, currentChild);
        }
        else if (searchChar == firstChar)
        {
            edgeFound = true;
            break;
        }
        else
        {
            //as soon as the searchCharacter is smaller than the edge one, no match found
            edgeFound = false;
            mismatchFinal = true;
            break;
        }
    }
    if (edgeFound || mismatchFinal)
    {
        SEQAN_ASSERT((edgeFound ? !mismatchFinal : mismatchFinal));    //should be exclusive
        //currentChild is either the owner of the right edge (edgeFound)
        //or the right brother of the leaf that has to be inserted (misMatchFinal)
        return currentChild;
    }

    SEQAN_ASSERT(_isLastChild(index.nodeTable, index.leafTable, currentChild));
    if (currentChild.nodeType == LEAF)
    {
        setPosition(textIt, currentChild.posLeaf + _getDepth(index.nodeTable, father.posNode, lastInsertedNodePos));
    }
    else
    {
        setPosition(textIt, _getHeadPosition(index.nodeTable, currentChild.posNode, lastInsertedNodePos)
                            + _getDepth(index.nodeTable, father.posNode, lastInsertedNodePos));
    }
    firstChar = *textIt;

    if (searchChar == firstChar)
    {
        edgeFound = true;
        return currentChild;
    }
    else if (searchChar < firstChar)
    {
        //the leftbrother is correct, and a leaf would have to inserted to the right of it
        SEQAN_ASSERT_NOT(edgeFound);
        return currentChild;    //as something has to be returned
    }
    else
    {
        leftBrother = currentChild;
        currentChild.posNode = BM::NoPos;
        currentChild.posLeaf = BM::NoPos;
        return currentChild;
    }

}

template<typename TText, typename TConfig, typename TSpec>
inline void
_scanPrefix(const Index<TText, IndexStKurtz<TConfig, TSpec> > & index,
StKurtzHeadLoc<TText, typename TConfig::NODE_TYPE, typename TConfig::LEAF_TYPE> & headLoc,
    StKurtzTailPtr<TText> & tailPtr,
    unsigned lastInsertedNodePos)
{
    //std::cout << "scanPrefix: headLoc.fp: " << headLoc.fatherPos << ". headLoc.beginR: "
    //    << beginPosition(headLoc.rest) << ". headLoc.endR: " << endPosition(headLoc.rest)
    //    << std::endl  << ". headLoc.left.np: "
    //    << headLoc.leftBrother.posNode << ". headLoc.left.lp: " << headLoc.leftBrother.posLeaf
    //    << "." << "tailbegin: " << beginPosition(tailPtr.tail) << "." << std::endl;
    //this function searches the tree for the given tail and sets headLoc and tailPtr as follows
    //(at the start headLoc points to the start node for the search, tailPtr contains the string to search from there on)
    //at the end, headLoc contains 1) the position of the edge where the searched tail ended to match the tree path
    //i.e. the fatherPos contains the father node of this edge, and the leftBrother the left brother of the node
    //to which that edge is incoming. rest contains the part of that edge which matched the searched string
    //or 2) if the matching string ends at a node, this node is stored in fatherPos, the leftBrother's values
    //are undefined, and rest is empty
    //tailPtr contains the tail, i.e. the suffix of the searched string, that was not in the tree

    //CAREFUL WHEN SETTING THE TAIL POSITION!!! ALWAYS SET RELATIVE TO TAIL POSITION, NOT TO A TOTAL POS IN TEXT
    //at a closer look, this should be the same as setting it to the tailIt pos
    typedef typename TConfig::NODE_TYPE TNodes;
    typedef typename TConfig::LEAF_TYPE TLeaves;
    typedef StKurtzBitMasks BM;
    //SEQAN_ASSERT_EQ(length(headLoc.rest), 0);
    typename Value<TText>::Type currentCharEdge;
    typename Value<TText>::Type currentCharTail;
    StKurtzTarget<TNodes, TLeaves> leftBrotherEdgeNode;
    StKurtzTarget<TNodes, TLeaves> edgeNode;
    StKurtzTarget<TNodes, TLeaves> currentNode;
    bool misMatchOccured = false;
    unsigned currentEdgeLength;
    typename Iterator<TText const, Rooted>::Type edgeIt = begin(indexText(index));
    typename Iterator<TText const, Rooted>::Type tailIt = begin(indexText(index));
    typename Position<TText>::Type tailBeginPos;
    typename Position<TText>::Type currentEdgeBeginPos = BM::NoPos;
    typename Position<TText>::Type currentEdgeEndPos = BM::NoPos;

    setPosition(tailIt, beginPosition(tailPtr.tail));
    currentNode.posNode = headLoc.fatherPos;
    currentNode.posLeaf = BM::NoPos;
    if (_getNodeDistance(index.nodeTable, headLoc.fatherPos, lastInsertedNodePos) == 0)
    {
        currentNode.nodeType = BIG_NODE;
    }
    else
    {
        currentNode.nodeType = SMALL_NODE;
    }

    bool anEdgeExists;

    while (!misMatchOccured)
    {
        edgeNode = _getNodeForEdge
                (index, currentNode,
                 *tailIt,
                 leftBrotherEdgeNode, anEdgeExists, lastInsertedNodePos);

        if (anEdgeExists)
        {
            SEQAN_ASSERT(_nodeValid(edgeNode));

            _determineEdgePositions(index, currentNode, edgeNode, lastInsertedNodePos, currentEdgeBeginPos,
                                    currentEdgeEndPos);

            currentEdgeLength = currentEdgeEndPos - currentEdgeBeginPos + 1;
            setPosition(edgeIt, currentEdgeBeginPos);
            tailBeginPos = beginPosition(tailPtr.tail);
            setPosition(tailIt, tailBeginPos);
            SEQAN_ASSERT_EQ(*edgeIt, *tailIt);    //the first characters must match, as the edge must be the right one
            edgeIt++;
            tailIt++;
            while (position(edgeIt) <= currentEdgeEndPos)
            {    //compares at most to the last one of the edge
                //compare characterwise until there is the first mismatch
                currentCharEdge = *edgeIt;
                currentCharTail = *tailIt;
                if (currentCharEdge != currentCharTail)
                {
                    misMatchOccured = true;
                    break;
                }
                edgeIt++;
                tailIt++;
                //TODO(aumann): if scanprefix is reused in a tree algorithm, take out this assertion
                //a tail cannot match to the very end, as no leaf is inserted twice!
                SEQAN_ASSERT(tailIt != end(indexText(index)));
            }
            if (misMatchOccured)
            {
                //the iterators point to the mismatch-positions
                //the misMatch occured in the middle of an edge
                headLoc.fatherPos = currentNode.posNode;
                setBeginPosition(headLoc.rest, tailBeginPos);    //the rest starts with the first tail char
                setEndPosition(headLoc.rest, position(tailIt));            //the rest does not contain the mismatch
                setBeginPosition(tailPtr.tail, tailBeginPos
                                         + (position(edgeIt) - currentEdgeBeginPos));//the tail starts at the mismatch char
                SEQAN_ASSERT_EQ(beginPosition(tailPtr.tail), position(tailIt));
                headLoc.leftBrother = leftBrotherEdgeNode;
            }
            else
            {    //the edgeNode is the starting point for further search
                SEQAN_ASSERT(position(edgeIt) == currentEdgeEndPos + 1);
                headLoc.fatherPos = edgeNode.posNode;
                currentNode = edgeNode;
                setBeginPosition(headLoc.rest, position(edgeIt));
                setEndPosition(headLoc.rest, position(edgeIt));    //empty rest
                setBeginPosition(tailPtr.tail, tailBeginPos + currentEdgeLength);//first character that is not part of the edge
                //recursion would of course be the most elegant solution, but imho performance reasons forbid it
            }
        }
        else
        {
            //the match ends at the currentNode
            headLoc.fatherPos = currentNode.posNode;
            setBeginPosition(headLoc.rest, beginPosition(tailPtr.tail));    //rest is empty
            setEndPosition(headLoc.rest, beginPosition(tailPtr.tail));
            //tailptr is already correct, leftBrother is defined as the left brother of
            //a leaf that would have to inserted now (and should already be set accordingly)
            headLoc.leftBrother = leftBrotherEdgeNode;
            misMatchOccured = true;
        }

    }

}

template<typename TText, typename TConfig, typename TSpec>
inline void
_determineEdgePositions(const Index<TText, IndexStKurtz<TConfig, TSpec> > & index,
const StKurtzTarget<typename TConfig::NODE_TYPE, typename TConfig::LEAF_TYPE> & father,
                        const StKurtzTarget<typename TConfig::NODE_TYPE, typename TConfig::LEAF_TYPE> & child,
                        unsigned lastInsertedNodePos,
                        typename Position<TText>::Type & startPos,
                        typename Position<TText>::Type & endPos)
{

    SEQAN_ASSERT(father.nodeType == BIG_NODE || father.nodeType == SMALL_NODE);
    if (child.nodeType == LEAF)
    {
        startPos = child.posLeaf + _getDepth(index.nodeTable, father.posNode, lastInsertedNodePos);
        endPos = length(indexText(index)) - 1;
    }
    else
    {
        startPos = _getHeadPosition(index.nodeTable, child.posNode, lastInsertedNodePos)
                + _getDepth(index.nodeTable, father.posNode, lastInsertedNodePos);
        endPos = startPos
                + _getDepth(index.nodeTable, child.posNode, lastInsertedNodePos)
                - _getDepth(index.nodeTable, father.posNode, lastInsertedNodePos)
                - 1;
    }

}

template<typename TText, typename TConfig, typename TSpec>
inline void
_rescan(const Index<TText, IndexStKurtz<TConfig, TSpec> > & index,
StKurtzHeadLoc<TText, typename TConfig::NODE_TYPE, typename TConfig::LEAF_TYPE> & headLoc,
    unsigned lastInsertedNodePos)
{
    SEQAN_CHECKPOINT;
    typedef StKurtzBitMasks BM;
    //std::cout << "rescan: headLoc.fp: " << headLoc.fatherPos << ". headLoc.beginR: "
    //    << beginPosition(headLoc.rest) << ". headLoc.endR: " << endPosition(headLoc.rest)
    //    << std::endl  << ". headLoc.left.np: "
    //    << headLoc.leftBrother.posNode << ". headLoc.left.lp: " << headLoc.leftBrother.posLeaf
    //    << "." << std::endl;
    //rescan means that one scans for the headLoc's rest in the tree, from the headLoc's father pos
    //it is known, that this rest is in the tree, i.e. only the first edge char has to be compared

    //TODO(aumann): recheck if an empty headloc is valid, and means that nothing has to be done
    //atm I guess this is actually not allowed. but occurrs many times in the tests yet
    if (length(headLoc.rest) == 0)
    {
        //SEQAN_ASSERT_FALSE;
        //=> invalidate the leftBrother
        headLoc.leftBrother.nodeType = BIG_NODE;
        headLoc.leftBrother.posNode = BM::NoPos;
        headLoc.leftBrother.posLeaf = BM::NoPos;
        return;    //nothing else to do
    }

    typedef typename TConfig::NODE_TYPE TNodes;
    typedef typename TConfig::LEAF_TYPE TLeaves;
    typedef StKurtzBitMasks BM;
    SEQAN_ASSERT_GEQ(length(headLoc.rest), 1u);
//        typename Value<TText>::Type currentCharEdge;
//        typename Value<TText>::Type currentCharRest;
    StKurtzTarget<TNodes, TLeaves> leftBrotherEdgeNode;
    StKurtzTarget<TNodes, TLeaves> edgeNode;
    StKurtzTarget<TNodes, TLeaves> currentNode;
    unsigned currentEdgeLength;
    //typename Iterator<TText const, Rooted>::Type edgeIt = begin(indexText(index));
    typename Iterator<TText const, Rooted>::Type restIt = begin(indexText(index));
    typename Position<TText>::Type restBeginPos;
    typename Position<TText>::Type currentEdgeBeginPos = BM::NoPos;
    typename Position<TText>::Type currentEdgeEndPos = BM::NoPos;

    currentNode.posNode = headLoc.fatherPos;
    currentNode.posLeaf = BM::NoPos;
    if (_getNodeDistance(index.nodeTable, headLoc.fatherPos, lastInsertedNodePos) == 0)
    {
        currentNode.nodeType = BIG_NODE;
    }
    else
    {
        currentNode.nodeType = SMALL_NODE;
    }

    bool endReached = false;
    bool edgeExists = false;
    while (!endReached)
    {
        restBeginPos = beginPosition(headLoc.rest);
        setPosition(restIt, restBeginPos);
        edgeNode = _getNodeForEdge(index, currentNode, *restIt, leftBrotherEdgeNode, edgeExists, lastInsertedNodePos);
        SEQAN_ASSERT(edgeExists);
        SEQAN_ASSERT(_nodeValid(edgeNode));
        _determineEdgePositions(index, currentNode, edgeNode, lastInsertedNodePos, currentEdgeBeginPos,
                                currentEdgeEndPos);
        currentEdgeLength = currentEdgeEndPos - currentEdgeBeginPos + 1;
        if (currentEdgeLength > length(headLoc.rest))
        {
            //the current edge is the end position for the search
            headLoc.leftBrother = leftBrotherEdgeNode;
            headLoc.fatherPos = currentNode.posNode; //remains unchanged
            endReached = true;
        }
        else if (currentEdgeLength == length(headLoc.rest))
        {
            //the edgeNode is the exact end position for the search
            //as the next node inserted will be a leaf, the leftbrother reference is normally used to determine the exact
            //insertion position. in this case this is not possible (will yet have to be searched)
            //=> invalidate the leftBrother
            headLoc.leftBrother.nodeType = BIG_NODE;
            headLoc.leftBrother.posNode = BM::NoPos;
            headLoc.leftBrother.posLeaf = BM::NoPos;

            //cannot end at a leaf
            SEQAN_ASSERT(edgeNode.nodeType == BIG_NODE || edgeNode.nodeType == SMALL_NODE);
            headLoc.fatherPos = edgeNode.posNode;
            setBeginPosition(headLoc.rest, endPosition(headLoc.rest)); //no more rest

            endReached = true;
        }
        else
        {
            currentNode = edgeNode;
            setBeginPosition(headLoc.rest, beginPosition(headLoc.rest) + currentEdgeLength);

        }
    }

}

//temporarily just a wrapper around the other helper functions
//but should replace them eventually
template<typename TNodes, typename TLeaves>
inline StKurtzTarget<TNodes, TLeaves>
_getBBOrSlinkFatherTarget(const TNodes & nodeTable, const TLeaves & leafTable,
                          StKurtzTarget<TNodes, TLeaves> const & node)
{
    SEQAN_ASSERT(_nodeValid(node));
    if (node.nodeType == LEAF)
    {
        return _followReference(nodeTable, leafTable,
                                _getLeafBBRefOrSlinkFather(leafTable, node.posLeaf));
    }
    else
    {
        return _followReference(nodeTable, leafTable,
                                _getNodeBBRefOrSlinkFather(nodeTable, node.posNode));

    }
}

//helper function, that returns true if a node is valid
template<typename TNodes, typename TLeaves>
inline bool
_nodeValid(StKurtzTarget<TNodes, TLeaves> node)
{
    typedef StKurtzBitMasks BM;
    if (!(node.nodeType == BIG_NODE || node.nodeType == SMALL_NODE || node.nodeType == LEAF))
    {
        return false;
    }

    if (node.nodeType == BIG_NODE || node.nodeType == SMALL_NODE)
    {
        return (node.posLeaf == BM::NoPos && node.posNode != BM::NoPos);
    }
    else
    {
        return (node.posNode == BM::NoPos && node.posLeaf != BM::NoPos);
    }

}

//template <typename TNodes, typename TLeaves>
//inline void
//_setLeafBranchBrother(const TLeaves & leafTable, typename const Position<TLeaves>::Type & leafPos,
//                        const StKurtzTarget<TNodes, TLeaves>& branchBrother) {
//    SEQAN_CHECK_POINT;
//    typedef StKurtzBitMasks BM;
//    Iterator<TLeaves, Rooted>::Type leafIt = begin(leafTable);
//    SEQAN_ASSERT_NOT(leafIt == 0);    //iterator valid
//    SEQAN_ASSERT_GEQ(length(leafTable), leafPos);
//    setPosition(leafIt, leafPos);
//    unsigned leafContent = value(leafIt);
//    leafContent &= (BM::AllBitsSet - BM::NotNilBitLeaf - BM::SlinkFatherOrBBRefLeaf);//delete any old ref
//    if (branchBrother.nodeType == LEAF) {
//        leafContent += branchBrother.posLeaf;
//        leafContent += BM::RawRefsLeafBit;
//        leafContent += BM::NotNilBitLeaf;
//    } else {
//        SEQAN_ASSERT_EQ(branchBrother.posNode % 2, 0);    //should always be true
//        leafContent += branchBrother.posNode / 2;
//        leafContent += BM::NotNilBitLeaf;
//    }
//    assignValue(leafIt, leafContent);
//}
//
//template <typename TNodes, typename TLeaves>
//inline void
//_setNodeBBOrSlinkFather(const TNodes & nodeTable, typename const Position<TNodes>::Type & nodePos,
//                        const StKurtzTarget<TNodes, TLeaves>& branchBrother) {
//    typedef StKurtzBitMasks BM;
//    Iterator<TNodes, Rooted>::Type nodeIt = begin(nodeTable);
//    SEQAN_ASSERT_NOT(nodeIt == 0);
//    SEQAN_ASSERT_GEQ(length(nodeTable), nodePos);
//    setPosition(nodeIt, nodePos);
//    unsigned nodeInteger2 = value(nodeIt + 1);
//    nodeInteger2 &= (BM::AllBitsSet - BM::BBRefOrSlinkFatherInt2 - BM::NilBitInt2);
//    if (branchBrother.nodeType == LEAF) {
//        nodeInteger2 += branchBrother.posLeaf + BM::RawRefsLeafBit;
//    } else {
//        SEQAN_ASSERT_EQ(branchBrother.posNode % 2, 0);
//        nodeInteger2 += branchBrother.posNode / 2;
//    }
//
//    assignValue(nodeIt + 1, nodeInteger2);
//}

template <typename TText, typename TSpec>
void _indexRequireTopDownIteration(Index<TText, IndexStKurtz<TSpec> > &index)
{
    indexRequire(index, StKurtzNodeTab());
    indexRequire(index, StKurtzLeafTab());
}

template <typename TText, typename TSpec, typename TSpecAlg>
inline bool indexCreate(Index<TText, IndexStKurtz<TSpec> > &index, StKurtzNodeTab, TSpecAlg const) {
    SEQAN_CHECKPOINT;
    //atm only one algo, so std creation routine is called
    return _createIndex(index);
}

template <typename TText, typename TSpec, typename TSpecAlg>
inline bool indexCreate(Index<TText, IndexStKurtz<TSpec> >  &index, StKurtzLeafTab, TSpecAlg const) {
    SEQAN_CHECKPOINT;
    //atm only one algo, so std creation routine is called
    return _createIndex(index);
}

template <typename TText, typename TSpec>
inline typename Fibre<Index<TText, TSpec> const, StKurtzNodeTab>::Type &
getFibre(Index<TText, TSpec> const &index, StKurtzNodeTab) {
    return index.nodeTable;
}

template <typename TText, typename TSpec>
inline typename Fibre<Index<TText, TSpec> const, StKurtzLeafTab>::Type &
getFibre(Index<TText, TSpec> const &index, StKurtzLeafTab) {
    return index.leafTable;
}

//as soon as operating on StringSets, the text getFibres can be removed (index_shims.h should provide)
template <typename TText, typename TSpec>
inline typename Fibre<Index<TText, IndexStKurtz<TSpec> >, FibreText>::Type &
getFibre(Index<TText, IndexStKurtz<TSpec> > &index, FibreText) {
    return value(index.text);
}

template <typename TText, typename TSpec>
inline typename Fibre<Index<TText, IndexStKurtz<TSpec> > const, FibreText>::Type &
getFibre(Index<TText, IndexStKurtz<TSpec> > const &index, FibreText) {
    return value(index.text);
}

template <typename TText, typename TSpec>
inline typename Fibre<Index<TText, TSpec>, FibreRawText>::Type &
getFibre(Index<TText, IndexStKurtz<TSpec> > &index, FibreRawText) {
    return value(index.text);
}

template <typename TText, typename TSpec>
inline typename Fibre<Index<TText, TSpec> const, FibreRawText>::Type &
getFibre(Index<TText, IndexStKurtz<TSpec> > const &index, FibreRawText) {
    return value(index.text);
}

}  // namespace seqan

#endif  // SANDBOX_TUM_INCLUDE_SEQAN_INDEX_SUFFIX_TREES_INDEX_STKURTZ_BASE_H_
