// ==========================================================================
//                             index_suffix_trees
// ==========================================================================
// Copyright (c) 2006-2011, Knut Reinert, FU Berlin
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * Neither the name of Knut Reinert or the FU Berlin nor the names of
//       its contributors may be used to endorse or promote products derived
//       from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL KNUT REINERT OR THE FU BERLIN BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
// DAMAGE.
//
// ==========================================================================
// Author: Alexander Aumann <a.aumann@web.de>
// ==========================================================================
// Tests for the SeqAn module index_suffix_trees.
// ==========================================================================

#ifndef SANDBOX_TUM_TESTS_INDEX_SUFFIX_TREES_TEST_INDEX_STKURTZ_H_
#define SANDBOX_TUM_TESTS_INDEX_SUFFIX_TREES_TEST_INDEX_STKURTZ_H_

#include <seqan/basic.h>
#include <seqan/sequence.h>
#include <seqan/index.h>

#include <seqan/index_suffix_trees.h>

namespace seqan {

typedef StKurtzBigNode   BigNode;
typedef StKurtzSmallNode SmallNode;
typedef StKurtzLeafNode  LeafNode;
typedef StKurtzBitMasks  BM;

String<unsigned> buildLeafTableABAB() {
    LeafNode ln0, ln1, ln2, ln3, ln4;
    ln0.rightBrother = 2;
    ln0.rightBrother += BM::NotNilBitLeaf;      //da nicht nil, das nil-Bit setzen
    ln0.rightBrother += BM::RawRefsLeafBit;     //Verweis auf ein Blatt

    ln1.rightBrother = 3;
    ln1.rightBrother += BM::NotNilBitLeaf;
    ln1.rightBrother += BM::RawRefsLeafBit;

    ln2.rightBrother = 0;    //nil
    ln3.rightBrother = 0;
    ln4.rightBrother = 0;

    String<unsigned> leafTable;
    appendValue(leafTable, ln0.rightBrother);
    appendValue(leafTable, ln1.rightBrother);
    appendValue(leafTable, ln2.rightBrother);
    appendValue(leafTable, ln3.rightBrother);
    appendValue(leafTable, ln4.rightBrother);

    return leafTable;
}

seqan::String<unsigned> buildNodeTableABAB() {
    BigNode root, bn2;
    SmallNode sn1;

    unsigned reference = 2;                     //Verweis auf 4. Integer in node-Tabelle
    //Aufspalten: 27 Bits nach Int1, 2 höchste Bits nach Int2 (hier obsolet, da nicht vorhanden)
    root.integer1 = reference;
    root.integer2 = 0;                          //kein slink/BB, zwei höchsten bits des firstchild 0, sonderfall
    root.integer3 = 0 + BM::NotDeepBitInt3;     //Wurzel hat keinen suffix link, und ist nicht tief, Tiefe ist 0
        //Integer 3 sieht exklusiv nur bei der Wurzel so aus!
    root.integer4 = 0;                          //kein slink und headposition der wurzel ist 0

    sn1.integer1 = 1 << 27;                     //distance des kleinen knoten zum rechtesten Knoten der Kette
    reference = 0;                              //first child reference zu leaf 0
    reference += BM::RawRefsLeafBit;            //_leaf_ 0
    sn1.integer1 += reference & BM::FirstChildRefInt1;     //die niedrigsten Bits gehen in Int 1
    sn1.integer2 = (reference & BM::TwoHighestRawRefBits) << BM::ShiftTwoHighestRawRefBitsToInt2;
    sn1.integer2 += 3;                          //BB Referenz zu Knoten an Speicherstelle 6

    reference = 1;                              //first child ref to leaf 1
    reference += BM::RawRefsLeafBit;
    bn2.integer1 = reference & BM::FirstChildRefInt1;
    bn2.integer2 = (reference & BM::TwoHighestRawRefBits) << BM::ShiftTwoHighestRawRefBitsToInt2;
    reference = 4;                              //construct ref to leaf 4
    reference += BM::RawRefsLeafBit;
    bn2.integer2 += reference;
    //normally leaf nr.4 would have to store the slink of the father of the chain sn1-bn2-leaf4
    //but as this is the root which has no slink, that's obsolete
    bn2.integer3 = 0;
    bn2.integer3 += BM::NotDeepBitInt3;         //node isn't deeper than 2^10-1
    bn2.integer3 += 1;                          //the depth of the node
    reference = 0;                              //construct suffix link, goes to the root
    //slink is distributed between int3, int4 and leaf[headposition], but as its zero, obsolete
    bn2.integer4 = 3;                           //the headposition

    String<unsigned> branchTable;
    appendValue(branchTable, root.integer1);
    appendValue(branchTable, root.integer2);
    appendValue(branchTable, root.integer3);
    appendValue(branchTable, root.integer4);
    appendValue(branchTable, sn1.integer1);
    appendValue(branchTable, sn1.integer2);
    appendValue(branchTable, bn2.integer1);
    appendValue(branchTable, bn2.integer2);
    appendValue(branchTable, bn2.integer3);
    appendValue(branchTable, bn2.integer4);
    return branchTable;
}

struct TestSample {
    unsigned rawRef;
    StKurtzNodeType nodeType;
    Position<String<unsigned> >::Type posNode;
    Position<String<unsigned> >::Type posLeaf;

    TestSample():
        rawRef(0xFFFFFFFF), nodeType(BIG_NODE), posNode(1000), posLeaf(1000) {}
    TestSample(unsigned rawRef, StKurtzNodeType type, Position<String<unsigned> >::Type posNode,
                Position<String<unsigned> >::Type posLeaf) :
        rawRef(rawRef), nodeType(type), posNode(posNode), posLeaf(posLeaf) {}

};

CharString unsignedAsBitString(unsigned);

void printTableBits(seqan::String<unsigned> table) {
    Iterator<String<unsigned> , Rooted >::Type it;
    unsigned int currentInt;
    for (it = begin(table); it != end(table); ++it) {
        std::cout << std::setw(2) << (int) position(it) << " :";
        currentInt = *it;
        //Print bit by bit
        std::cout << unsignedAsBitString(currentInt);
        std::cout << std::endl;
    }
}

void setUpComparisonTables(String<unsigned> & compNodeTable, String<unsigned> & compLeafTable,
                                const CharString & comparisonStringTable) {
    Iterator<const CharString>::Type it;
    clear(compNodeTable);
    clear(compLeafTable);
    unsigned currentValue = 0;
    int count = 0;
    bool isStillLeafTable = true;
    bool firstColon = true;
    char currentChar;
    for (it = begin(comparisonStringTable);    it != end(comparisonStringTable); it++) {
        currentChar = *it;
        if (currentChar == ':') {
            if (!firstColon) {
                SEQAN_ASSERT_EQ(count, 32);
                if (isStillLeafTable) {
                    appendValue(compLeafTable, currentValue);
                    currentValue = 0;
                } else {
                    appendValue(compNodeTable, currentValue);
                    currentValue = 0;
                }
            } else {
                firstColon = false;
            }
            count = 0;
            SEQAN_ASSERT(*(it + 1 ) == '0' || *(it + 1) == '1');
        } else if (currentChar == '-' && isStillLeafTable) {
            SEQAN_ASSERT_EQ(count, 32);
            isStillLeafTable = false;
            appendValue(compLeafTable, currentValue);
            currentValue = 0;
            firstColon = true;
        } else if (currentChar == '1') {
            if (count < 31) {
                count++;
                currentValue+= 1;
                currentValue*=2;
            } else if (count == 31) {
                count++;
                currentValue+= 1;
            }
        } else if (currentChar == '0') {
            if (count < 31) {
                currentValue*=2;
                count++;
            } else if (count == 31) {
                count++;
            }
        }
    }

    SEQAN_ASSERT_EQ(count , 32);
    appendValue(compNodeTable, currentValue);

}

CharString unsignedAsBitString (unsigned val) {
    CharString retVal;
    char digit;
    unsigned digitMask;
    for (int i = 0; i < 32; i++) {
        digitMask = 1 << (31 - i);
        digit = ((digitMask & val) >> (31 - i)) + 48;
        appendValue(retVal, digit);
    }
    return retVal;
}

bool tablesEqual (const String<unsigned> & nodeTable, const String<unsigned> & leafTable,
                    const String<unsigned> & compNodeTable, const String<unsigned> & compLeafTable) {
    if (length(nodeTable) != length(compNodeTable)) {
        std::cout << "Length node table == " << length(nodeTable)
                <<    " != " << length(compNodeTable) << " == length comparison node table"
                << std::endl;
        return false;
    }
    if (length(leafTable) != length(compLeafTable)) {
        std::cout << "Length leaf table == " << length(leafTable)
                <<    " != " << length(compLeafTable) << " == length comparison leaf table"
                << std::endl;
        return false;
    }

    Iterator<String<unsigned> const, Rooted>::Type it = begin(nodeTable);
    Iterator<String<unsigned> const, Rooted>::Type compIt = begin(compNodeTable);
    while (it != end(nodeTable)) {
        if (*it != *compIt) {
            std::cout << std::endl << "Node Table inequality at position " << position(it)
                    << std::endl;
            std::cout << "Node Table:\t" << unsignedAsBitString(*it) << std::endl;
            std::cout << "Comp Table:\t" << unsignedAsBitString(*compIt) << std::endl;

            std::cout << "The full tables are: " << std::endl;
            std::cout << "Index tables:" << std::endl;
            printTableBits(leafTable);
            std::cout << "----------------------" << std::endl;
            printTableBits(nodeTable);

            std::cout << "Comparision tables:" << std::endl;
            printTableBits(compLeafTable);
            std::cout << "----------------------" << std::endl;
            printTableBits(compNodeTable);
            return false;
        }
        it++;
        compIt++;
    }

    it = begin(leafTable);
    compIt = begin(compLeafTable);
    while (it != end(leafTable)) {
        if (*it != *compIt) {
            std::cout << std::endl << "Leaf Table inequality at position " << position(it)
                    << std::endl;
            std::cout << "Leaf Table:\t" << unsignedAsBitString(*it) << std::endl;
            std::cout << "Comp Table:\t" << unsignedAsBitString(*compIt) << std::endl;
            std::cout << "Index tables:" << std::endl;
            printTableBits(leafTable);
            std::cout << "----------------------" << std::endl;
            printTableBits(nodeTable);

            std::cout << "Comparision tables:" << std::endl;
            printTableBits(compLeafTable);
            std::cout << "----------------------" << std::endl;
            return false;
        }
        it++;
        compIt++;
    }

    return true;
}


SEQAN_DEFINE_TEST(test_index_stkurtz_getNodeDistance)
{
    using namespace seqan;
    String<unsigned> tableLeaf = buildLeafTableABAB();
    String<unsigned> tableNodes = buildNodeTableABAB();

    std::cout << std::endl;
    std::cout << "The suffix tree abab$ that is used for testing the helper functions." << std::endl;
    std::cout << "Note: This tree assumes that $ is greater than a or b, in reality false." << std::endl;
    std::cout << "Note2: tests for insertLeaf and inserts (onwards) create their own tables." << std::endl;
    std::cout << "Leaf Table: " << std::endl;
    printTableBits(tableLeaf);
    std::cout << std::endl << "Node Table: " << std::endl;
    printTableBits(tableNodes);
    std::cout << std::endl;

    SEQAN_ASSERT_EQ(_getNodeDistance<String<unsigned> >(tableNodes, 0), 0u);
    SEQAN_ASSERT_EQ(_getNodeDistance<String<unsigned> >(tableNodes, 4), 1u);
    SEQAN_ASSERT_EQ(_getNodeDistance<String<unsigned> >(tableNodes, 6), 0u);
}

SEQAN_DEFINE_TEST(test_index_stkurtz_getFirstChildRef)
{
    using namespace seqan;
    typedef seqan::StKurtzBitMasks BM;
    //String<unsigned> tableLeaf = buildLeafTableABAB();
    String<unsigned> tableNodes = buildNodeTableABAB();
    SEQAN_ASSERT_EQ(_getFirstChildRef<String<unsigned> >(tableNodes, 0) , 2u);//raw value 2 (node 4)
    SEQAN_ASSERT_EQ(_getFirstChildRef<String<unsigned> >(tableNodes, 4) ,
            0 + BM::RawRefsLeafBit);
    SEQAN_ASSERT_EQ(_getFirstChildRef<String<unsigned> >(tableNodes, 6) ,
            1 + BM::RawRefsLeafBit);

}

SEQAN_DEFINE_TEST(test_index_stkurtz_getNodeBBRefOrSlinkFather)
{
    using namespace seqan;
    typedef seqan::StKurtzBitMasks BM;
    String<unsigned> tableNodes = buildNodeTableABAB();
    //TODO(aumann): assert that functions like these are not even called for the root node
    SEQAN_ASSERT_EQ(_getNodeBBRefOrSlinkFather<String<unsigned> >(tableNodes, 0) , 0u);
    SEQAN_ASSERT_EQ(_getNodeBBRefOrSlinkFather<String<unsigned> >(tableNodes, 4) , 3u);
    SEQAN_ASSERT_EQ(_getNodeBBRefOrSlinkFather<String<unsigned> >(tableNodes, 6) ,
            4 + BM::RawRefsLeafBit);
}

SEQAN_DEFINE_TEST(test_index_stkurtz_getLeafBBRefOrSlinkFather)
{
    using namespace seqan;
    typedef seqan::StKurtzBitMasks BM;
    String<unsigned> tableLeaf = buildLeafTableABAB();
    SEQAN_ASSERT_EQ(_getLeafBBRefOrSlinkFather<String<unsigned> >(tableLeaf, 0) ,
            2 + BM::RawRefsLeafBit);
    SEQAN_ASSERT_EQ(_getLeafBBRefOrSlinkFather<String<unsigned> >(tableLeaf, 1) ,
            3 + BM::RawRefsLeafBit);
    SEQAN_ASSERT_EQ(_getLeafBBRefOrSlinkFather<String<unsigned> >(tableLeaf, 2) ,
            0u);
    SEQAN_ASSERT_EQ(_getLeafBBRefOrSlinkFather<String<unsigned> >(tableLeaf, 3) ,
            0u);
    SEQAN_ASSERT_EQ(_getLeafBBRefOrSlinkFather<String<unsigned> >(tableLeaf, 4) ,
            0u);
}

SEQAN_DEFINE_TEST(test_index_stkurtz_getDepth)
{
    using namespace seqan;
    String<unsigned> tableNodes = buildNodeTableABAB();

    SEQAN_ASSERT_EQ(_getDepth(tableNodes, 0, StKurtzBitMasks::NoPos), 0u);
    SEQAN_ASSERT_EQ(_getDepth(tableNodes, 4, StKurtzBitMasks::NoPos), 2u);
    SEQAN_ASSERT_EQ(_getDepth(tableNodes, 6, StKurtzBitMasks::NoPos), 1u);
}


SEQAN_DEFINE_TEST(test_index_stkurtz_getHeadPosition)
{
    using namespace seqan;
    String<unsigned> tableNodes = buildNodeTableABAB();

    SEQAN_ASSERT_EQ(_getHeadPosition(tableNodes, 0, StKurtzBitMasks::NoPos), 0u);
    SEQAN_ASSERT_EQ(_getHeadPosition(tableNodes, 4, StKurtzBitMasks::NoPos), 2u);
    SEQAN_ASSERT_EQ(_getHeadPosition(tableNodes, 6, StKurtzBitMasks::NoPos), 3u);
}


SEQAN_DEFINE_TEST(test_index_stkurtz_followReference)
{
    using namespace seqan;
    typedef StKurtzBitMasks BM;
    String<unsigned> tableLeaf = buildLeafTableABAB();
    String<unsigned> tableNodes = buildNodeTableABAB();
    StKurtzTarget<String<unsigned>, String<unsigned> > target;

//    struct TestSample {
//        unsigned rawRef;
//        StKurtzNodeType nodeType;
//        Position<String<unsigned> >::Type posNode;
//        Position<String<unsigned> >::Type posLeaf;
//
//        TestSample():
//            rawRef(0xFFFFFFFF), nodeType(BIG_NODE), posNode(1000), posLeaf(1000) {}
//        TestSample(unsigned rawRef, StKurtzNodeType type, Position<String<unsigned> >::Type posNode,
//                    Position<String<unsigned> >::Type posLeaf) :
//            rawRef(rawRef), nodeType(type), posNode(posNode), posLeaf(posLeaf) {}
//
//    };

    std::list<TestSample> sampleArray;

    TestSample currentSample;
    //test raw references to nodes
    sampleArray.push_back(TestSample(0, BIG_NODE, 0, BM::NoPos));
    sampleArray.push_back(TestSample(2, SMALL_NODE, 4, BM::NoPos));
    sampleArray.push_back(TestSample(3, BIG_NODE, 6, BM::NoPos));
    //test references to leaves
    sampleArray.push_back(TestSample(0 + BM::RawRefsLeafBit, LEAF, BM::NoPos, 0));
    sampleArray.push_back(TestSample(1 + BM::RawRefsLeafBit, LEAF, BM::NoPos, 1));
    sampleArray.push_back(TestSample(2 + BM::RawRefsLeafBit, LEAF, BM::NoPos, 2));
    sampleArray.push_back(TestSample(3 + BM::RawRefsLeafBit, LEAF, BM::NoPos, 3));
    sampleArray.push_back(TestSample(4 + BM::RawRefsLeafBit, LEAF, BM::NoPos, 4));

    std::list<TestSample>::iterator it;
    for (it = sampleArray.begin(); it != sampleArray.end(); ++it) {
        currentSample = *it;
        target = _followReference(tableNodes, tableLeaf, currentSample.rawRef);
        SEQAN_ASSERT_EQ(currentSample.nodeType, target.nodeType);
        SEQAN_ASSERT_EQ(currentSample.posNode, target.posNode);
        SEQAN_ASSERT_EQ(currentSample.posLeaf, target.posLeaf);
    }
}

SEQAN_DEFINE_TEST(test_index_stkurtz_isLastChild)
{
    using namespace seqan;
    typedef StKurtzBitMasks BM;
    String<unsigned> tableLeaf = buildLeafTableABAB();
    String<unsigned> tableNodes = buildNodeTableABAB();
    StKurtzTarget<String<unsigned>, String<unsigned> > target;

    target.nodeType = BIG_NODE;
    target.posNode = 4;
    target.posLeaf = BM::NoPos;
    SEQAN_ASSERT_NOT(_isLastChild(tableNodes, tableLeaf, target));

    target.nodeType = SMALL_NODE;
    target.posNode = 6;
    SEQAN_ASSERT_NOT(_isLastChild(tableNodes, tableLeaf, target));

    target.nodeType = LEAF;
    target.posNode = BM::NoPos;
    target.posLeaf = 4;
    SEQAN_ASSERT(_isLastChild(tableNodes, tableLeaf, target));

    target.posLeaf = 0;
    SEQAN_ASSERT_NOT(_isLastChild(tableNodes, tableLeaf, target));

    target.posLeaf = 2;
    SEQAN_ASSERT(_isLastChild(tableNodes, tableLeaf, target));

    target.posLeaf = 1;
    SEQAN_ASSERT_NOT(_isLastChild(tableNodes, tableLeaf, target));

    target.posLeaf = 3;
    SEQAN_ASSERT(_isLastChild(tableNodes, tableLeaf, target));
}


SEQAN_DEFINE_TEST(test_index_stkurtz_getOwnSlink)
{
    //TODO(aumann): unbedingt auch noch für einen Baum mit tiefen Knoten testen!
    using namespace seqan;
    //typedef StKurtzBitMasks BM;
    String<unsigned> tableLeaf = buildLeafTableABAB();
    String<unsigned> tableNodes = buildNodeTableABAB();

    SEQAN_ASSERT_EQ(_getOwnSlink(tableNodes, tableLeaf, 4), 3u);
    SEQAN_ASSERT_EQ(_getOwnSlink(tableNodes, tableLeaf, 6), 0u);
}

SEQAN_DEFINE_TEST(test_index_stkurtz_insertLeaf)
{
    using namespace seqan;
    CharString text = "dcab$";
    Index<CharString, IndexStKurtz<> > stkIndex(text);
    _initIndexStKurtz(stkIndex);
    CharString comparisonCharTable;
    String<unsigned> comparisonNodeTable;
    String<unsigned> comparisonLeafTable;

    StKurtzHeadLoc<CharString, String<unsigned>, String<unsigned> > headloc;
    headloc.fatherPos = 0;
    headloc.rest = infix(text, 0, 0);
    StKurtzTailPtr<CharString> tailptr;
    tailptr.tail = suffix(text,0);

    _insertLeaf(stkIndex, headloc, tailptr);
    comparisonCharTable =
            "0 :00000000000000000000000000000000\
             1 :00000000000000000000000000000000\
             2 :00000000000000000000000000000000\
             3 :00000000000000000000000000000000\
             4 :00000000000000000000000000000000\
             ------------------\
             0 :00000000000000000000000000000000\
             1 :10000000000000000000000000000000\
             2 :10000000000000000000000000000000\
             3 :00000000000000000000000000000000";
    setUpComparisonTables(comparisonNodeTable, comparisonLeafTable, comparisonCharTable);
    SEQAN_ASSERT(tablesEqual(stkIndex.nodeTable, stkIndex.leafTable,
                            comparisonNodeTable, comparisonLeafTable));


    //insert the leaf "cab$"
    setBeginPosition(tailptr.tail, 1);
    _insertLeaf(stkIndex, headloc, tailptr);
    comparisonCharTable =
            "0 :00000000000000000000000000000000\
             1 :00110000000000000000000000000000\
             2 :00000000000000000000000000000000\
             3 :00000000000000000000000000000000\
             4 :00000000000000000000000000000000\
             -------------\
             0 :00000000000000000000000000000001\
             1 :10000000000000000000000000000000\
             2 :10000000000000000000000000000000\
             3 :00000000000000000000000000000000";
    setUpComparisonTables(comparisonNodeTable, comparisonLeafTable, comparisonCharTable);
    SEQAN_ASSERT(tablesEqual(stkIndex.nodeTable, stkIndex.leafTable,
                            comparisonNodeTable, comparisonLeafTable));

    //leaf ab$
    setBeginPosition(tailptr.tail, 2);
    _insertLeaf(stkIndex, headloc, tailptr);
    comparisonCharTable =
            "0 :00000000000000000000000000000000\
             1 :00110000000000000000000000000000\
             2 :00110000000000000000000000000001\
             3 :00000000000000000000000000000000\
             4 :00000000000000000000000000000000\
            ------------\
             0 :00000000000000000000000000000010\
             1 :10000000000000000000000000000000\
             2 :10000000000000000000000000000000\
             3 :00000000000000000000000000000000";
    setUpComparisonTables(comparisonNodeTable, comparisonLeafTable, comparisonCharTable);
    SEQAN_ASSERT(tablesEqual(stkIndex.nodeTable, stkIndex.leafTable,
                            comparisonNodeTable, comparisonLeafTable));

    //leaf b$
    setBeginPosition(tailptr.tail, 3);
    _insertLeaf(stkIndex, headloc, tailptr);
    comparisonCharTable =
            "0 :00000000000000000000000000000000\
             1 :00110000000000000000000000000000\
             2 :00110000000000000000000000000011\
             3 :00110000000000000000000000000001\
             4 :00000000000000000000000000000000\
             ------------\
             0 :00000000000000000000000000000010\
             1 :10000000000000000000000000000000\
             2 :10000000000000000000000000000000\
             3 :00000000000000000000000000000000";
    setUpComparisonTables(comparisonNodeTable, comparisonLeafTable, comparisonCharTable);
    SEQAN_ASSERT(tablesEqual(stkIndex.nodeTable, stkIndex.leafTable,
                            comparisonNodeTable, comparisonLeafTable));

    //leaf $
    setBeginPosition(tailptr.tail, 4);
    _insertLeaf(stkIndex, headloc, tailptr);
    comparisonCharTable =
            "0 :00000000000000000000000000000000\
             1 :00110000000000000000000000000000\
             2 :00110000000000000000000000000011\
             3 :00110000000000000000000000000001\
             4 :00110000000000000000000000000010\
             ------------\
             0 :00000000000000000000000000000100\
             1 :10000000000000000000000000000000\
             2 :10000000000000000000000000000000\
             3 :00000000000000000000000000000000";
    setUpComparisonTables(comparisonNodeTable, comparisonLeafTable, comparisonCharTable);
    SEQAN_ASSERT(tablesEqual(stkIndex.nodeTable, stkIndex.leafTable,
                            comparisonNodeTable, comparisonLeafTable));
}


SEQAN_DEFINE_TEST(test_index_stkurtz_inserts)
{
    using namespace seqan;
    String<unsigned> compLeafTable;
    String<unsigned> compNodeTable;
    CharString comparisonStringTable;
    CharString text = "abab$";
    Index<CharString, IndexStKurtz<> > stkIndex(text);
    _initIndexStKurtz(stkIndex);

    StKurtzHeadLoc<CharString, String<unsigned>, String<unsigned> > headloc;
    StKurtzTailPtr<CharString> tailptr;

    headloc.fatherPos = 0;                //leaf abab$
    headloc.rest = infix(text, 0, 0);
    tailptr.tail = suffix(text, 0);
    _insertLeaf(stkIndex, headloc, tailptr);

    comparisonStringTable =
            "0 :00000000000000000000000000000000\
             1 :00000000000000000000000000000000\
             2 :00000000000000000000000000000000\
             3 :00000000000000000000000000000000\
             4 :00000000000000000000000000000000\
            ----------------------\
             0 :00000000000000000000000000000000\
             1 :10000000000000000000000000000000\
             2 :10000000000000000000000000000000\
             3 :00000000000000000000000000000000";


    setUpComparisonTables(compNodeTable, compLeafTable, comparisonStringTable);
    SEQAN_ASSERT(tablesEqual(stkIndex.nodeTable, stkIndex.leafTable, compNodeTable, compLeafTable));


    headloc.fatherPos = 0;                //leaf bab$
    setBeginPosition(headloc.rest, 0);
    setEndPosition(headloc.rest, 0);
    setBeginPosition(tailptr.tail, 1);
    _insertLeaf(stkIndex, headloc, tailptr);

    comparisonStringTable =
            "0 :00110000000000000000000000000001\
             1 :00000000000000000000000000000000\
             2 :00000000000000000000000000000000\
             3 :00000000000000000000000000000000\
             4 :00000000000000000000000000000000\
            ----------------------\
             0 :00000000000000000000000000000000\
             1 :10000000000000000000000000000000\
             2 :10000000000000000000000000000000\
             3 :00000000000000000000000000000000";

    setUpComparisonTables(compNodeTable, compLeafTable, comparisonStringTable);
    SEQAN_ASSERT(tablesEqual(stkIndex.nodeTable, stkIndex.leafTable, compNodeTable, compLeafTable));

    headloc.fatherPos = 0;                //node ab
    headloc.leftBrother.nodeType = BIG_NODE;
    headloc.leftBrother.posNode = headloc.fatherPos;    //signal that new node splits the first child
    headloc.leftBrother.posLeaf = BM::NoPos;
    setBeginPosition(headloc.rest, 2);
    setEndPosition(headloc.rest, 4);
    setBeginPosition(tailptr.tail, 4);    //only $

    appendValue(stkIndex.nodeTable, 0);    //always two empty fields available before node insertion
    appendValue(stkIndex.nodeTable, 0);
    _insertNode(stkIndex, 4, headloc, tailptr);

    comparisonStringTable =
            "0 :00000000000000000000000000000000\
             1 :00000000000000000000000000000000\
             2 :00110000000000000000000000000000\
             3 :00000000000000000000000000000000\
             4 :00000000000000000000000000000000\
            ----------------------\
             0 :00000000000000000000000000000010\
             1 :00000000000000000000000000000000\
             2 :10000000000000000000000000000000\
             3 :00000000000000000000000000000000\
             4 :00000000000000000000000000000010\
             5 :10010000000000000000000000000001\
             6 :10000000000000000000000000000010\
             7 :00000000000000000000000000000010";

    setUpComparisonTables(compNodeTable, compLeafTable, comparisonStringTable);
    SEQAN_ASSERT(tablesEqual(stkIndex.nodeTable, stkIndex.leafTable, compNodeTable, compLeafTable));

    headloc.fatherPos = 0;                //node b
    headloc.leftBrother.nodeType = SMALL_NODE;
    headloc.leftBrother.posNode = 4;
    headloc.leftBrother.posLeaf = BM::NoPos;
    setBeginPosition(headloc.rest, 3);
    setEndPosition(headloc.rest, 4);
    setBeginPosition(tailptr.tail, 4);    //only $
    _insertNode(stkIndex, 6, headloc, tailptr);

    comparisonStringTable =
            "0 :00000000000000000000000000000000\
             1 :00000000000000000000000000000000\
             2 :00110000000000000000000000000000\
             3 :00110000000000000000000000000001\
             4 :00000000000000000000000000000000\
            ----------------------\
             0 :00000000000000000000000000000010\
             1 :00000000000000000000000000000000\
             2 :10000000000000000000000000000000\
             3 :00000000000000000000000000000000\
             4 :00000000000000000000000000000010\
             5 :10000000000000000000000000000011\
             6 :00000000000000000000000000000011\
             7 :10100000000000000000000000000000\
             8 :10000000000000000000000000000001\
             9 :00000000000000000000000000000011";


    setUpComparisonTables(compNodeTable, compLeafTable, comparisonStringTable);
    SEQAN_ASSERT(tablesEqual(stkIndex.nodeTable, stkIndex.leafTable, compNodeTable, compLeafTable));

    headloc.fatherPos = 0;                                    //leaf $
    setBeginPosition(headloc.rest, 0);
    setEndPosition(headloc.rest, 0);
    setBeginPosition(tailptr.tail, 4); //only $
    _insertLeaf(stkIndex, headloc, tailptr);

    comparisonStringTable =
            "0 :00000000000000000000000000000000\
             1 :00000000000000000000000000000000\
             2 :00110000000000000000000000000000\
             3 :00110000000000000000000000000001\
             4 :00100000000000000000000000000010\
             ----------------------\
             0 :00000000000000000000000000000100\
             1 :10000000000000000000000000000000\
             2 :10000000000000000000000000000000\
             3 :00000000000000000000000000000000\
             4 :00000000000000000000000000000010\
             5 :10000000000000000000000000000011\
             6 :00000000000000000000000000000011\
             7 :10100000000000000000000000000000\
             8 :10000000000000000000000000000001\
             9 :00000000000000000000000000000011";

    setUpComparisonTables(compNodeTable, compLeafTable, comparisonStringTable);
    SEQAN_ASSERT(tablesEqual(stkIndex.nodeTable, stkIndex.leafTable, compNodeTable, compLeafTable));
}

SEQAN_DEFINE_TEST(test_index_stkurtz_scanPrefix)
{
    using namespace seqan;
    //table for the abab$ tree
    CharString comparisonStringTable;
    typedef StKurtzBitMasks BM;

    String<unsigned> compNodeTable; String<unsigned> compLeafTable;
    CharString text = "abab$abac";
    typedef StKurtzBitMasks BM;


    Index<CharString, IndexStKurtz<> > stkIndex(text);
    _initIndexStKurtz(stkIndex);
    StKurtzHeadLoc<CharString, String<unsigned>, String<unsigned> > headloc;
    StKurtzTailPtr<CharString> tailptr;

    headloc.fatherPos = 0;                //leaf abab$
    headloc.rest = infix(text, 0, 0);
    tailptr.tail = suffix(text, 0);
    _insertLeaf(stkIndex, headloc, tailptr);

    //scan bab$ in this tree, from the root

    headloc.fatherPos = 0;
    setBeginPosition(headloc.rest, 0);
    setEndPosition(headloc.rest, 0);
    headloc.leftBrother.posNode = BM::NoPos;
    headloc.leftBrother.posLeaf = BM::NoPos;
    headloc.leftBrother.nodeType = BIG_NODE;

    setBeginPosition(tailptr.tail , 1);
    _scanPrefix(stkIndex, headloc, tailptr, BM::NoPos);
    //no character can be followed, thus only the left brother should change
    SEQAN_ASSERT_EQ(headloc.fatherPos , 0u);
    SEQAN_ASSERT_EQ(length(headloc.rest) , 0u);
    SEQAN_ASSERT_EQ(beginPosition(tailptr.tail) , 1u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.nodeType , LEAF);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posLeaf , 0u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posNode , (unsigned) BM::NoPos);

    _insertLeaf(stkIndex, headloc, tailptr, BM::NoPos);

    setBeginPosition(tailptr.tail, 2);
    headloc.leftBrother.posNode = BM::NoPos;
    headloc.leftBrother.posLeaf = BM::NoPos;
    headloc.leftBrother.nodeType = BIG_NODE;

    _scanPrefix(stkIndex, headloc, tailptr, BM::NoPos);
    SEQAN_ASSERT_EQ(headloc.fatherPos , 0u);
    SEQAN_ASSERT_EQ(length(headloc.rest) , 2u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posNode , 0u);    //first child of father need be splitted
    SEQAN_ASSERT_EQ(headloc.leftBrother.nodeType , BIG_NODE);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posLeaf , (unsigned) BM::NoPos);
    SEQAN_ASSERT_EQ(beginPosition(tailptr.tail) , 4u);

    appendValue(stkIndex.nodeTable, 0);    //always two empty fields available before node insertion
    appendValue(stkIndex.nodeTable, 0);
    _insertNode(stkIndex, 4, headloc, tailptr);

    //set distance of the small node to max
    assignValue(stkIndex.nodeTable, 4, value(stkIndex.nodeTable, 4) + (31 << BM::ShiftDistanceBitsInt1));


    headloc.leftBrother.posNode = BM::NoPos;
    headloc.leftBrother.posLeaf = BM::NoPos;
    headloc.fatherPos = 0;
    setBeginPosition(headloc.rest, 0);
    setEndPosition(headloc.rest, 0);
    setBeginPosition(tailptr.tail, 3);
    _scanPrefix(stkIndex, headloc, tailptr, 4);

    SEQAN_ASSERT_EQ(length(headloc.rest), 1u);
    SEQAN_ASSERT_EQ(beginPosition(headloc.rest) , 3u);
    SEQAN_ASSERT_EQ(headloc.fatherPos, 0u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posNode , 4u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posLeaf , (unsigned) BM::NoPos);
    SEQAN_ASSERT_EQ(headloc.leftBrother.nodeType , SMALL_NODE);
    SEQAN_ASSERT_EQ(beginPosition(tailptr.tail) , 4u);

    _insertNode(stkIndex, 6, headloc, tailptr);
    //artificially set the distance of the small node before further testing
    Iterator<String<unsigned>, Rooted>::Type nodeIt = begin(stkIndex.nodeTable);
    setPosition(nodeIt, 4);
    unsigned nodeSmallInt1 = *nodeIt;
    nodeSmallInt1 &= BM::AllBitsSet - BM::DistanceInt1;
    nodeSmallInt1 += (1 << BM::ShiftDistanceBitsInt1);
    assignValue(nodeIt, nodeSmallInt1);

    //set the headposition of the big node artificially too
    //setPosition(nodeIt, 9); //already done
    //unsigned nodeBigInt4 = *nodeIt;

    //nodeBigInt4 += 3;
    //assignValue(nodeIt, nodeBigInt4);
    headloc.leftBrother.posNode = BM::NoPos;
    headloc.leftBrother.posLeaf = BM::NoPos;
    headloc.fatherPos = 0;
    setBeginPosition(headloc.rest, 0);
    setEndPosition(headloc.rest, 0);
    setBeginPosition(tailptr.tail, 4);
    _scanPrefix(stkIndex, headloc, tailptr, 6);

    SEQAN_ASSERT_EQ(length(headloc.rest), 0u);
    SEQAN_ASSERT_EQ(beginPosition(headloc.rest) , 4u);
    SEQAN_ASSERT_EQ(headloc.fatherPos, 0u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posNode , 0u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posLeaf , (unsigned) BM::NoPos);
    SEQAN_ASSERT_EQ(headloc.leftBrother.nodeType , BIG_NODE);    //always while still inserting into a chain
    SEQAN_ASSERT_EQ(beginPosition(tailptr.tail) , 4u);

    _insertLeaf(stkIndex, headloc, tailptr, 6);

    //check if tree is correct now
    comparisonStringTable =
            "0 :00000000000000000000000000000000\
             1 :00000000000000000000000000000000\
             2 :00110000000000000000000000000000\
             3 :00110000000000000000000000000001\
             4 :00100000000000000000000000000010\
             5 :00000000000000000000000000000000\
             6 :00000000000000000000000000000000\
             7 :00000000000000000000000000000000\
             8 :00000000000000000000000000000000\
             ----------------------\
             0 :00000000000000000000000000000100\
             1 :10000000000000000000000000000000\
             2 :10000000000000000000000000000000\
             3 :00000000000000000000000000000000\
             4 :00001000000000000000000000000010\
             5 :10000000000000000000000000000011\
             6 :00000000000000000000000000000011\
             7 :10100000000000000000000000000000\
             8 :10000000000000000000000000000001\
             9 :00000000000000000000000000000011";

    setUpComparisonTables(compNodeTable, compLeafTable, comparisonStringTable);
    SEQAN_ASSERT(tablesEqual(stkIndex.nodeTable, stkIndex.leafTable, compNodeTable, compLeafTable));

    headloc.leftBrother.posNode = BM::NoPos;
    headloc.leftBrother.posLeaf = BM::NoPos;
    headloc.fatherPos = 0;
    setBeginPosition(headloc.rest, 0);
    setEndPosition(headloc.rest, 0);
    setBeginPosition(tailptr.tail, 5);
    _scanPrefix(stkIndex, headloc, tailptr, 6);
    //scanned for leaf abac    (abab$abac)
    SEQAN_ASSERT_EQ(length(headloc.rest), 1u);
    SEQAN_ASSERT_EQ(beginPosition(headloc.rest) , 7u);
    SEQAN_ASSERT_EQ(headloc.fatherPos, 4u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posNode , (unsigned) BM::NoPos);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posLeaf , 2u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.nodeType , LEAF);
    SEQAN_ASSERT_EQ(beginPosition(tailptr.tail) , 8u);
}

SEQAN_DEFINE_TEST(test_index_stkurtz_getNodeForEdge)
{
    using namespace seqan;
    typedef String<unsigned> TNodes;
    typedef String<unsigned> TLeaves;
    CharString comparisonStringTable;
    TNodes nodesABAB;
    TLeaves leavesABAB;
    comparisonStringTable =
                "0 :00000000000000000000000000000000\
                 1 :00000000000000000000000000000000\
                 2 :00110000000000000000000000000000\
                 3 :00110000000000000000000000000001\
                 4 :00100000000000000000000000000010\
                 ----------------------\
                 0 :00000000000000000000000000000100\
                 1 :10000000000000000000000000000000\
                 2 :10000000000000000000000000000000\
                 3 :00000000000000000000000000000000\
                 4 :00000000000000000000000000000010\
                 5 :10000000000000000000000000000011\
                 6 :00000000000000000000000000000011\
                 7 :10100000000000000000000000000000\
                 8 :10000000000000000000000000000001\
                 9 :00000000000000000000000000000000";
    setUpComparisonTables(nodesABAB, leavesABAB, comparisonStringTable);
    StKurtzTarget<TNodes, TLeaves> father;
    father.posNode = 0;
    father.posLeaf = StKurtzBitMasks::NoPos;
    father.nodeType = BIG_NODE;
    bool foundEdge = true;
    Value<CharString>::Type chr = 'b';
    StKurtzTarget<TNodes, TLeaves> leftBrother;
    leftBrother.nodeType = BIG_NODE;
    leftBrother.posNode = 0;
    leftBrother.posLeaf = StKurtzBitMasks::NoPos;
    //_getNodeForEdge<CharString, String<unsigned>, String<unsigned> >(nodesABAB, leavesABAB, father, chr, leftBrother, foundEdge);
}

SEQAN_DEFINE_TEST(test_index_stkurtz_rescan)
{
    using namespace seqan;
    typedef StKurtzBitMasks BM;

    typedef String<unsigned> TNodes;
    typedef String<unsigned> TLeaves;
    CharString text = "abab$";
    CharString comparisonStringTable;
    TNodes nodesABAB;
    TLeaves leavesABAB;

    //first set up the abab$ tree
    comparisonStringTable =
                "0 :00000000000000000000000000000000\
                 1 :00000000000000000000000000000000\
                 2 :00110000000000000000000000000000\
                 3 :00110000000000000000000000000001\
                 4 :00100000000000000000000000000010\
                 ----------------------\
                 0 :00000000000000000000000000000100\
                 1 :10000000000000000000000000000000\
                 2 :10000000000000000000000000000000\
                 3 :00000000000000000000000000000000\
                 4 :00001000000000000000000000000010\
                 5 :10000000000000000000000000000011\
                 6 :00000000000000000000000000000011\
                 7 :10100000000000000000000000000000\
                 8 :10000000000000000000000000000001\
                 9 :00000000000000000000000000000011";
    setUpComparisonTables(nodesABAB, leavesABAB, comparisonStringTable);

    Index<CharString, IndexStKurtz<> > index(text);
    index.nodeTable = nodesABAB;
    index.leafTable = leavesABAB;

    StKurtzHeadLoc<CharString, TNodes, TLeaves> headloc;
    //1
    headloc.fatherPos = 0;
    headloc.leftBrother.nodeType = BIG_NODE;
    headloc.leftBrother.posNode = BM::NoPos;
    headloc.leftBrother.posLeaf = BM::NoPos;
    headloc.rest = infix(text, 0, 0);
    setBeginPosition(headloc.rest, 0);
    setEndPosition(headloc.rest, 1);
    //headloc = root->a
    _rescan(index, headloc, BM::NoPos);
    SEQAN_ASSERT_EQ(headloc.fatherPos , 0u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posLeaf , 4u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.nodeType , LEAF);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posNode , (unsigned) BM::NoPos);
    SEQAN_ASSERT_EQ(beginPosition(headloc.rest) , 0u);
    SEQAN_ASSERT_EQ(endPosition(headloc.rest), 1u);

    //2
    headloc.fatherPos = 0;
    headloc.leftBrother.nodeType = BIG_NODE;
    headloc.leftBrother.posNode = BM::NoPos;
    headloc.leftBrother.posLeaf = BM::NoPos;
    setBeginPosition(headloc.rest, 0);
    setEndPosition(headloc.rest, 2);
    //headloc = root->ab
    _rescan(index, headloc, BM::NoPos);
    SEQAN_ASSERT_EQ(headloc.fatherPos , 4u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posLeaf , (unsigned) BM::NoPos);
    SEQAN_ASSERT_EQ(headloc.leftBrother.nodeType , BIG_NODE);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posNode , (unsigned) BM::NoPos);
    SEQAN_ASSERT_EQ(beginPosition(headloc.rest) , 2u);
    SEQAN_ASSERT_EQ(endPosition(headloc.rest), 2u);

    //3
    headloc.fatherPos = 0;
    headloc.leftBrother.nodeType = BIG_NODE;
    headloc.leftBrother.posNode = BM::NoPos;
    headloc.leftBrother.posLeaf = BM::NoPos;
    setBeginPosition(headloc.rest, 0);
    setEndPosition(headloc.rest, 3);
    //headloc = root->aba
    _rescan(index, headloc, BM::NoPos);
    SEQAN_ASSERT_EQ(headloc.fatherPos , 4u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posLeaf , 2u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.nodeType , LEAF);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posNode , (unsigned) BM::NoPos);
    SEQAN_ASSERT_EQ(beginPosition(headloc.rest) , 2u);
    SEQAN_ASSERT_EQ(endPosition(headloc.rest), 3u);

    //4
    headloc.fatherPos = 0;
    headloc.leftBrother.nodeType = BIG_NODE;
    headloc.leftBrother.posNode = BM::NoPos;
    headloc.leftBrother.posLeaf = BM::NoPos;
    setBeginPosition(headloc.rest, 0);
    setEndPosition(headloc.rest, 4);
    //headloc = root->abab
    _rescan(index, headloc, BM::NoPos);
    SEQAN_ASSERT_EQ(headloc.fatherPos , 4u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posLeaf , 2u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.nodeType , LEAF);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posNode , (unsigned) BM::NoPos);
    SEQAN_ASSERT_EQ(beginPosition(headloc.rest) , 2u);
    SEQAN_ASSERT_EQ(endPosition(headloc.rest), 4u);

    //5
    headloc.fatherPos = 4;
    headloc.leftBrother.nodeType = BIG_NODE;
    headloc.leftBrother.posNode = BM::NoPos;
    headloc.leftBrother.posLeaf = BM::NoPos;
    setBeginPosition(headloc.rest, 2);
    setEndPosition(headloc.rest, 3);
    //headloc = (ab)->a
    _rescan(index, headloc, BM::NoPos);
    SEQAN_ASSERT_EQ(headloc.fatherPos , 4u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posLeaf , 2u);
    SEQAN_ASSERT_EQ(headloc.leftBrother.nodeType , LEAF);
    SEQAN_ASSERT_EQ(headloc.leftBrother.posNode , (unsigned) BM::NoPos);
    SEQAN_ASSERT_EQ(beginPosition(headloc.rest) , 2u);
    SEQAN_ASSERT_EQ(endPosition(headloc.rest), 3u);
}

SEQAN_DEFINE_TEST(test_index_stkurtz_buildIndexStKurtz)
{
    CharString text = "abab$";
    Index<CharString, IndexStKurtz<> > index(text);
    _initIndexStKurtz(index);
    _buildIndexStKurtz(index);

    CharString comparisonStringTable;
    String<unsigned> nodesABAB;
    String<unsigned> leavesABAB;

    //first set up the abab$ tree (NOTE: the last two node ints are the ones
    //already reserved for the "next node")
    comparisonStringTable =
                "0 :00000000000000000000000000000000\
                 1 :00000000000000000000000000000000\
                 2 :00110000000000000000000000000000\
                 3 :00110000000000000000000000000001\
                 4 :00100000000000000000000000000010\
                 ----------------------\
                 0 :00000000000000000000000000000100\
                 1 :10000000000000000000000000000000\
                 2 :10000000000000000000000000000000\
                 3 :00000000000000000000000000000000\
                 4 :00001000000000000000000000000010\
                 5 :10000000000000000000000000000011\
                 6 :00000000000000000000000000000011\
                 7 :10100000000000000000000000000000\
                 8 :10000000000000000000000000000001\
                 9 :00000000000000000000000000000011\
                 10:00000000000000000000000000000000\
                 11:00000000000000000000000000000000";
    setUpComparisonTables(nodesABAB, leavesABAB, comparisonStringTable);
    SEQAN_ASSERT(tablesEqual(index.nodeTable, index.leafTable, nodesABAB, leavesABAB));
    StKurtzTarget<String<unsigned>, String<unsigned> > root;
    root.nodeType = BIG_NODE;
    root.posNode = 0;
    root.posLeaf = StKurtzBitMasks::NoPos;
    _printTree(index, root);
}

SEQAN_DEFINE_TEST(test_index_stkurtz_buildIndexStKurtz2)
{
    CharString text = "AGAGAGCTT$";
    Index<CharString, IndexStKurtz<> > index(text);
    _createIndex(index);
    CharString comparisonStringTable;
    String<unsigned> nodesABAB;
    String<unsigned> leavesABAB;

    comparisonStringTable =
            " 0 :00110000000000000000000000000010\
              1 :00110000000000000000000000000011\
              2 :00000000000000000000000000000000\
              3 :00000000000000000000000000000000\
              4 :00000000000000000000000000000000\
              5 :00000000000000000000000000000000\
              6 :00100000000000000000000000000111\
              7 :00000000000000000000000000000000\
              8 :00110000000000000000000000000111\
              9 :00100000000000000000000000000100\
             --------------------- \
              0 :00000000000000000000000000001001\
              1 :10000000000000000000000000000000\
              2 :10000000000000000000000000000000\
              3 :00000000000000000000000000000000\
              4 :00011000000000000000000000000000\
              5 :10010000000000000000000000000100\
              6 :00010000000000000000000000000001\
              7 :10010000000000000000000000000101\
              8 :00001000000000000000000000000010\
              9 :00000000000000000000000000000101\
             10 :00000000000000000000000000000011\
             11 :00010000000000000000000000000110\
             12 :10000000000000000000000000000001\
             13 :00000000000000000000000000000101\
             14 :00000000000000000000000000001000\
             15 :10100000000000000000000000000000\
             16 :10000000000000000000000000000001\
             17 :00000000000000000000000000001000\
             18 :00000000000000000000000000000000\
             19 :00000000000000000000000000000000";

    setUpComparisonTables(nodesABAB, leavesABAB, comparisonStringTable);
    SEQAN_ASSERT(tablesEqual(index.nodeTable, index.leafTable, nodesABAB, leavesABAB));
    StKurtzTarget<String<unsigned>, String<unsigned> > root;
    root.nodeType = BIG_NODE;
    root.posNode = 0;
    root.posLeaf = StKurtzBitMasks::NoPos;
    _printTree(index, root);
}

SEQAN_DEFINE_TEST(test_index_stkurtz_buildIndexStKurtz3)
{
    CharString text = "BANANA$";
    Index<CharString, IndexStKurtz<> > index(text);
    _createIndex(index);
    CharString comparisonStringTable;
    String<unsigned> nodesABAB;
    String<unsigned> leavesABAB;

    comparisonStringTable =
            "0 :00100000000000000000000000000011\
             1 :00000000000000000000000000000000\
             2 :00000000000000000000000000000000\
             3 :00110000000000000000000000000001\
             4 :00110000000000000000000000000010\
             5 :00100000000000000000000000000010\
             6 :00100000000000000000000000000100\
            ------------------\
             0 :00000000000000000000000000000110\
             1 :10000000000000000000000000000000\
             2 :10000000000000000000000000000000\
             3 :00000000000000000000000000000000\
             4 :00010000000000000000000000000011\
             5 :10100000000000000000000000000000\
             6 :00001000000000000000000000000100\
             7 :10100000000000000000000000000000\
             8 :00000000000000000000000000000101\
             9 :10010000000000000000000000000000\
            10 :10000000000000000000000000000001\
            11 :00000000000000000000000000000101\
            12 :00000000000000000000000000000000\
            13 :00000000000000000000000000000000";

    setUpComparisonTables(nodesABAB, leavesABAB, comparisonStringTable);
    SEQAN_ASSERT(tablesEqual(index.nodeTable, index.leafTable, nodesABAB, leavesABAB));
    StKurtzTarget<String<unsigned>, String<unsigned> > root;
    root.nodeType = BIG_NODE;
    root.posNode = 0;
    root.posLeaf = StKurtzBitMasks::NoPos;
    _printTree(index, root);
}


SEQAN_DEFINE_TEST(test_index_stkurtz_iterateTopDown)
{
    using namespace seqan;
    typedef Index<String<Dna5>, IndexStKurtz<> > TIndex;
    TIndex index("AGAGAGCTT$");
    Iterator< TIndex, TopDown<> >::Type it(index);

    String<Dna5> pattern = "AG";
    bool notFound = false;
    while (repLength(it) < length(pattern))
    {
        // go down edge starting with the next pattern character
        if (!goDown(it, pattern[repLength(it)]))
        {
            notFound = true;
            break;
        }
        unsigned endPos = _min(repLength(it), length(pattern));
        // compare remaining edge characters with pattern
        if (infix(representative(it), parentRepLength(it) + 1, endPos) !=
            infix(pattern, parentRepLength(it) + 1, endPos))
        {
            notFound = true;
            break;
        }
    }
    SEQAN_ASSERT_NOT(notFound);
    String<typename SAValue<TIndex>::Type > occurrences = getOccurrences(it);
    SEQAN_ASSERT_EQ(length(occurrences) , 3u);
    SEQAN_ASSERT(contains(occurrences, 0u));
    SEQAN_ASSERT(contains(occurrences, 2u));
    SEQAN_ASSERT(contains(occurrences, 4u));
}

SEQAN_DEFINE_TEST(test_index_stkurtz_iterateTopDown2)
{
    using namespace seqan;
    typedef Index<String<Dna5>, IndexStKurtz<> > TIndex;
    TIndex index("AGAGACAGGCAGGG$");
    Iterator< TIndex, TopDown<> >::Type it(index);

    StKurtzTarget<String<unsigned>, String<unsigned> > root;
    root.posNode = 0;
    root.nodeType = BIG_NODE;
    root.posLeaf = StKurtzBitMasks::NoPos;
    _printTree(index, root);

    String<Dna5> pattern = "AG";
    bool notFound = false;
    while (repLength(it) < length(pattern))
    {
        // go down edge starting with the next pattern character
        if (!goDown(it, pattern[repLength(it)]))
        {
            notFound = true;
            break;
        }
        unsigned endPos = _min(repLength(it), length(pattern));
        // compare remaining edge characters with pattern
        if (infix(representative(it), parentRepLength(it) + 1, endPos) !=
            infix(pattern, parentRepLength(it) + 1, endPos))
        {
            notFound = true;
            break;
        }
    }
    SEQAN_ASSERT_NOT(notFound);
    String<typename SAValue<TIndex>::Type > occurrences = getOccurrences(it);
    SEQAN_ASSERT_EQ(length(occurrences) , 4u);
    SEQAN_ASSERT(contains(occurrences, 0u));
    SEQAN_ASSERT(contains(occurrences, 2u));
    SEQAN_ASSERT(contains(occurrences, 6u));
    SEQAN_ASSERT(contains(occurrences, 10u));


    clear(it);
    pattern = "G";
    notFound = false;
    while (repLength(it) < length(pattern))
    {
        // go down edge starting with the next pattern character
        if (!goDown(it, pattern[repLength(it)]))
        {
            notFound = true;
            break;
        }
        unsigned endPos = _min(repLength(it), length(pattern));
        // compare remaining edge characters with pattern
        if (infix(representative(it), parentRepLength(it) + 1, endPos) !=
            infix(pattern, parentRepLength(it) + 1, endPos))
        {
            notFound = true;
            break;
        }
    }
    SEQAN_ASSERT_NOT(notFound);

    occurrences = getOccurrences(it);
    SEQAN_ASSERT_EQ(length(occurrences) , 7u);
    SEQAN_ASSERT(contains(occurrences, 1u));
    SEQAN_ASSERT(contains(occurrences, 3u));
    SEQAN_ASSERT(contains(occurrences, 7u));
    SEQAN_ASSERT(contains(occurrences, 8u));
    SEQAN_ASSERT(contains(occurrences, 11u));
    SEQAN_ASSERT(contains(occurrences, 12u));
    SEQAN_ASSERT(contains(occurrences, 13u));

    clear(it);
    pattern = "CAGG";
    notFound = false;
    while (repLength(it) < length(pattern))
    {
        // go down edge starting with the next pattern character
        if (!goDown(it, pattern[repLength(it)]))
        {
            notFound = true;
            break;
        }
        unsigned endPos = _min(repLength(it), length(pattern));
        // compare remaining edge characters with pattern
        std::cout << representative(it) << std::endl;
        if (infix(representative(it), parentRepLength(it) + 1, endPos) !=
            infix(pattern, parentRepLength(it) + 1, endPos))
        {
            notFound = true;
            break;
        }
    }
    SEQAN_ASSERT_NOT(notFound);
    occurrences = getOccurrences(it);
    SEQAN_ASSERT_EQ(length(occurrences) , 2u);
    SEQAN_ASSERT(contains(occurrences, 5u));
    SEQAN_ASSERT(contains(occurrences, 9u));
}

}  // namespace seqan

#endif  // SANDBOX_TUM_TESTS_INDEX_SUFFIX_TREES_TEST_INDEX_STKURTZ_H_
